/*-
# X-BASED RUBIK'S CUBE(tm)
#
#  Rubik2d.c
#
###
#
#  Copyright (c) 1994 - 2007	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Rubik2d */

#include "RubikP.h"
#include "Rubik2dP.h"

#ifndef WINVER
static Boolean SetValuesPuzzle2D(Widget current, Widget request, Widget renew);
static void ResizePuzzle2D(Rubik2DWidget w);
static void InitializePuzzle2D(Widget request, Widget renew);
static void ExposePuzzle2D(Widget renew, XEvent *event, Region region);
static void MovePuzzle2DTop(Rubik2DWidget w,
			   XEvent *event, char **args, int nArgs);
static void MovePuzzle2DLeft(Rubik2DWidget w,
			    XEvent *event, char **args, int nArgs);
static void MovePuzzle2DRight(Rubik2DWidget w,
			     XEvent *event, char **args, int nArgs);
static void MovePuzzle2DBottom(Rubik2DWidget w,
			      XEvent *event, char **args, int nArgs);

static char defaultTranslationsPuzzle2D[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>r: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>d: Decrement()\n\
 <KeyPress>i: Increment()\n\
 <KeyPress>x: IncrementX()\n\
 <KeyPress>y: IncrementY()\n\
 <KeyPress>f: IncrementZ()\n\
 <KeyPress>v: View()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListPuzzle2D[] =
{
	{(char *) "Quit", (XtActionProc) QuitPuzzle},
	{(char *) "Hide", (XtActionProc) HidePuzzle},
	{(char *) "MoveCcw", (XtActionProc) MovePuzzleCcw},
	{(char *) "MoveTop", (XtActionProc) MovePuzzle2DTop},
	{(char *) "MoveLeft", (XtActionProc) MovePuzzle2DLeft},
	{(char *) "MoveCw", (XtActionProc) MovePuzzleCw},
	{(char *) "MoveRight", (XtActionProc) MovePuzzle2DRight},
	{(char *) "MoveBottom", (XtActionProc) MovePuzzle2DBottom},
	{(char *) "Select", (XtActionProc) SelectPuzzle},
	{(char *) "Release", (XtActionProc) ReleasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) PracticePuzzleMaybe},
	{(char *) "Practice2", (XtActionProc) PracticePuzzle2},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizePuzzleMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizePuzzle2},
	{(char *) "Get", (XtActionProc) GetPuzzle},
	{(char *) "Write", (XtActionProc) WritePuzzle},
	{(char *) "Undo", (XtActionProc) UndoPuzzle},
	{(char *) "Redo", (XtActionProc) RedoPuzzle},
	{(char *) "Clear", (XtActionProc) ClearPuzzle},
	{(char *) "Randomize", (XtActionProc) RandomizePuzzle},
	{(char *) "Solve", (XtActionProc) SolvePuzzle},
	{(char *) "Practice", (XtActionProc) PracticePuzzle},
	{(char *) "Orientize", (XtActionProc) OrientizePuzzle},
	{(char *) "Decrement", (XtActionProc) DecrementPuzzle},
	{(char *) "Increment", (XtActionProc) IncrementPuzzle},
	{(char *) "IncrementX", (XtActionProc) IncrementXPuzzle},
	{(char *) "IncrementY", (XtActionProc) IncrementYPuzzle},
	{(char *) "IncrementZ", (XtActionProc) IncrementZPuzzle},
	{(char *) "View", (XtActionProc) ViewPuzzle},
	{(char *) "Enter", (XtActionProc) EnterPuzzle},
        {(char *) "Leave", (XtActionProc) LeavePuzzle}
};

static XtResource resourcesPuzzle2D[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(RubikWidget, core.width),
	 XtRString, (caddr_t) "300"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(RubikWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[0]),
	 XtRString, (caddr_t) "red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[1]),
	 XtRString, (caddr_t) "yellow"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[2]),
	 XtRString, (caddr_t) "white"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[3]),
	 XtRString, (caddr_t) "green"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[4]),
	 XtRString, (caddr_t) "orange"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[5]),
	 XtRString, (caddr_t) "blue"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.delay),
	 XtRString, (caddr_t) "10"},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNsizex, XtCSizeX, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizex),
	 XtRString, (caddr_t) "3"},	/* DEFAULTFACETS */
	{XtNsizey, XtCSizeY, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizey),
	 XtRString, (caddr_t) "3"},	/* DEFAULTFACETS */
	{XtNsizez, XtCSizeZ, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizez),
	 XtRString, (caddr_t) "3"},	/* DEFAULTFACETS */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULTORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.practice),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULTPRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.menu),
	 XtRString, (caddr_t) "-1"},
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNcontrol, XtCControl, XtRInt, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.currentControl),
	 XtRString, (caddr_t) "-1"},
	{XtNfast, XtCFast, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentFast),
	 XtRString, (caddr_t) "1"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(RubikWidget, rubik.select),
	 XtRCallback, (caddr_t) NULL}
};

Rubik2DClassRec rubik2dClassRec =
{
	{
		(WidgetClass) & rubikClassRec,	/* superclass */
		(char *) "Rubik2D",	/* class name */
		sizeof (Rubik2DRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializePuzzle2D,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListPuzzle2D,	/* actions */
		XtNumber(actionsListPuzzle2D),	/* num actions */
		resourcesPuzzle2D,	/* resources */
		XtNumber(resourcesPuzzle2D),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) ResizePuzzle2D,	/* resize */
		(XtExposeProc) ExposePuzzle2D,	/* expose */
		(XtSetValuesFunc) SetValuesPuzzle2D,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsPuzzle2D,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass rubik2dWidgetClass = (WidgetClass) & rubik2dClassRec;
#endif

static RowNext rotateToRow[MAXFACES] =	/*CW to min face */
{
	{1, LEFT, TOP},
	{0, BOTTOM, RIGHT},
	{0, RIGHT, BOTTOM},
	{0, TOP, LEFT},
	{1, RIGHT, BOTTOM},
	{0, LEFT, TOP}
};
static int  planeToCube[MAXRECT] =
{6, 0, 6, 1, 2, 3, 6, 4, 6, 6, 5, 6};
static int  cubeToPlane[MAXFACES] =
{1, 3, 4, 5, 7, 10};

static Pixmap dr = 0; /* dummy for future double buffering */

static void
FacePosition(Rubik2DWidget w, int faceX, int faceY,
	     int *viewLengthx, int *viewLengthy)
{
	int         i;

	*viewLengthx = w->rubik.puzzleOffset.x;
	*viewLengthy = w->rubik.puzzleOffset.y;
	for (i = 0; i < faceX; i++) {
		if (i & 1) {
			*viewLengthx += w->rubik2d.viewLengthx;
		} else {
			*viewLengthx += w->rubik2d.viewLengthz;
		}
	}
	for (i = 0; i < faceY; i++) {
		if (i & 1) {
			*viewLengthy += w->rubik2d.viewLengthy;
		} else {
			*viewLengthy += w->rubik2d.viewLengthz;
		}
	}
}

static void
DrawOrientLine(Rubik2DWidget w, int orient, int dx, int dy, GC borderGC)
{
	switch (orient) {
	case TOP:
		DRAWLINE(w, dr, borderGC,
			dx + w->rubik2d.cubeLength / 2,
			dy,
			dx + w->rubik2d.cubeLength / 2,
			dy + w->rubik.orientLineLength);
		return;
	case RIGHT:
		DRAWLINE(w, dr, borderGC,
			dx + w->rubik2d.cubeLength,
			dy + w->rubik2d.cubeLength / 2,
			dx + w->rubik2d.cubeLength -
				w->rubik.orientLineLength - 1,
			dy + w->rubik2d.cubeLength / 2);
		return;
	case BOTTOM:
		DRAWLINE(w, dr, borderGC,
			dx + w->rubik2d.cubeLength / 2,
			dy + w->rubik2d.cubeLength,
			dx + w->rubik2d.cubeLength / 2,
			dy + w->rubik2d.cubeLength -
			w->rubik.orientLineLength - 1);
		return;
	case LEFT:
		DRAWLINE(w, dr, borderGC,
			dx,
			dy + w->rubik2d.cubeLength / 2,
			dx + w->rubik.orientLineLength,
			dy + w->rubik2d.cubeLength / 2);
		return;
	default:
		{
			char *buf;

			intCat(&buf, "DrawOrientLine: orient ",
				orient);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
}

void
DrawSquare2D(Rubik2DWidget w, int face, int position, int offset)
{
	GC faceGC, borderGC;
	int dx, dy, orient, i, j, sizeOfRow, sizeOfColumn;

	faceSizes((RubikWidget) w, face, &sizeOfRow, &sizeOfColumn);
	i = position % sizeOfRow;
	j = position / sizeOfRow;
	orient = w->rubik.cubeLoc[face][position].rotation;
	if (w->rubik.vertical || face != MAXFACES - 1) {
		FacePosition(w,
			cubeToPlane[face] % MAXX, cubeToPlane[face] / MAXX,
			&dx, &dy);
		dx += i * (w->rubik2d.cubeLength + w->rubik.delta - 1);
		dy += j * (w->rubik2d.cubeLength + w->rubik.delta - 1);
	} else {
		FacePosition(w,
			cubeToPlane[face] / MAXX, cubeToPlane[face] % MAXX,
			&dx, &dy);
		dx += (sizeOfRow - 1 - i) *
			(w->rubik2d.cubeLength + w->rubik.delta - 1);
		dy += (sizeOfColumn - 1 - j) *
			(w->rubik2d.cubeLength + w->rubik.delta - 1);
		orient = (orient + HALF) % STRT;
	}
	dx += w->rubik.delta - 1;
	dy += w->rubik.delta - 1;
	if (offset) {
		borderGC = w->rubik.faceGC[(int) w->rubik.cubeLoc[face][position].face];
		if (w->rubik.mono) {
			faceGC = w->rubik.inverseGC;
		} else {
			faceGC = w->rubik.borderGC;
		}
	} else {
		faceGC = w->rubik.faceGC[(int) w->rubik.cubeLoc[face][position].face];
		borderGC = w->rubik.borderGC;
	}
	FILLRECTANGLE2(w, dr, faceGC, borderGC,
		dx, dy, w->rubik2d.cubeLength, w->rubik2d.cubeLength);
	if (w->rubik.mono) {
		int letterX, letterY;
		char buf[2];

		buf[0] =
#ifdef WINVER
			w->rubik.faceChar[(int) w->rubik.cubeLoc
				[face][position].face];
#else
			w->rubik.faceName[(int) w->rubik.cubeLoc
				[face][position].face][0];
#endif
		buf[1] = '\0';
		letterX = dx + w->rubik2d.cubeLength / 2 +
			w->rubik.letterOffset.x;
		letterY = dy + w->rubik2d.cubeLength / 2 +
			w->rubik.letterOffset.y;
		if (offset) {
			borderGC = w->rubik.borderGC;
		} else {
			borderGC = w->rubik.inverseGC;
		}
		DRAWTEXT(w, dr, borderGC, letterX, letterY, buf, 1);
	}
	if (w->rubik.orient)
		DrawOrientLine(w, orient, dx, dy, borderGC);
}

void
DrawFrame2D(Rubik2DWidget w, Boolean focus)
{
	int i;
	Point tempLength;
	Point pos[MAXXY + 1], letters;
	GC gc = (focus) ? w->rubik.frameGC : w->rubik.borderGC;

	tempLength.x = w->rubik.puzzleOffset.x;
	tempLength.y = w->rubik.puzzleOffset.y;
	for (i = 0; i <= MAXY; i++) {
		pos[i] = tempLength;
		if (i & 1) {
			tempLength.x += w->rubik2d.viewLengthx;
			tempLength.y += w->rubik2d.viewLengthy;
		} else {
			tempLength.x += w->rubik2d.viewLengthz;
			tempLength.y += w->rubik2d.viewLengthz;
		}
	}
	DRAWLINE(w, dr, gc, pos[1].x, pos[0].y, pos[2].x, pos[0].y);
	DRAWLINE(w, dr, gc, pos[3].x, pos[1].y, pos[3].x, pos[2].y);
	DRAWLINE(w, dr, gc, pos[1].x, pos[3].y, pos[2].x, pos[3].y);
	DRAWLINE(w, dr, gc, pos[0].x, pos[1].y, pos[0].x, pos[2].y);
	letters.x = (pos[0].x + pos[1].x) / 2 - w->rubik.delta;
	letters.y = (pos[0].y + pos[1].y) / 2;
	DRAWTEXT(w, dr, w->rubik.borderGC,
		(int) (letters.x + 5 * w->rubik.letterOffset.x),
		(int) (letters.y + w->rubik.letterOffset.y), "Front", 5);
	letters.x = (pos[2].x + pos[3].x) / 2 - w->rubik.delta;
	letters.y = (pos[2].y + pos[3].y) / 2;
	DRAWTEXT(w, dr, w->rubik.borderGC,
		 (int) (letters.x + 4 * w->rubik.letterOffset.x),
		 (int) (letters.y + w->rubik.letterOffset.y), "Back", 4);
	if (w->rubik.vertical) {
		DRAWLINE(w, dr, gc, pos[1].x, pos[0].y, pos[1].x, pos[4].y);
		DRAWLINE(w, dr, gc, pos[2].x, pos[0].y, pos[2].x, pos[4].y);
		DRAWLINE(w, dr, gc, pos[0].x, pos[1].y, pos[3].x, pos[1].y);
		DRAWLINE(w, dr, gc, pos[0].x, pos[2].y, pos[3].x, pos[2].y);
		DRAWLINE(w, dr, gc, pos[1].x, pos[4].y, pos[2].x, pos[4].y);
	} else {
		DRAWLINE(w, dr, gc, pos[0].x, pos[1].y, pos[4].x, pos[1].y);
		DRAWLINE(w, dr, gc, pos[0].x, pos[2].y, pos[4].x, pos[2].y);
		DRAWLINE(w, dr, gc, pos[1].x, pos[0].y, pos[1].x, pos[3].y);
		DRAWLINE(w, dr, gc, pos[2].x, pos[0].y, pos[2].x, pos[3].y);
		DRAWLINE(w, dr, gc, pos[4].x, pos[1].y, pos[4].x, pos[2].y);
	}
}

static void
ResizeCubie(Rubik2DWidget w)
{
	Point flatlength, flatlengthh;
	int tempLength;

	w->rubik.delta = 3;
	flatlength.x = w->rubik.sizex + 2 * w->rubik.sizez;
	flatlength.y = 2 * w->rubik.sizey + 2 * w->rubik.sizez;
	flatlengthh.x = 2 * w->rubik.sizex + 2 * w->rubik.sizez;
	flatlengthh.y = w->rubik.sizey + 2 * w->rubik.sizez;
	w->rubik.vertical =	/* w->core.height >= w->core.width */
		(w->core.height / flatlength.y + w->core.width / flatlength.x <=
		w->core.height / flatlengthh.y + w->core.width / flatlengthh.x);
	if (!w->rubik.vertical)
		flatlength = flatlengthh;
	tempLength = MIN((w->core.height - 2 * w->rubik.delta - 2) / flatlength.y,
		(w->core.width - 2 * w->rubik.delta - 2) / flatlength.x);
	w->rubik2d.cubeLength = MAX(tempLength - w->rubik.delta + 1, 0);
	w->rubik2d.faceLengthx = w->rubik.sizex * (w->rubik2d.cubeLength +
		w->rubik.delta - 1);
	w->rubik2d.faceLengthy = w->rubik.sizey * (w->rubik2d.cubeLength +
		w->rubik.delta - 1);
	w->rubik2d.faceLengthz = w->rubik.sizez * (w->rubik2d.cubeLength +
		w->rubik.delta - 1);
	w->rubik2d.viewLengthx = w->rubik2d.faceLengthx + w->rubik.delta - 1;
	w->rubik2d.viewLengthy = w->rubik2d.faceLengthy + w->rubik.delta - 1;
	w->rubik2d.viewLengthz = w->rubik2d.faceLengthz + w->rubik.delta - 1;
}

static void
ResizePieces(Rubik2DWidget w)
{
	ResizeCubie(w);
	w->rubik.orientLineLength = w->rubik2d.cubeLength / 4;
	w->rubik.letterOffset.x = -2;
	w->rubik.letterOffset.y = 4;
}

Boolean
SelectPieces2D(Rubik2DWidget w, int x, int y, int *face, int *position)
{
	int faceX, faceY, i, j, sizeOfRow, sizeOfColumn, tempLength;

	x -= w->rubik.puzzleOffset.x;
	y -= w->rubik.puzzleOffset.y;
	/* faceX = x / w->rubik2d.viewLength; */
	for (faceX = 0; faceX <= MAXY; faceX++) {
		if (faceX & 1)
			tempLength = w->rubik2d.viewLengthx;
		else
			tempLength = w->rubik2d.viewLengthz;
		if (x <= tempLength)
			break;
		else
			x -= tempLength;
	}
	/* faceY = y / w->rubik2d.viewLength; */
	for (faceY = 0; faceY <= MAXY; faceY++) {
		if (faceY & 1)
			tempLength = w->rubik2d.viewLengthy;
		else
			tempLength = w->rubik2d.viewLengthz;
		if (y <= tempLength)
			break;
		else
			y -= tempLength;
	}
	i = MAX((x - w->rubik.delta) /
		(w->rubik2d.cubeLength + w->rubik.delta - 1), 0);
	j = MAX((y - w->rubik.delta) /
		(w->rubik2d.cubeLength + w->rubik.delta - 1), 0);
	if ((faceX != 1 && faceY != 1) ||
			(faceX >= 3 && w->rubik.vertical) ||
			(faceY >= 3 && !w->rubik.vertical))
		return False;
	if (faceX == 3)
		*face = MAXFACES - 1;
	else
		*face = planeToCube[faceX + faceY * MAXX];
	faceSizes((RubikWidget) w, *face, &sizeOfRow, &sizeOfColumn);
	if (i >= sizeOfRow)
		i = sizeOfRow - 1;
	if (j >= sizeOfColumn)
		j = sizeOfColumn - 1;
	if (faceX == 3) {
		i = sizeOfRow - 1 - i;
		j = sizeOfColumn - 1 - j;
	}
	*position = j * sizeOfRow + i;
	return True;
}

Boolean
NarrowSelection2D(Rubik2DWidget w, int *face, int *position, int *direction)
{
	if (*face == MAXFACES - 1 && *direction < MAXORIENT && !w->rubik.vertical)
		*direction = (*direction + HALF) % MAXORIENT;
	/* Remap to row movement */
	if (*direction == CW || *direction == CCW) {
		int i, j, sizeOfRow, sizeOfColumn, newFace;

		newFace = rotateToRow[*face].face;
		faceSizes((RubikWidget) w, newFace, &sizeOfRow, &sizeOfColumn);
		*direction = (*direction == CCW) ?
			(rotateToRow[*face].direction + 2) % MAXORIENT :
			rotateToRow[*face].direction;
		if (rotateToRow[*face].sideFace == LEFT ||
				rotateToRow[*face].sideFace == BOTTOM) {
			i = sizeOfRow - 1;
			j = sizeOfColumn - 1;
		} else {
			i = j = 0;
		}
		*face = newFace;
		*position = j * sizeOfRow + i;
	}
	return True;
}

#ifndef WINVER
static Boolean
SetValuesPuzzle2D(Widget current, Widget request, Widget renew)
{
	Rubik2DWidget c = (Rubik2DWidget) current, w = (Rubik2DWidget) renew;
	Boolean redraw = False;

	if (w->rubik.sizex != c->rubik.sizex ||
			w->rubik.sizey != c->rubik.sizey ||
			w->rubik.sizez != c->rubik.sizez) {
		ResetPieces((RubikWidget) w);
		ResizePuzzle2D(w);
		redraw = True;
	}
	if (w->rubik2d.cubeLength != c->rubik2d.cubeLength) {
		ResizePuzzle2D(w);
		redraw = True;
	}
	return (redraw);
}
#endif

#ifndef WINVER
static
#endif
void
ResizePuzzle2D(Rubik2DWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif

	ResizeCubie(w);
	if (w->rubik.vertical) {
		w->rubik.puzzleSize.x = (w->rubik2d.viewLengthx - 1) +
			2 * (w->rubik2d.viewLengthz - 1) + w->rubik.delta;
		w->rubik.puzzleSize.y = 2 * (w->rubik2d.viewLengthy - 1) +
			2 * (w->rubik2d.viewLengthz - 1) + w->rubik.delta;
	} else {
		w->rubik.puzzleSize.x = 2 * (w->rubik2d.viewLengthx - 1) +
			2 * (w->rubik2d.viewLengthz - 1) + w->rubik.delta;
		w->rubik.puzzleSize.y = (w->rubik2d.viewLengthy - 1) +
			2 * (w->rubik2d.viewLengthz - 1) + w->rubik.delta;
	}
	w->rubik.puzzleOffset.x = ((int) w->core.width - w->rubik.puzzleSize.x)
		/ 2;
	w->rubik.puzzleOffset.y = ((int) w->core.height - w->rubik.puzzleSize.y)
		/ 2;
	ResizePieces(w);
}

#ifndef WINVER
static
#endif
void
InitializePuzzle2D(
#ifdef WINVER
Rubik2DWidget w
#else
Widget request, Widget renew
#endif
)
{
#ifndef WINVER
	Rubik2DWidget w = (Rubik2DWidget) renew;

	SetAllColors((RubikWidget) w);
#endif
	w->rubik.dim = 2;
	ResizePuzzle2D(w);
}

#ifndef WINVER
static
#endif
void
ExposePuzzle2D(
#ifdef WINVER
Rubik2DWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifndef WINVER
	Rubik2DWidget w = (Rubik2DWidget) renew;

	if (!w->core.visible)
		return;
#endif
	FILLRECTANGLE(w, dr, w->rubik.inverseGC,
		0, 0, w->core.width, w->core.height);
	DrawFrame2D(w, w->rubik.focus);
	DrawAllPieces((RubikWidget) w);
}

#ifndef WINVER
static void
MovePuzzle2DTop(Rubik2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		TOP, (int) (event->xkey.state & ControlMask));
}

static void
MovePuzzle2DLeft(Rubik2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		LEFT, (int) (event->xkey.state & ControlMask));
}

static void
MovePuzzle2DRight(Rubik2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		RIGHT, (int) (event->xkey.state & ControlMask));
}

static void
MovePuzzle2DBottom(Rubik2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		BOTTOM, (int) (event->xkey.state & ControlMask));
}
#endif
