/*-
# X-BASED SKEWB
#
#  SkewbGL.c
#
###
#
#  Copyright (c) 2003 - 2007	David Albert Bagley, bagleyd@tux.org
#
#		   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 *event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * Thanks goes also to Brian Paul for making it possible and inexpensive
 * to use OpenGL at home.
 *
 * Based on rubik.c by Marcelo F. Vianna <mfvianna@centroin.com.br>
 */

#ifdef HAVE_OPENGL

/* Methods file for SkewbGL */

#include "SkewbP.h"
#include "SkewbGLP.h"

static float front_shininess[] =
{60.0};
static float front_specular[] =
{0.7, 0.7, 0.7, 1.0};
static float ambient[] =
{0.0, 0.0, 0.0, 1.0};
static float diffuse[] =
{1.0, 1.0, 1.0, 1.0};
static float position0[] =
{1.0, 1.0, 1.0, 0.0};
static float position1[] =
{-1.0, -1.0, 1.0, 0.0};
static float lmodel_ambient[] =
{0.5, 0.5, 0.5, 1.0};
static float lmodel_twoside[] =
{GL_TRUE};

static float MaterialMagenta[] =
{0.7, 0.0, 0.7, 1.0};
static float MaterialCyan[] =
{0.0, 0.7, 0.7, 1.0};
static float MaterialGray[] =
{0.2, 0.2, 0.2, 1.0};

#ifdef HACK
static SkewbLocPos orthToDiag[MAXFACES][MAXORIENT][MAXORIENT] =
{
	{
		{
			{3, 0, 1},
			{5, 1, 0},
			{3, 0, 3},
			{5, 1, 2}
		},
		{
			{3, 3, 0},
			{2, 0, 1},
			{3, 3, 2},
			{2, 0, 3}
		},
		{
			{1, 0, 3},
			{2, 3, 0},
			{1, 0, 1},
			{2, 3, 2}
		},
		{
			{1, 3, 2},
			{5, 2, 1},
			{1, 3, 0},
			{5, 2, 3}
		}
	},
	{
		{
			{2, 3, 0},
			{0, 2, 1},
			{2, 3, 2},
			{0, 2, 3}
		},
		{
			{2, 2, 3},
			{4, 3, 0},
			{2, 2, 1},
			{4, 3, 2}
		},
		{
			{5, 3, 2},
			{4, 2, 3},
			{5, 3, 0},
			{4, 2, 1}
		},
		{
			{5, 2, 1},
			{0, 3, 2},
			{5, 2, 3},
			{0, 3, 0}
		}
	},
	{
		{
			{3, 3, 0},
			{0, 1, 0},
			{3, 3, 2},
			{0, 1, 2}
		},
		{
			{3, 2, 3},
			{4, 0, 1},
			{3, 2, 1},
			{4, 0, 3}
		},
		{
			{1, 1, 0},
			{4, 3, 0},
			{1, 1, 2},
			{4, 3, 2}
		},
		{
			{1, 0, 3},
			{0, 2, 1},
			{1, 0, 1},
			{0, 2, 3}
		}
	},
	{
		{
			{5, 1, 2},
			{0, 0, 3},
			{5, 1, 0},
			{0, 0, 1}
		},
		{
			{5, 0, 1},
			{4, 1, 2},
			{5, 0, 3},
			{4, 1, 0}
		},
		{
			{2, 1, 0},
			{4, 0, 1},
			{2, 1, 2},
			{4, 0, 3}
		},
		{
			{2, 0, 3},
			{0, 1, 0},
			{2, 0, 1},
			{0, 1, 2}
		}
	},
	{
		{
			{3, 2, 3},
			{2, 1, 0},
			{3, 2, 1},
			{2, 1, 2}
		},
		{
			{3, 1, 2},
			{5, 0, 1},
			{3, 1, 0},
			{5, 0, 3}
		},
		{
			{1, 2, 1},
			{5, 3, 0},
			{1, 2, 3},
			{5, 3, 2}
		},
		{
			{1, 1, 0},
			{2, 2, 1},
			{1, 1, 2},
			{2, 2, 3}
		}
	},
	{
		{
			{3, 1, 2},
			{4, 1, 0},
			{3, 1, 0},
			{4, 1, 2}
		},
		{
			{3, 0, 1},
			{0, 0, 1},
			{3, 0, 3},
			{0, 0, 3}
		},
		{
			{1, 3, 2},
			{0, 3, 0},
			{1, 3, 0},
			{0, 3, 2}
		},
		{
			{1, 2, 1},
			{4, 2, 1},
			{1, 2, 3},
			{4, 2, 3}
		}
	}
};
#endif

/*-
 * Color labels mapping:
 * =====================
 *
 *	+------+
 *	|3    0|
 *	|      |
 *	| TOP  |
 *	| (0)  |
 *	| RED  |
 *	|2    1|
 * +------+------+------+
 * |3    0|3    0|3    0|
 * |      |      |      |
 * | LEFT |FRONT |RIGHT |
 * | (1)  | (2)  | (3)  |
 * | BLUE |WHITE |GREEN |
 * |2    1|2    1|2    1|
 * +------+------+------+
 *	|3    0|
 *	|      |
 *	|BOTTOM|
 *	| (4)  |
 *	| PINK |
 *	|2    1|
 *	+------+	 +------+
 *	|3    0|	 |3 /\ 0|
 *	|      |	 | /  \ |
 *	| BACK |	 |/xxxx\|
 *	| (5)  |	 |\(N) /|
 *	|YELLOW|	 | \  / |
 *	|2    1|	 |2 \/ 1|
 *	+------+	 +------+
 *
 *  Map to 3d
 *  FRONT  => X, Y
 *  BACK   => X, Y
 *  LEFT   => Z, Y
 *  RIGHT  => Z, Y
 *  TOP    => X, Z
 *  BOTTOM => X, Z
 */

static Boolean madeCurrent = False;

#ifdef WINVER
static HGLRC hRC = NULL;
#else
static GLXContext *glXContext = (GLXContext *) NULL;
static Boolean SetValuesPuzzleGL(Widget current, Widget request, Widget renew);
static void ResizePuzzleGL(SkewbGLWidget w);
static void InitializePuzzleGL(Widget request, Widget renew);
static void ExposePuzzleGL(Widget renew, XEvent *event, Region region);
static void MovePuzzleGLTl(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLTop(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLTr(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLLeft(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLRight(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLBl(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLBottom(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLBr(SkewbGLWidget w,
	XEvent *event, char **args, int nArgs);

static char defaultTranslationsPuzzleGL[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>r: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>v: View()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListPuzzleGL[] =
{
	{(char *) "Quit", (XtActionProc) QuitPuzzle},
	{(char *) "Hide", (XtActionProc) HidePuzzle},
	{(char *) "MoveCcw", (XtActionProc) MovePuzzleCcw},
	{(char *) "MoveTl", (XtActionProc) MovePuzzleGLTl},
	{(char *) "MoveTop", (XtActionProc) MovePuzzleGLTop},
	{(char *) "MoveTr", (XtActionProc) MovePuzzleGLTr},
	{(char *) "MoveLeft", (XtActionProc) MovePuzzleGLLeft},
	{(char *) "MoveCw", (XtActionProc) MovePuzzleCw},
	{(char *) "MoveRight", (XtActionProc) MovePuzzleGLRight},
	{(char *) "MoveBl", (XtActionProc) MovePuzzleGLBl},
	{(char *) "MoveBottom", (XtActionProc) MovePuzzleGLBottom},
	{(char *) "MoveBr", (XtActionProc) MovePuzzleGLBr},
	{(char *) "Select", (XtActionProc) SelectPuzzle},
	{(char *) "Release", (XtActionProc) ReleasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) PracticePuzzleMaybe},
	{(char *) "Practice2", (XtActionProc) PracticePuzzle2},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizePuzzleMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizePuzzle2},
	{(char *) "Get", (XtActionProc) GetPuzzle},
	{(char *) "Write", (XtActionProc) WritePuzzle},
	{(char *) "Undo", (XtActionProc) UndoPuzzle},
	{(char *) "Redo", (XtActionProc) RedoPuzzle},
	{(char *) "Clear", (XtActionProc) ClearPuzzle},
	{(char *) "Randomize", (XtActionProc) RandomizePuzzle},
	{(char *) "Solve", (XtActionProc) SolvePuzzle},
	{(char *) "Practice", (XtActionProc) PracticePuzzle},
	{(char *) "Orientize", (XtActionProc) OrientizePuzzle},
	{(char *) "View", (XtActionProc) ViewPuzzle},
	{(char *) "Enter", (XtActionProc) EnterPuzzle},
	{(char *) "Leave", (XtActionProc) LeavePuzzle}
};

static XtResource resourcesPuzzleGL[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(SkewbWidget, core.width),
	 XtRString, (caddr_t) "250"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(SkewbWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.frameColor),
	 XtRString, (caddr_t) "Cyan" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[0]),
	 XtRString, (caddr_t) "Red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[1]),
	 XtRString, (caddr_t) "Blue"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[2]),
	 XtRString, (caddr_t) "White"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[3]),
	 XtRString, (caddr_t) "Green"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[4]),
	 XtRString, (caddr_t) "Pink"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[5]),
	 XtRString, (caddr_t) "Yellow"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.delay),
	 XtRString, (caddr_t) "10"},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNview, XtCView, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.view),
	 XtRString, (caddr_t) "1"},
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULTORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.practice),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULTPRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.menu),
	 XtRString, (caddr_t) "-1"},
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(SkewbWidget, skewb.select),
	 XtRCallback, (caddr_t) NULL}
};

SkewbGLClassRec skewbGLClassRec =
{
	{
		(WidgetClass) & skewbClassRec,	/* superclass */
		(char *) "SkewbGL",	/* class name */
		sizeof (SkewbGLRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializePuzzleGL,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListPuzzleGL,	/* actions */
		XtNumber(actionsListPuzzleGL),	/* num actions */
		resourcesPuzzleGL,	/* resources */
		XtNumber(resourcesPuzzleGL),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) ResizePuzzleGL,	/* resize */
		(XtExposeProc) ExposePuzzleGL,	/* expose */
		(XtSetValuesFunc) SetValuesPuzzleGL,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsPuzzleGL,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass skewbGLWidgetClass = (WidgetClass) & skewbGLClassRec;
#endif

#ifdef DEBUG
void
printCube(SkewbGLWidget w)
{
	int face, position;

	for (face = 0; face < MAXFACES; face++) {
		for (position = 0; position < MAXCUBES; position++) {
			(void) printf("%d %d  ",
				w->skewb.cubeLoc[face][position].face,
				w->skewb.cubeLoc[face][position].rotation);
		}
		(void) printf("\n");
	}
	(void) printf("\n");
}
#endif

static void
pickColor(SkewbGLWidget w, int C, int mono)
{
	float Material[] = {1.0, 1.0, 1.0, 1.0};

	switch (C) {
	case TOP_FACE:
	case LEFT_FACE:
	case FRONT_FACE:
	case RIGHT_FACE:
	case BOTTOM_FACE:
	case BACK_FACE:
		break;
	case TEST_FACE:
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			MaterialMagenta);
		return;
	default:
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			MaterialCyan);
		return;
	}
        {
#ifdef WINVER
	struct tagColor {
		int red, green, blue;
	} color;

	color.red = GetRValue(w->skewb.faceGC[C]);
	color.green = GetGValue(w->skewb.faceGC[C]);
	color.blue = GetBValue(w->skewb.faceGC[C]);
#define MAXCOLOR 0xFF
#else
	XColor color;

	color.pixel = w->skewb.faceColor[C];
	XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
		&color);
#define MAXCOLOR 0xFFFF
#endif
#ifdef DEBUG
	(void) printf("i%d %d %d\n", color.red, color.green, color.blue);
#endif
	if (mono) {
		/* really GrayScale */
		float intensity = 0.3 * color.red + 0.59 * color.green +
			0.11 * color.blue;

#ifdef DEBUG
		(void) printf("m%g\n", intensity);
#endif
		Material[0] *= (float) intensity / MAXCOLOR;
		Material[1] *= (float) intensity / MAXCOLOR;
		Material[2] *= (float) intensity / MAXCOLOR;
	} else {
		Material[0] *= (float) color.red / MAXCOLOR;
		Material[1] *= (float) color.green / MAXCOLOR;
		Material[2] *= (float) color.blue / MAXCOLOR;
	}
	}
#ifdef DEBUG
	(void) printf("f%g %g %g\n", Material[0], Material[1], Material[2]);
#endif
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, Material);
}

static Boolean
draw_stickerless_cubit(void)
{
	glBegin(GL_QUADS);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, MaterialGray);
	/* Edge of cubit */
	glNormal3f(1.0, 1.0, 0.0);
	glVertex3f(CUBEROUND, CUBELEN, -CUBEROUND);
	glVertex3f(CUBEROUND, CUBELEN, CUBEROUND);
	glVertex3f(CUBELEN, CUBEROUND, CUBEROUND);
	glVertex3f(CUBELEN, CUBEROUND, -CUBEROUND);
	glNormal3f(0.0, 1.0, 1.0);
	glVertex3f(-CUBEROUND, CUBEROUND, CUBELEN);
	glVertex3f(CUBEROUND, CUBEROUND, CUBELEN);
	glVertex3f(CUBEROUND, CUBELEN, CUBEROUND);
	glVertex3f(-CUBEROUND, CUBELEN, CUBEROUND);
	glNormal3f(1.0, 0.0, 1.0);
	glVertex3f(CUBELEN, -CUBEROUND, CUBEROUND);
	glVertex3f(CUBELEN, CUBEROUND, CUBEROUND);
	glVertex3f(CUBEROUND, CUBEROUND, CUBELEN);
	glVertex3f(CUBEROUND, -CUBEROUND, CUBELEN);
	glEnd();
	glBegin(GL_TRIANGLES);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, MaterialGray);
	/* Put sticker here */
	glNormal3f(0.0, 0.0, 1.0);
	glVertex3f(CUBEROUND, -CUBEROUND, CUBELEN);
	glVertex3f(CUBEROUND, CUBEROUND, CUBELEN);
	glVertex3f(-CUBEROUND, CUBEROUND, CUBELEN);
	glNormal3f(1.0, 0.0, 0.0);
	glVertex3f(CUBELEN, CUBEROUND, -CUBEROUND);
	glVertex3f(CUBELEN, CUBEROUND, CUBEROUND);
	glVertex3f(CUBELEN, -CUBEROUND, CUBEROUND);
	glNormal3f(0.0, 1.0, 0.0);
	glVertex3f(-CUBEROUND, CUBELEN, CUBEROUND);
	glVertex3f(CUBEROUND, CUBELEN, CUBEROUND);
	glVertex3f(CUBEROUND, CUBELEN, -CUBEROUND);
	/* Corner of cubit */
	glNormal3f(1.0, 1.0, 1.0);
	glVertex3f(CUBEROUND, CUBEROUND, CUBELEN);
	glVertex3f(CUBELEN, CUBEROUND, CUBEROUND);
	glVertex3f(CUBEROUND, CUBELEN, CUBEROUND);
	/* Sharper corners of cubit */
	glNormal3f(-1.0, 1.0, 1.0);
	glVertex3f(-CUBELEN, CUBEROUND, CUBELEN);
	glVertex3f(-CUBELEN, CUBELEN, CUBEROUND);
	glVertex3f(-CUBELEN, CUBEROUND, CUBEROUND);
	glNormal3f(1.0, -1.0, 1.0);
	glVertex3f(CUBEROUND, -CUBELEN, CUBELEN);
	glVertex3f(CUBEROUND, -CUBELEN, CUBEROUND);
	glVertex3f(CUBELEN, -CUBELEN, CUBEROUND);
	glNormal3f(1.0, 1.0, -1.0);
	glVertex3f(CUBELEN, CUBEROUND, -CUBELEN);
	glVertex3f(CUBEROUND, CUBEROUND, -CUBELEN);
	glVertex3f(CUBEROUND, CUBELEN, -CUBELEN);
	glEnd();
	glBegin(GL_POLYGON);
	glNormal3f(-1.0, 1.0, 1.0);
	glVertex3f(-CUBEROUND, CUBEROUND, CUBELEN);
	glVertex3f(-CUBEROUND, CUBELEN, CUBEROUND);
	glVertex3f(CUBEROUND, CUBELEN, -CUBEROUND);
	glVertex3f(CUBELEN, CUBEROUND, -CUBEROUND);
	glVertex3f(CUBELEN, -CUBEROUND, CUBEROUND);
	glVertex3f(CUBEROUND, -CUBEROUND, CUBELEN);
	glEnd();
	return True;
}

static Boolean
draw_stickerless_facit(void)
{
	glBegin(GL_QUADS);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, MaterialGray);
	/* Edge of facit */
	glNormal3f(0.0, 0.0, 1.0);
	glVertex3f(-FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, -FACEROUND, FACELEN);
	glVertex3f(FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, FACEROUND, FACELEN);
	glEnd();
	glBegin(GL_TRIANGLES);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, MaterialGray);
	glNormal3f(0.0, 0.0, 1.0);
	glVertex3f(0.0, -FACEROUND, FACELEN);
	glVertex3f(-FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, 0.0, 0.0);
	glVertex3f(FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, -FACEROUND, FACELEN);
	glVertex3f(0.0, 0.0, 0.0);
	glVertex3f(0.0, FACEROUND, FACELEN);
	glVertex3f(FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, 0.0, 0.0);
	glVertex3f(-FACEROUND, 0.0, FACELEN);
	glVertex3f(0.0, FACEROUND, FACELEN);
	glVertex3f(0.0, 0.0, 0.0);
	glEnd();
	return True;
}

static void
DrawOrientLine(SkewbGLWidget w, int face, int rotation,
		float size, float x, float y, float z)
{
	int err = 0;
	char *buf1, *buf2;
	float stickerLong = STICKERLONG * size;

	glTranslatef(x, y, z);
	glBegin(GL_QUADS);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, MaterialGray);
	switch (face) {
	case 0:
		switch (rotation) {
		case 0:
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, -stickerLong);
			glVertex3f(-CUT, 0.0, -stickerLong);
			break;
		case 1:
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(stickerLong, 0.0, CUT);
			glVertex3f(stickerLong, 0.0, -CUT);
			break;
		case 2:
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, stickerLong);
			glVertex3f(CUT, 0.0, stickerLong);
			break;
		case 3:
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(-stickerLong, 0.0, -CUT);
			glVertex3f(-stickerLong, 0.0, CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 1:
		switch (rotation) {
		case 0:
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, stickerLong, CUT);
			glVertex3f(0.0, stickerLong, -CUT);
			break;
		case 1:
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -CUT, stickerLong);
			glVertex3f(0.0, CUT, stickerLong);
			break;
		case 2:
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, -stickerLong, -CUT);
			glVertex3f(0.0, -stickerLong, CUT);
			break;
		case 3:
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, CUT, -stickerLong);
			glVertex3f(0.0, -CUT, -stickerLong);
			break;
		default:
			err = 1;
		}
		break;
	case 2:
		switch (rotation) {
		case 0:
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(CUT, stickerLong, 0.0);
			glVertex3f(-CUT, stickerLong, 0.0);
			break;
		case 1:
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(stickerLong, -CUT, 0.0);
			glVertex3f(stickerLong, CUT, 0.0);
			break;
		case 2:
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-CUT, -stickerLong, 0.0);
			glVertex3f(CUT, -stickerLong, 0.0);
			break;
		case 3:
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(-stickerLong, CUT, 0.0);
			glVertex3f(-stickerLong, -CUT, 0.0);
			break;
		default:
			err = 1;
		}
		break;
	case 3:
		switch (rotation) {
		case 0:
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, stickerLong, -CUT);
			glVertex3f(0.0, stickerLong, CUT);
			glVertex3f(0.0, CUT, CUT);
			break;
		case 1:
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, -CUT, -stickerLong);
			glVertex3f(0.0, CUT, -stickerLong);
			glVertex3f(0.0, CUT, -CUT);
			break;
		case 2:
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -stickerLong, CUT);
			glVertex3f(0.0, -stickerLong, -CUT);
			glVertex3f(0.0, -CUT, -CUT);
			break;
		case 3:
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, CUT, stickerLong);
			glVertex3f(0.0, -CUT, stickerLong);
			glVertex3f(0.0, -CUT, CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 4:
		switch (rotation) {
		case 0:
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(CUT, 0.0, stickerLong);
			glVertex3f(-CUT, 0.0, stickerLong);
			glVertex3f(-CUT, 0.0, CUT);
			break;
		case 1:
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(stickerLong, 0.0, -CUT);
			glVertex3f(stickerLong, 0.0, CUT);
			glVertex3f(CUT, 0.0, CUT);
			break;
		case 2:
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(-CUT, 0.0, -stickerLong);
			glVertex3f(CUT, 0.0, -stickerLong);
			glVertex3f(CUT, 0.0, -CUT);
			break;
		case 3:
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-stickerLong, 0.0, CUT);
			glVertex3f(-stickerLong, 0.0, -CUT);
			glVertex3f(-CUT, 0.0, -CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 5:
		switch (rotation) {
		case 0:
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(CUT, -stickerLong, 0.0);
			glVertex3f(-CUT, -stickerLong, 0.0);
			glVertex3f(-CUT, -CUT, 0.0);
			break;
		case 1:
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(stickerLong, CUT, 0.0);
			glVertex3f(stickerLong, -CUT, 0.0);
			glVertex3f(CUT, -CUT, 0.0);
			break;
		case 2:
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(-CUT, stickerLong, 0.0);
			glVertex3f(CUT, stickerLong, 0.0);
			glVertex3f(CUT, CUT, 0.0);
			break;
		case 3:
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-stickerLong, -CUT, 0.0);
			glVertex3f(-stickerLong, CUT, 0.0);
			glVertex3f(-CUT, CUT, 0.0);
			break;
		default:
			err = 1;
		}
		break;
	}
	glEnd();
	if (err == 1) {
		intCat(&buf1, "draw_orient_line: face ", face);
		stringCat(&buf2, buf1, ", rotation ");
		free(buf1);
		intCat(&buf1, buf2, rotation);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
}

static void
DrawFacit(SkewbGLWidget w, int face)
{
	Boolean mono = w->skewb.mono;

	glPushMatrix();
	pickColor(w, w->skewb.cubeLoc[face][MAXORIENT].face, mono);
	switch (face) {
	case TOP_FACE:
		glBegin(GL_QUADS);
		glNormal3f(0.0, 1.0, 0.0);
		glVertex3f(0.0, FSTICKERDEPTH, -FSTICKERLONG);
		glVertex3f(-FSTICKERLONG, FSTICKERDEPTH, 0.0);
		glVertex3f(0.0, FSTICKERDEPTH, FSTICKERLONG);
		glVertex3f(FSTICKERLONG, FSTICKERDEPTH, 0.0);
		glEnd();
		if (w->skewb.orient) {
			DrawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAXORIENT].rotation,
				2.0,
				0.0, FCUTDEPTH, 0.0);
		}
		break;
	case LEFT_FACE:
		glBegin(GL_QUADS);
		glNormal3f(-1.0, 0.0, 0.0);
		glVertex3f(-FSTICKERDEPTH, 0.0, -FSTICKERLONG);
		glVertex3f(-FSTICKERDEPTH, -FSTICKERLONG, 0.0);
		glVertex3f(-FSTICKERDEPTH, 0.0, FSTICKERLONG);
		glVertex3f(-FSTICKERDEPTH, FSTICKERLONG, 0.0);
		glEnd();
		if (w->skewb.orient) {
			DrawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAXORIENT].rotation,
				2.0,
				-FCUTDEPTH, 0.0, 0.0);
		}
		break;
	case FRONT_FACE:
		glBegin(GL_QUADS);
		glNormal3f(0.0, 0.0, 1.0);
		glVertex3f(-FSTICKERLONG, 0.0, FSTICKERDEPTH);
		glVertex3f(0.0, -FSTICKERLONG, FSTICKERDEPTH);
		glVertex3f(FSTICKERLONG, 0.0, FSTICKERDEPTH);
		glVertex3f(0.0, FSTICKERLONG, FSTICKERDEPTH);
		glEnd();
		if (w->skewb.orient) {
			DrawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAXORIENT].rotation,
				2.0,
				0.0, 0.0, FCUTDEPTH);
		}
		break;
	case RIGHT_FACE:
		glBegin(GL_QUADS);
		glNormal3f(1.0, 0.0, 0.0);
		glVertex3f(FSTICKERDEPTH, FSTICKERLONG, 0.0);
		glVertex3f(FSTICKERDEPTH, 0.0, FSTICKERLONG);
		glVertex3f(FSTICKERDEPTH, -FSTICKERLONG, 0.0);
		glVertex3f(FSTICKERDEPTH, 0.0, -FSTICKERLONG);
		glEnd();
		if (w->skewb.orient) {
			DrawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAXORIENT].rotation,
				2.0,
				FCUTDEPTH, 0.0, 0.0);
		}
		break;
	case BOTTOM_FACE:
		glBegin(GL_QUADS);
		glNormal3f(0.0, -1.0, 0.0);
		glVertex3f(FSTICKERLONG, -FSTICKERDEPTH, 0.0);
		glVertex3f(0.0, -FSTICKERDEPTH, FSTICKERLONG);
		glVertex3f(-FSTICKERLONG, -FSTICKERDEPTH, 0.0);
		glVertex3f(0.0, -FSTICKERDEPTH, -FSTICKERLONG);
		glEnd();
		if (w->skewb.orient) {
			DrawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAXORIENT].rotation,
				2.0,
				0.0, -FCUTDEPTH, 0.0);
		}
		break;
	case BACK_FACE:
		glBegin(GL_QUADS);
		glNormal3f(0.0, 0.0, -1.0);
		glVertex3f(0.0, FSTICKERLONG, -FSTICKERDEPTH);
		glVertex3f(FSTICKERLONG, 0.0, -FSTICKERDEPTH);
		glVertex3f(0.0, -FSTICKERLONG, -FSTICKERDEPTH);
		glVertex3f(-FSTICKERLONG, 0.0, -FSTICKERDEPTH);
		glEnd();
		if (w->skewb.orient) {
			DrawOrientLine(w, face,
				w->skewb.cubeLoc[face][MAXORIENT].rotation,
				2.0,
				0.0, 0.0, -FCUTDEPTH);
		}
		break;
	}
	glPopMatrix();
}

static void
DrawCubit(SkewbGLWidget w,
	int back, int front, int left, int right, int bottom, int top)
{
	Boolean mono = w->skewb.mono;

	if (back != NO_FACE) { /* Yellow */
		glPushMatrix();
		glBegin(GL_POLYGON);
		pickColor(w, back, mono);
		glNormal3f(0.0, 0.0, -1.0);
		if (top != NO_FACE) {
			glVertex3f(-STICKERSHORT, STICKERLONG, -STICKERDEPTH);
			glVertex3f(STICKERSHORT, STICKERLONG, -STICKERDEPTH);
		}
		if (left != NO_FACE) {
			glVertex3f(-STICKERLONG, -STICKERSHORT, -STICKERDEPTH);
			glVertex3f(-STICKERLONG, STICKERSHORT, -STICKERDEPTH);
		}
		if (bottom != NO_FACE) {
			glVertex3f(STICKERSHORT, -STICKERLONG, -STICKERDEPTH);
			glVertex3f(-STICKERSHORT, -STICKERLONG, -STICKERDEPTH);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKERLONG, STICKERSHORT, -STICKERDEPTH);
			glVertex3f(STICKERLONG, -STICKERSHORT, -STICKERDEPTH);
		}
		glEnd();
		if (w->skewb.orient) {
			if (right != NO_FACE && top != NO_FACE) {
				DrawOrientLine(w, BACK_FACE,
					w->skewb.cubeLoc[BACK_FACE][1].rotation,
					0.5,
					0.1, 0.5 * STICKERSHORT, -CUTDEPTH);
			}
			if (left != NO_FACE && top != NO_FACE) {
				DrawOrientLine(w, BACK_FACE,
					w->skewb.cubeLoc[BACK_FACE][2].rotation,
					0.5,
					-0.5 * STICKERSHORT, 0.1, -CUTDEPTH);
			}
			if (right != NO_FACE && bottom != NO_FACE) {
				DrawOrientLine(w, BACK_FACE,
					w->skewb.cubeLoc[BACK_FACE][0].rotation,
					0.5,
					0.1, -0.5 * STICKERSHORT, -CUTDEPTH);
			}
			if (left != NO_FACE && bottom != NO_FACE) {
				DrawOrientLine(w, BACK_FACE,
					w->skewb.cubeLoc[BACK_FACE][3].rotation,
					0.5,
					-0.5 * STICKERSHORT, -0.1, -CUTDEPTH);
			}
		}
		glPopMatrix();
	}
	if (front != NO_FACE) { /* White */
		glPushMatrix();
		glBegin(GL_POLYGON);
		pickColor(w, front, mono);
		glNormal3f(0.0, 0.0, 1.0);
		if (top != NO_FACE) {
			glVertex3f(STICKERSHORT, STICKERLONG, STICKERDEPTH);
			glVertex3f(-STICKERSHORT, STICKERLONG, STICKERDEPTH);
		}
		if (left != NO_FACE) {
			glVertex3f(-STICKERLONG, STICKERSHORT, STICKERDEPTH);
			glVertex3f(-STICKERLONG, -STICKERSHORT, STICKERDEPTH);
		}
		if (bottom != NO_FACE) {
			glVertex3f(-STICKERSHORT, -STICKERLONG, STICKERDEPTH);
			glVertex3f(STICKERSHORT, -STICKERLONG, STICKERDEPTH);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKERLONG, -STICKERSHORT, STICKERDEPTH);
			glVertex3f(STICKERLONG, STICKERSHORT, STICKERDEPTH);
		}
		glEnd();
		if (w->skewb.orient) {
			if (right != NO_FACE && top != NO_FACE) {
				DrawOrientLine(w, FRONT_FACE,
					w->skewb.cubeLoc[FRONT_FACE][0].rotation,
					0.5,
					0.1, 0.5 * STICKERSHORT, CUTDEPTH);
			}
			if (left != NO_FACE && top != NO_FACE) {
				DrawOrientLine(w, FRONT_FACE,
					w->skewb.cubeLoc[FRONT_FACE][3].rotation,
					0.5,
					-0.5 * STICKERSHORT, 0.1, CUTDEPTH);
			}
			if (right != NO_FACE && bottom != NO_FACE) {
				DrawOrientLine(w, FRONT_FACE,
					w->skewb.cubeLoc[FRONT_FACE][1].rotation,
					0.5,
					0.1, -0.5 * STICKERSHORT, CUTDEPTH);
			}
			if (left != NO_FACE && bottom != NO_FACE) {
				DrawOrientLine(w, FRONT_FACE,
					w->skewb.cubeLoc[FRONT_FACE][2].rotation,
					0.5,
					-0.5 * STICKERSHORT, -0.1, CUTDEPTH);
			}
		}
		glPopMatrix();
	}
	if (left != NO_FACE) { /* Blue */
		glPushMatrix();
		glBegin(GL_POLYGON);
		pickColor(w, left, mono);
		glNormal3f(-1.0, 0.0, 0.0);
		if (front != NO_FACE) {
			glVertex3f(-STICKERDEPTH, -STICKERSHORT, STICKERLONG);
			glVertex3f(-STICKERDEPTH, STICKERSHORT, STICKERLONG);
		}
		if (top != NO_FACE) {
			glVertex3f(-STICKERDEPTH, STICKERLONG, STICKERSHORT);
			glVertex3f(-STICKERDEPTH, STICKERLONG, -STICKERSHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(-STICKERDEPTH, STICKERSHORT, -STICKERLONG);
			glVertex3f(-STICKERDEPTH, -STICKERSHORT, -STICKERLONG);
		}
		if (bottom != NO_FACE) {
			glVertex3f(-STICKERDEPTH, -STICKERLONG, -STICKERSHORT);
			glVertex3f(-STICKERDEPTH, -STICKERLONG, STICKERSHORT);
		}
		glEnd();
		if (w->skewb.orient) {
			if (front != NO_FACE && top != NO_FACE) {
				DrawOrientLine(w, LEFT_FACE,
					w->skewb.cubeLoc[LEFT_FACE][0].rotation,
					0.5,
					-CUTDEPTH, 0.1, 0.5 * STICKERSHORT);
			}
			if (back != NO_FACE && top != NO_FACE) {
				DrawOrientLine(w, LEFT_FACE,
					w->skewb.cubeLoc[LEFT_FACE][3].rotation,
					0.5,
					-CUTDEPTH, 0.5 * STICKERSHORT, -0.1);
			}
			if (front != NO_FACE && bottom != NO_FACE) {
				DrawOrientLine(w, LEFT_FACE,
					w->skewb.cubeLoc[LEFT_FACE][1].rotation,
					0.5,
					-CUTDEPTH, -0.1, 0.5 * STICKERSHORT);
			}
			if (back != NO_FACE && bottom != NO_FACE) {
				DrawOrientLine(w, LEFT_FACE,
					w->skewb.cubeLoc[LEFT_FACE][2].rotation,
					0.5,
					-CUTDEPTH, -0.5 * STICKERSHORT, -0.1);
			}
		}
		glPopMatrix();
	}
	if (right != NO_FACE) { /* Green */
		glPushMatrix();
		glBegin(GL_POLYGON);
		pickColor(w, right, mono);
		glNormal3f(1.0, 0.0, 0.0);
		if (front != NO_FACE) {
			glVertex3f(STICKERDEPTH, STICKERSHORT, STICKERLONG);
			glVertex3f(STICKERDEPTH, -STICKERSHORT, STICKERLONG);
		}
		if (top != NO_FACE) {
			glVertex3f(STICKERDEPTH, STICKERLONG, -STICKERSHORT);
			glVertex3f(STICKERDEPTH, STICKERLONG, STICKERSHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(STICKERDEPTH, -STICKERSHORT, -STICKERLONG);
			glVertex3f(STICKERDEPTH, STICKERSHORT, -STICKERLONG);
		}
		if (bottom != NO_FACE) {
			glVertex3f(STICKERDEPTH, -STICKERLONG, STICKERSHORT);
			glVertex3f(STICKERDEPTH, -STICKERLONG, -STICKERSHORT);
		}
		glEnd();
		if (w->skewb.orient) {
			if (front != NO_FACE && top != NO_FACE) {
				DrawOrientLine(w, RIGHT_FACE,
					w->skewb.cubeLoc[RIGHT_FACE][3].rotation,
					0.5,
					CUTDEPTH, 0.1, 0.5 * STICKERSHORT);
			}
			if (back != NO_FACE && top != NO_FACE) {
				DrawOrientLine(w, RIGHT_FACE,
					w->skewb.cubeLoc[RIGHT_FACE][0].rotation,
					0.5,
					CUTDEPTH, 0.5 * STICKERSHORT, -0.1);
			}
			if (front != NO_FACE && bottom != NO_FACE) {
				DrawOrientLine(w, RIGHT_FACE,
					w->skewb.cubeLoc[RIGHT_FACE][2].rotation,
					0.5,
					CUTDEPTH, -0.1, 0.5 * STICKERSHORT);
			}
			if (back != NO_FACE && bottom != NO_FACE) {
				DrawOrientLine(w, RIGHT_FACE,
					w->skewb.cubeLoc[RIGHT_FACE][1].rotation,
					0.5,
					CUTDEPTH, -0.5 * STICKERSHORT, -0.1);
			}
		}
		glPopMatrix();
	}
	if (bottom != NO_FACE) { /* Pink */
		glPushMatrix();
		glBegin(GL_POLYGON);
		pickColor(w, bottom, mono);
		glNormal3f(0.0, -1.0, 0.0);
		if (left != NO_FACE) {
			glVertex3f(-STICKERLONG, -STICKERDEPTH, STICKERSHORT);
			glVertex3f(-STICKERLONG, -STICKERDEPTH, -STICKERSHORT);
		}
		if (front != NO_FACE) {
			glVertex3f(STICKERSHORT, -STICKERDEPTH, STICKERLONG);
			glVertex3f(-STICKERSHORT, -STICKERDEPTH, STICKERLONG);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKERLONG, -STICKERDEPTH, -STICKERSHORT);
			glVertex3f(STICKERLONG, -STICKERDEPTH, STICKERSHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(-STICKERSHORT, -STICKERDEPTH, -STICKERLONG);
			glVertex3f(STICKERSHORT, -STICKERDEPTH, -STICKERLONG);
		}
		glEnd();
		if (w->skewb.orient) {
			if (right != NO_FACE && back != NO_FACE) {
				DrawOrientLine(w, BOTTOM_FACE,
					w->skewb.cubeLoc[BOTTOM_FACE][1].rotation,
					0.5,
					0.5 * STICKERSHORT, -CUTDEPTH, -0.1);
			}
			if (left != NO_FACE && back != NO_FACE) {
				DrawOrientLine(w, BOTTOM_FACE,
					w->skewb.cubeLoc[BOTTOM_FACE][2].rotation,
					0.5,
					-0.5 * STICKERSHORT, -CUTDEPTH, -0.1);
			}
			if (right != NO_FACE && front != NO_FACE) {
				DrawOrientLine(w, BOTTOM_FACE,
					w->skewb.cubeLoc[BOTTOM_FACE][0].rotation,
					0.5,
					0.5 * STICKERSHORT, -CUTDEPTH, 0.1);
			}
			if (left != NO_FACE && front != NO_FACE) {
				DrawOrientLine(w, BOTTOM_FACE,
					w->skewb.cubeLoc[BOTTOM_FACE][3].rotation,
					0.5,
					-0.5 * STICKERSHORT, -CUTDEPTH, 0.1);
			}
		}
		glPopMatrix();
	}
	if (top != NO_FACE) { /* Red */
		glPushMatrix();
		glBegin(GL_POLYGON);
		pickColor(w, top, mono);
		glNormal3f(0.0, 1.0, 0.0);
		if (left != NO_FACE) {
			glVertex3f(-STICKERLONG, STICKERDEPTH, -STICKERSHORT);
			glVertex3f(-STICKERLONG, STICKERDEPTH, STICKERSHORT);
		}
		if (front != NO_FACE) {
			glVertex3f(-STICKERSHORT, STICKERDEPTH, STICKERLONG);
			glVertex3f(STICKERSHORT, STICKERDEPTH, STICKERLONG);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKERLONG, STICKERDEPTH, STICKERSHORT);
			glVertex3f(STICKERLONG, STICKERDEPTH, -STICKERSHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(STICKERSHORT, STICKERDEPTH, -STICKERLONG);
			glVertex3f(-STICKERSHORT, STICKERDEPTH, -STICKERLONG);
		}
		glEnd();
		if (w->skewb.orient) {
			if (right != NO_FACE && back != NO_FACE) {
				DrawOrientLine(w, TOP_FACE,
					w->skewb.cubeLoc[TOP_FACE][0].rotation,
					0.5,
					0.5 * STICKERSHORT, CUTDEPTH, -0.1);
			}
			if (left != NO_FACE && back != NO_FACE) {
				DrawOrientLine(w, TOP_FACE,
					w->skewb.cubeLoc[TOP_FACE][3].rotation,
					0.5,
					-0.5 * STICKERSHORT, CUTDEPTH, -0.1);
			}
			if (right != NO_FACE && front != NO_FACE) {
				DrawOrientLine(w, TOP_FACE,
					w->skewb.cubeLoc[TOP_FACE][1].rotation,
					0.5,
					0.5 * STICKERSHORT, CUTDEPTH, 0.1);
			}
			if (left != NO_FACE && front != NO_FACE) {
				DrawOrientLine(w, TOP_FACE,
					w->skewb.cubeLoc[TOP_FACE][2].rotation,
					0.5,
					-0.5 * STICKERSHORT, CUTDEPTH, 0.1);
			}
		}
		glPopMatrix();
	}
}

static Boolean
DrawCube(SkewbGLWidget w)
{
#define S1 1
#define DRAW_STICKERLESS_FACIT() if (!draw_stickerless_facit()) return False
#define DRAW_STICKERLESS_CUBIT() if (!draw_stickerless_cubit()) return False
	SkewbSlice slice;
	GLfloat rotateStep;
	int i;

	/*if (w->skewbGL.movement.face == NO_FACE) */ {
		slice.face = NO_FACE;
		slice.rotation = NO_ROTATION;
#ifdef FIXME
	} else {
		convertMove(w, w->skewbGL.movement, &slice);
#endif
	}
	rotateStep = (slice.rotation == CCW) ? w->skewbGL.rotateStep :
		-w->skewbGL.rotateStep;

/*-
 * The glRotatef() routine transforms the coordinate system for every future
 * vertex specification (this is not so simple, but by now comprehending this
 * is sufficient). So if you want to rotate the inner slice, you can draw
 * one slice, rotate the anglestep for the centerslice, draw the inner slice,
 * rotate reversely and draw the other slice.
 * There is a sequence for drawing cubies for each axis being moved...
 */
	switch (slice.face) {
	case NO_FACE:
	case TOP_FACE:	/* BOTTOM_FACE too */
		glPushMatrix();
		glRotatef(rotateStep, 0, 1, 0);

		/* glTranslatef(S1, 0, S1); */
		glPushMatrix();
		DRAW_STICKERLESS_FACIT();
		glPopMatrix();
		glPushMatrix();
		glRotatef(90.0, 0, 1, 0);
		DRAW_STICKERLESS_FACIT();
		glPopMatrix();
		glPushMatrix();
		glRotatef(180.0, 0, 1, 0);
		DRAW_STICKERLESS_FACIT();
		glPopMatrix();
		glPushMatrix();
		glRotatef(-90.0, 0, 1, 0);
		DRAW_STICKERLESS_FACIT();
		glPopMatrix();
		glPushMatrix();
		glRotatef(90.0, 1, 0, 0);
		DRAW_STICKERLESS_FACIT();
		glPopMatrix();
		glPushMatrix();
		glRotatef(-90.0, 1, 0, 0);
		DRAW_STICKERLESS_FACIT();
		glPopMatrix();
		for (i = 0; i < MAXFACES; i++) {
			glPushMatrix();
			DrawFacit(w, i);
			glPopMatrix();
		}
		glTranslatef(-0.5, -0.5, -0.5);
		glPushMatrix();
		glRotatef(90.0, 0, 1, 0);
		glRotatef(180.0, 1, 0, 0);
		DRAW_STICKERLESS_CUBIT();
		glPopMatrix();
		DrawCubit(w, w->skewb.cubeLoc[BACK_FACE][3].face,
			NO_FACE,
			w->skewb.cubeLoc[LEFT_FACE][2].face,
			NO_FACE,
			w->skewb.cubeLoc[BOTTOM_FACE][2].face,
			NO_FACE); /* BLL */
		glTranslatef(0, 0, S1);
		glPushMatrix();
		glRotatef(180.0, 0, 0, 1);
		DRAW_STICKERLESS_CUBIT();
		glPopMatrix();
		DrawCubit(w, NO_FACE,
			w->skewb.cubeLoc[FRONT_FACE][2].face,
			w->skewb.cubeLoc[LEFT_FACE][1].face,
			NO_FACE,
			w->skewb.cubeLoc[BOTTOM_FACE][3].face,
			NO_FACE); /* FLL */
		glTranslatef(S1, 0, -S1);
		glPushMatrix();
		glRotatef(90.0, 0, 1, 0);
		glRotatef(90.0, 1, 0, 0);
		DRAW_STICKERLESS_CUBIT();
		glPopMatrix();
		DrawCubit(w, w->skewb.cubeLoc[BACK_FACE][0].face,
			NO_FACE, NO_FACE,
			w->skewb.cubeLoc[RIGHT_FACE][1].face,
			w->skewb.cubeLoc[BOTTOM_FACE][1].face,
			NO_FACE); /* BLR */
		glTranslatef(0, 0, S1);
		glPushMatrix();
		glRotatef(90.0, 1, 0, 0);
		DRAW_STICKERLESS_CUBIT();
		glPopMatrix();
		DrawCubit(w, NO_FACE,
			w->skewb.cubeLoc[FRONT_FACE][1].face,
			NO_FACE,
			w->skewb.cubeLoc[RIGHT_FACE][2].face,
			w->skewb.cubeLoc[BOTTOM_FACE][0].face,
			NO_FACE); /* FLR */
		glPopMatrix();
		glPushMatrix();
		glTranslatef(-0.5, 0.5, -0.5);
		glPushMatrix();
		glRotatef(90.0, 0, -1, 0);
		glRotatef(90.0, 0, 0, 1);
		DRAW_STICKERLESS_CUBIT();
		glPopMatrix();
		DrawCubit(w,
			w->skewb.cubeLoc[BACK_FACE][2].face,
			NO_FACE,
			w->skewb.cubeLoc[LEFT_FACE][3].face,
			NO_FACE, NO_FACE,
			w->skewb.cubeLoc[TOP_FACE][3].face); /* BUL */
		glTranslatef(0, 0, S1);
		glPushMatrix();
		glRotatef(90.0, 0, 0, 1);
		DRAW_STICKERLESS_CUBIT();
		glPopMatrix();
		DrawCubit(w, NO_FACE,
			w->skewb.cubeLoc[FRONT_FACE][3].face,
			w->skewb.cubeLoc[LEFT_FACE][0].face,
			NO_FACE, NO_FACE,
			w->skewb.cubeLoc[TOP_FACE][2].face); /* FUL */
		glTranslatef(S1, 0, -S1);
		glPushMatrix();
		glRotatef(90.0, 0, 1, 0);
		DRAW_STICKERLESS_CUBIT();
		glPopMatrix();
		DrawCubit(w,
			w->skewb.cubeLoc[BACK_FACE][1].face,
			NO_FACE, NO_FACE,
			w->skewb.cubeLoc[RIGHT_FACE][0].face,
			NO_FACE,
			w->skewb.cubeLoc[TOP_FACE][0].face); /* BUR */
		glTranslatef(0, 0, S1);
		DRAW_STICKERLESS_CUBIT();
		DrawCubit(w, NO_FACE,
			w->skewb.cubeLoc[FRONT_FACE][0].face,
			NO_FACE,
			w->skewb.cubeLoc[RIGHT_FACE][3].face,
			NO_FACE,
			w->skewb.cubeLoc[TOP_FACE][1].face); /* FUR */
		glPopMatrix();
		break;
	}
	return True;
#undef S1
}

void
DrawAllPiecesGL(SkewbGLWidget w)
{
#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("Draw All GL error");
	}
#endif
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glPushMatrix();
	glTranslatef(0.0, 0.0, -10.0);
	if (w->core.height > w->core.width)
		glScalef(SCALE, SCALE * w->core.width / w->core.height, SCALE);
	else
		glScalef(SCALE * w->core.height / w->core.width, SCALE, SCALE);
	switch (w->skewb.view) {
	case 0: /* >-  vertical */
		glRotatef(w->skewbGL.step / 2, 0, 1, 0);
		glRotatef(w->skewbGL.step / 2, 1, 0, 0);
		break;
	case 1: /* Y  !vertical */
		glRotatef(w->skewbGL.step / 2, 1, 0, 0);
		glRotatef(w->skewbGL.step / 2, 0, 1, 0);
		break;
	case 2: /* -<  vertical */
		glRotatef(2 * w->skewbGL.step, 0, 0, 1);
		glRotatef(5 * w->skewbGL.step / 2, 0, 1, 0);
		glRotatef(w->skewbGL.step / 2, 1, 0, 0);
		break;
	case 3: /* ^  !vertical */
		glRotatef(5 * w->skewbGL.step / 2, 0, 1, 0);
		glRotatef(w->skewbGL.step / 2, 1, 0, 1);
		break;
	default:
		break;
	}
	if (!DrawCube(w)) {
		DISPLAY_WARNING("DrawCube error");
	}
	glPopMatrix();
	glFlush();
#ifdef WINVER
	SwapBuffers(w->core.hDC);
#else
	glXSwapBuffers(XtDisplay(w), XtWindow(w));
#endif
}

void
DrawFrameGL(SkewbGLWidget w, Boolean focus)
{
	/* UNUSED */
}

Boolean
SelectPiecesGL(SkewbGLWidget w, int x, int y, int *face, int *position)
{
	/* UNUSED */
	return True;
}

#ifdef WINVER
Boolean setupPixelFormat(SkewbGLWidget w, BYTE type, DWORD flags)
{
	PIXELFORMATDESCRIPTOR pfd;
	int pixelFormat;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));
	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | flags;
	pfd.dwLayerMask = PFD_MAIN_PLANE;
	/*
	pfd.cColorBits = 8;
	pfd.cDepthBits = 16;*/
	pfd.iPixelType = type;
	pfd.cColorBits = 24;
	pfd.cDepthBits = 32;

	if ((pixelFormat = ChoosePixelFormat(w->core.hDC, &pfd)) == 0) {
		DWORD err;
		char *buf1;

		err = GetLastError();
		/* 6 ERROR_INVALID_HANDLE */
		intCat(&buf1, "ChoosePixelFormat failed ", err);
		DISPLAY_WARNING(buf1);
		free(buf1);
		return FALSE;
	}

	if (SetPixelFormat(w->core.hDC, pixelFormat, &pfd) == FALSE) {
		MessageBox(NULL, "SetPixelFormat failed", "Error", MB_OK);
		return FALSE;
	}
	DescribePixelFormat(w->core.hDC, pixelFormat,
		sizeof(PIXELFORMATDESCRIPTOR), &pfd);
	return TRUE;
}

#else

static Boolean
SetValuesPuzzleGL(Widget current, Widget request, Widget renew)
{
	SkewbGLWidget c = (SkewbGLWidget) current, w = (SkewbGLWidget) renew;
	Boolean redraw = False;

	if (w->skewb.view != c->skewb.view) {
		ResizePuzzleGL(w);
		redraw = True;
	}
#if 0
	if (w->skewbGL.cubeSizex.x != c->skewbGL.cubeSizex.x ||
			w->skewbGL.cubeSizey.x != c->skewbGL.cubeSizey.x ||
			w->skewbGL.cubeSizez.x != c->skewbGL.cubeSizez.x ||
			w->skewbGL.cubeSizex.y != c->skewbGL.cubeSizex.y ||
			w->skewbGL.cubeSizey.y != c->skewbGL.cubeSizey.y ||
			w->skewbGL.cubeSizez.y != c->skewbGL.cubeSizez.y) {
		ResizePuzzleGL(w);
		redraw = True;
	}
#endif
	return (redraw);
}

static GLXContext *
initGL(SkewbGLWidget w) {
	XVisualInfo xviIn, *xviOut;
	int numVis;

	/*XGetWindowAttributes(XtDisplay(w), XtWindow(w), &xgwa); */
	xviIn.screen = DefaultScreen(XtDisplay(w));
	xviIn.visualid = XVisualIDFromVisual(DefaultVisual(XtDisplay(w),
		xviIn.screen));
	xviOut = XGetVisualInfo(XtDisplay(w), VisualScreenMask | VisualIDMask,
		&xviIn, &numVis);
	if (!xviOut) {
		XtWarning("Could not get XVisualInfo");
		return (GLXContext *) NULL;
	}
	if (glXContext)
		free(glXContext);
	if ((glXContext = (GLXContext *) malloc(sizeof (GLXContext))) ==
			NULL) {
		DISPLAY_ERROR("Not enough memory for glx info, exiting.");
	}
	*glXContext = glXCreateContext(XtDisplay(w), xviOut, 0, GL_TRUE);
	(void) XFree((char *) xviOut);
	if (!*glXContext) {
		XtWarning("Could not create rendering context");
		return (GLXContext *) NULL;
	}
	return glXContext;
}
#endif

#ifndef WINVER
static
#endif
void
ResizePuzzleGL(SkewbGLWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	glViewport(0, 0, (GLint) w->core.width, (GLint) w->core.height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glFrustum(-1.0, 1.0, -1.0, 1.0, 5.0, 15.0);
	glMatrixMode(GL_MODELVIEW);
}

static void
makeCurrentGL(SkewbGLWidget w)
{
	GLboolean rgbaMode;

#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("GL error");
	}
#endif
	madeCurrent = True;
	/* True Color junk */
	glGetBooleanv(GL_RGBA_MODE, &rgbaMode);
	if (!rgbaMode) {
#ifdef WINVER
		glClearIndex(0.0);
#else
		glIndexi(WhitePixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
		glClearIndex((float) BlackPixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
#endif
	}
	ResizePuzzleGL(w);

	glDrawBuffer(GL_BACK);
	glClearDepth(1.0);
	glClearColor(0.0, 0.0, 0.0, 1.0);
	glColor3f(1.0, 1.0, 1.0);

	glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT0, GL_POSITION, position0);
	glLightfv(GL_LIGHT1, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT1, GL_POSITION, position1);
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
	glLightModelfv(GL_LIGHT_MODEL_TWO_SIDE, lmodel_twoside);
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glEnable(GL_LIGHT1);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_NORMALIZE);
	glEnable(GL_CULL_FACE);
	glShadeModel(GL_FLAT);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, front_shininess);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, front_specular);

	w->skewbGL.angleStep = 90.0;
	w->skewbGL.step = 90;
#if 0
	w->skewbGL.vx = 0.05;
	if (LRAND() & 1)
		w->skewbGL.vx *= -1.0;
	w->skewbGL.vy = 0.05;
	if (LRAND() & 1)
		w->skewbGL.vy *= -1.0;
#endif
	w->skewbGL.rotateStep = 0.0;
}

#ifndef WINVER
static void
InitializePuzzleGL(Widget request, Widget renew)
{
	SkewbGLWidget w = (SkewbGLWidget) renew;

	SetAllColors((SkewbWidget) w);
	(void) initGL(w);
	w->skewb.dim = 4; /* 2 and 3 already taken */
	ResizePuzzleGL(w);
}
#endif

#ifndef WINVER
static
#endif
void
ExposePuzzleGL(
#ifdef WINVER 
SkewbGLWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifdef WINVER
	if (hRC == NULL) {
#if 1
		BYTE type = PFD_TYPE_RGBA;
#else
		BYTE type = PFD_TYPE_COLORINDEX;
#endif
		DWORD flags = PFD_DOUBLEBUFFER;
		(void) setupPixelFormat(w, type, flags);
		hRC = wglCreateContext(w->core.hDC);
	}
#else
	SkewbGLWidget w = (SkewbGLWidget) renew;

	if (!w->core.visible)
		return;
#endif
	if (!madeCurrent) {
		makeCurrentGL(w);
	}
	DrawFrameGL(w, w->skewb.focus);
	DrawAllPiecesGL(w);
}

#ifndef WINVER
static void
MovePuzzleGLTl(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TL,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLTop(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TOP,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLTr(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TR,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLLeft(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		LEFT,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLRight(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		RIGHT,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLBl(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BL,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLBottom(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BOTTOM,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLBr(SkewbGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BR,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}
#endif
#endif
