/*
 * Copyright 2005-2007 Luc Verhaegen.
 * Copyright 2004      The Unichrome Project  [unichrome.sf.net]
 * Copyright 1998-2003 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2003 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
/*
 * The "The Unichrome Project" copyright only really applies to the colourspace
 * code. No additional copyright claims were ever made on the bandwidth code.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef XvExtension

#include "xf86.h"
#include "xf86_OSproc.h"
#include "xf86Resources.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#include <unistd.h>
#include <errno.h>
#endif

#include "compiler.h"
#include "xf86PciInfo.h"
#include "xf86Pci.h"
#include "xf86fbman.h"
#include "regionstr.h"
#include "via_driver.h"

#include "X11/extensions/Xv.h"
#include "xaa.h"
#include "xaalocal.h"
#include "dixstruct.h"
#include "via_memory.h"
#include "via_videoregs.h"
#include "via_id.h"
#include "fourcc.h"
#include "via_vgahw.h"

/*
 * Number of loops any idle loop should run before a debug message is produced.
 */
#define IDLELOOP_LEVEL 0

#define VIA_MAX_XVIMAGE_X 1920
#define VIA_MAX_XVIMAGE_Y 1200

#define ALIGN_TO(f, alignment) (((f) + ((alignment)-1)) & ~((alignment)-1))

#define  XV_IMAGE 0

static XF86VideoEncodingRec DummyEncoding[1] = {
    {XV_IMAGE, "XV_IMAGE", VIA_MAX_XVIMAGE_X, VIA_MAX_XVIMAGE_Y, {1, 1}},
};

#define NUM_FORMATS_G 9

static XF86VideoFormatRec FormatsG[NUM_FORMATS_G] =
{
  { 8, TrueColor }, /* Dithered */
  { 8, PseudoColor }, /* Using .. */
  { 8, StaticColor },
  { 8, GrayScale },
  { 8, StaticGray }, /* .. TexelLUT */
  {16, TrueColor},
  {24, TrueColor},
  {16, DirectColor},
  {24, DirectColor}
};

#define MAKE_ATOM(a) MakeAtom(a, sizeof(a) - 1, TRUE)
static Atom xvBrightness, xvContrast, xvColorKey, xvHue, xvSaturation,
    xvAutoPaint, xvOpacity;

#define NUM_ATTRIBUTES_G 7

static XF86AttributeRec AttributesG[NUM_ATTRIBUTES_G] =
{
   {XvSettable | XvGettable,    0, 0x00FFFFFF, "XV_COLORKEY"},
   {XvSettable | XvGettable,    0,      10000, "XV_BRIGHTNESS"},
   {XvSettable | XvGettable,    0,      20000, "XV_CONTRAST"},
   {XvSettable | XvGettable,    0,      20000, "XV_SATURATION"},
   {XvSettable | XvGettable, -180,        180, "XV_HUE"},
   {XvSettable | XvGettable,    0,          1, "XV_AUTOPAINT_COLORKEY"},
   {XvSettable | XvGettable,    0,       0x0F, "XV_OPACITY"}
};

/*
 * I'm unable to set the ordering of packed YUV, so only YUY2 is possible.
 * A conversion routine could be implemented, but it is more efficient
 * to let the client ignore/convert it. -- Luc.
 */

/*
 * VIA seems to have it's RGB components reversed. RGBA tends to mean that
 * A is the high byte and R the low. Sadly, VIA is not alone in this, and
 * there is no consensus between devices here either.
 *
 * This severely reduces the chances of any inclusion in fourcc.h. But in
 * case this does happen, ignore it and define our own. -- Luc.
 */

/*
 * RGB 555
 */
#ifndef FOURCC_RV15
#define FOURCC_RV15 0x35315652
#endif

#ifdef XVIMAGE_RV15
#undef XVIMAGE_RV15
#endif

#define XVIMAGE_RV15 \
    { \
	FOURCC_RV15, \
        XvRGB, \
        LSBFirst, \
        {'R','V','1','5'}, \
	16, \
	XvPacked, \
	1, \
	15, \
        0x7C00, 0x03E0, 0x001F, \
	0, 0, 0, \
	0, 0, 0, \
	0, 0, 0, \
        {'R','V','B' }, \
	XvTopToBottom \
   }

/*
 * RGB 565
 */
#ifndef FOURCC_RV16
#define FOURCC_RV16 0x36315652
#endif

#ifdef XVIMAGE_RV16
#undef XVIMAGE_RV16
#endif

#define XVIMAGE_RV16 \
   { \
	FOURCC_RV16, \
        XvRGB, \
	LSBFirst, \
        {'R','V','1','6' }, \
	16, \
	XvPacked, \
	1, \
	16, \
        0xF800, 0x07E0, 0x001F, \
	0, 0, 0, \
	0, 0, 0, \
	0, 0, 0, \
        {'R','V','B' }, \
	XvTopToBottom \
   }

/*
 * RGB 888
 */
#ifndef FOURCC_RV32
#define FOURCC_RV32 0x32335652
#endif

#ifdef XVIMAGE_RV32
#undef XVIMAGE_RV32
#endif

#define XVIMAGE_RV32 \
    { \
        FOURCC_RV32, \
        XvRGB, \
        LSBFirst, \
        { 'R', 'V' ,'3', '2' }, \
        32, \
        XvPacked, \
        1, \
        24, \
        0xFF0000, 0x00FF00, 0x0000FF, \
        0, 0, 0, \
        0, 0, 0, \
        0, 0, 0, \
        { 'R', 'V', 'B' }, \
        XvTopToBottom \
    }

#define NUM_IMAGES_G 6
static XF86ImageRec ImagesG[NUM_IMAGES_G] =
{
    XVIMAGE_YUY2,
    XVIMAGE_YV12,
    XVIMAGE_I420,
    XVIMAGE_RV15,
    XVIMAGE_RV16,
    XVIMAGE_RV32
};

/*
 *
 * Dump Video registers.
 *
 */
static void
ViaSwovPrintRegs(struct ViaSwov *Swov, const char *function)
{
    VIAFUNC(Swov->scrnIndex);

    ViaDebug(Swov->scrnIndex, "%s: Video Map:\n", function);
    ViaDebug(Swov->scrnIndex, "Flags: 0x%08X.\n", Swov->Video->Flags);
    ViaDebug(Swov->scrnIndex, "Status: 0x%08X.\n", Swov->Video->Status);
    ViaDebug(Swov->scrnIndex, "CRTC1StartAddress: 0x%08X.\n", Swov->Video->CRTC1StartAddress);
    ViaDebug(Swov->scrnIndex, "CRTC2StartAddress2: 0x%08X.\n", Swov->Video->CRTC2StartAddress);
    ViaDebug(Swov->scrnIndex, "ColorKey: 0x%08X.\n", Swov->Video->ColorKey);
    ViaDebug(Swov->scrnIndex, "ColorKeySecond: 0x%08X.\n", Swov->Video->ColorKeySecond);
    ViaDebug(Swov->scrnIndex, "ChromaKeyLow: 0x%08X.\n", Swov->Video->ChromaKeyLow);
    ViaDebug(Swov->scrnIndex, "ChromaKeyHigh: 0x%08X.\n", Swov->Video->ChromaKeyHigh);
    ViaDebug(Swov->scrnIndex, "HighControl: 0x%08X.\n", Swov->Video->HighControl);
    ViaDebug(Swov->scrnIndex, "Compose: 0x%08X.\n", Swov->Video->Compose);

    ViaDebug(Swov->scrnIndex, "AlphaWindowStart: 0x%08X.\n", Swov->Video->AlphaWindowStart);
    ViaDebug(Swov->scrnIndex, "AlphaWindowEnd: 0x%08X.\n", Swov->Video->AlphaWindowEnd);
    ViaDebug(Swov->scrnIndex, "AlphaControl: 0x%08X.\n", Swov->Video->AlphaControl);
    ViaDebug(Swov->scrnIndex, "AlphaStride: 0x%08X.\n", Swov->Video->AlphaStride);
    ViaDebug(Swov->scrnIndex, "AlphaAddress: 0x%08X.\n", Swov->Video->AlphaAddress);
    ViaDebug(Swov->scrnIndex, "AlphaColorHigh: 0x%08X.\n", Swov->Video->AlphaColorHigh);

    ViaDebug(Swov->scrnIndex, "Video1Control: 0x%08X.\n", Swov->Video->Video1Control);
    ViaDebug(Swov->scrnIndex, "Video1Fetch: 0x%08X.\n", Swov->Video->Video1Fetch);
    ViaDebug(Swov->scrnIndex, "Video1Stride: 0x%08X.\n", Swov->Video->Video1Stride);
    ViaDebug(Swov->scrnIndex, "Video1WindowStart: 0x%08X.\n", Swov->Video->Video1WindowStart);
    ViaDebug(Swov->scrnIndex, "Video1WindowEnd: 0x%08X.\n", Swov->Video->Video1WindowEnd);
    ViaDebug(Swov->scrnIndex, "Video1Zoom: 0x%08X.\n", Swov->Video->Video1Zoom);
    ViaDebug(Swov->scrnIndex, "Video1Minify: 0x%08X.\n", Swov->Video->Video1Minify);
    ViaDebug(Swov->scrnIndex, "Video1Fifo: 0x%08X.\n", Swov->Video->Video1Fifo);
    ViaDebug(Swov->scrnIndex, "Video1SrcSize: 0x%08X.\n", Swov->Video->Video1SrcSize);
    ViaDebug(Swov->scrnIndex, "Video1DisplayTemp: 0x%08X.\n", Swov->Video->Video1DisplayTemp);
    ViaDebug(Swov->scrnIndex, "Video1ColorSpace1: 0x%08X.\n", Swov->Video->Video1ColorSpace1);
    ViaDebug(Swov->scrnIndex, "Video1ColorSpace2: 0x%08X.\n", Swov->Video->Video1ColorSpace2);
    ViaDebug(Swov->scrnIndex, "Video1Opacity: 0x%08X.\n", Swov->Video->Video1Opacity);
    ViaDebug(Swov->scrnIndex, "Video1Address0: 0x%08X.\n", Swov->Video->Video1Address0);
    ViaDebug(Swov->scrnIndex, "Video1Address1: 0x%08X.\n", Swov->Video->Video1Address1);
    ViaDebug(Swov->scrnIndex, "Video1Address2: 0x%08X.\n", Swov->Video->Video1Address2);
    ViaDebug(Swov->scrnIndex, "Video1Address3: 0x%08X.\n", Swov->Video->Video1Address3);
    ViaDebug(Swov->scrnIndex, "Video1UAddress0: 0x%08X.\n", Swov->Video->Video1UAddress0);
    ViaDebug(Swov->scrnIndex, "Video1UAddress1: 0x%08X.\n", Swov->Video->Video1UAddress1);
    ViaDebug(Swov->scrnIndex, "Video1UAddress2: 0x%08X.\n", Swov->Video->Video1UAddress2);
    ViaDebug(Swov->scrnIndex, "Video1UAddress3: 0x%08X.\n", Swov->Video->Video1UAddress3);
    ViaDebug(Swov->scrnIndex, "Video1VAddress0: 0x%08X.\n", Swov->Video->Video1VAddress0);
    ViaDebug(Swov->scrnIndex, "Video1VAddress1: 0x%08X.\n", Swov->Video->Video1VAddress1);
    ViaDebug(Swov->scrnIndex, "Video1VAddress2: 0x%08X.\n", Swov->Video->Video1VAddress2);
    ViaDebug(Swov->scrnIndex, "Video1VAddress3: 0x%08X.\n", Swov->Video->Video1VAddress3);

    ViaDebug(Swov->scrnIndex, "Video3SrcWidth: 0x%08X.\n", Swov->Video->Video3SrcWidth);
    ViaDebug(Swov->scrnIndex, "Video3ColorKey: 0x%08X.\n", Swov->Video->Video3ColorKey);
    ViaDebug(Swov->scrnIndex, "Video3Opacity: 0x%08X.\n", Swov->Video->Video3Opacity);
    ViaDebug(Swov->scrnIndex, "Video3Control: 0x%08X.\n", Swov->Video->Video3Control);
    ViaDebug(Swov->scrnIndex, "Video3Stride: 0x%08X.\n", Swov->Video->Video3Stride);
    ViaDebug(Swov->scrnIndex, "Video3WindowStart: 0x%08X.\n", Swov->Video->Video3WindowStart);
    ViaDebug(Swov->scrnIndex, "Video3WindowEnd: 0x%08X.\n", Swov->Video->Video3WindowEnd);
    ViaDebug(Swov->scrnIndex, "Video3AlphaFetch: 0x%08X.\n", Swov->Video->Video3AlphaFetch);
    ViaDebug(Swov->scrnIndex, "Video3Zoom: 0x%08X.\n", Swov->Video->Video3Zoom);
    ViaDebug(Swov->scrnIndex, "Video3Minify: 0x%08X.\n", Swov->Video->Video3Minify);
    ViaDebug(Swov->scrnIndex, "Video3ColorSpace1: 0x%08X.\n", Swov->Video->Video3ColorSpace1);
    ViaDebug(Swov->scrnIndex, "Video3ColorSpace2: 0x%08X.\n", Swov->Video->Video3ColorSpace2);
    ViaDebug(Swov->scrnIndex, "Video3DisplayTemp: 0x%08X.\n", Swov->Video->Video3DisplayTemp);
    ViaDebug(Swov->scrnIndex, "Video3AlphaFifo: 0x%08X.\n", Swov->Video->Video3AlphaFifo);
    ViaDebug(Swov->scrnIndex, "Video3AlphaPreFifo: 0x%08X.\n", Swov->Video->Video3AlphaPreFifo);
    ViaDebug(Swov->scrnIndex, "Video3Address0: 0x%08X.\n", Swov->Video->Video3Address0);
    ViaDebug(Swov->scrnIndex, "Video3Address1: 0x%08X.\n", Swov->Video->Video3Address1);
    ViaDebug(Swov->scrnIndex, "Video3Address2: 0x%08X.\n", Swov->Video->Video3Address2);

    ViaDebug(Swov->scrnIndex, "HQV Map:\n");

    ViaDebug(Swov->scrnIndex, "Control: 0x%08X.\n", Swov->HQV->Control);
    ViaDebug(Swov->scrnIndex, "SrcAddressY: 0x%08X.\n", Swov->HQV->SrcAddressY);
    ViaDebug(Swov->scrnIndex, "SrcAddressU: 0x%08X.\n", Swov->HQV->SrcAddressU);
    ViaDebug(Swov->scrnIndex, "SrcAddressV: 0x%08X.\n", Swov->HQV->SrcAddressV);
    ViaDebug(Swov->scrnIndex, "SrcFetch: 0x%08X.\n", Swov->HQV->SrcFetch);
    ViaDebug(Swov->scrnIndex, "Filter: 0x%08X.\n", Swov->HQV->Filter);
    ViaDebug(Swov->scrnIndex, "Minify: 0x%08X.\n", Swov->HQV->Minify);
    ViaDebug(Swov->scrnIndex, "DstAddress0: 0x%08X.\n", Swov->HQV->DstAddress0);
    ViaDebug(Swov->scrnIndex, "DstAddress1: 0x%08X.\n", Swov->HQV->DstAddress1);
    ViaDebug(Swov->scrnIndex, "DstStride: 0x%08X.\n", Swov->HQV->DstStride);
    ViaDebug(Swov->scrnIndex, "SrcStride: 0x%08X.\n", Swov->HQV->SrcStride);
    ViaDebug(Swov->scrnIndex, "DstAddress2: 0x%08X.\n", Swov->HQV->DstAddress2);
}

/*
 * There was no via_drm.h versioning so we were forced to revert to using
 * defines directly...
 */
#ifdef DRM_VIA_DMA_BLIT
/*
 * Clean Xv code has the client align data for the driver and thus the hardware.
 * This means that a block needs to be dmaed over, pitch alignment has already
 * been taken care of.
 */
static Bool
ViaSwovCopyDMA(VIAPtr pVia, struct ViaMem *Mem, CARD8 *buf)
{
    CARD8 *aligned_buf, *new_buf = NULL;
    drm_via_dmablit_t blit;
    drm_via_blitsync_t *sync = &(blit.sync);
    int error, i;

    /* Everything VIA requires 16byte alignment -- idiots */
    if (((int) buf & 0x0F)) {
        ViaDebug(pVia->scrnIndex, "%s: Buffer Misalignment!\n", __FUNCTION__);
        new_buf = xnfalloc(Mem->Size + 0x0F);
        aligned_buf = (CARD8 *) (((int) new_buf + 0x0F) & ~0x0F);
        memcpy(aligned_buf, buf, Mem->Size);
    } else
        aligned_buf = buf;

    /* hrm */
    blit.num_lines = 1;
    blit.line_length = Mem->Size;
    blit.fb_addr = Mem->Base;
    blit.fb_stride = blit.line_length;
    blit.mem_addr = aligned_buf;
    blit.mem_stride = blit.line_length;
    /* not used anyway. */
#if defined(VIA_DRMH_MAJOR) || defined(VIA_DRM_DRIVER_MAJOR)
    blit.flags = 0;
#else
    blit.bounce_buffer = 0;
#endif
    blit.to_fb = 1;

    /* shedule this copy */
    for (i = 0; i < 10; i++) {
        error = drmCommandWriteRead(pVia->drmFD, DRM_VIA_DMA_BLIT, &blit,
                                    sizeof(drm_via_dmablit_t));
        if (error == -EAGAIN)
            usleep(1);
        else
            break;
    }

    if (error) {
        xf86DrvMsg(pVia->scrnIndex, X_ERROR, "%s: DMA copy sheduling failed: %d.\n",
                   __FUNCTION__, error);
        xf86DrvMsg(pVia->scrnIndex, X_ERROR, "%s: Copying %p to %08X\n",
                   __func__, blit.mem_addr, blit.fb_addr);
        if (new_buf)
            xfree(new_buf);

        return FALSE;
    }

    /* wait for this copy to complete */
    for (i = 0; i < 10; i++) {
        error = drmCommandWriteRead(pVia->drmFD, DRM_VIA_BLIT_SYNC, sync,
                                    sizeof(drm_via_blitsync_t));
        if (error == -EAGAIN)
            usleep(1);
        else
            break;
    }

    if (new_buf)
        xfree(new_buf);

    if (error) {
        xf86DrvMsg(pVia->scrnIndex, X_ERROR, "%s: DMA copy sync failed: %d - %d\n",
                   __FUNCTION__, error, -EAGAIN);
        xf86DrvMsg(pVia->scrnIndex, X_ERROR, "%s: Copying %p to %08X\n",
                   __func__, blit.mem_addr, blit.fb_addr);
        return FALSE;
    }
    return TRUE;
}
#endif /* DRM_VIA_DMA_BLIT */

/*
 *
 */
static Bool
ViaSwovCopy(VIAPtr pVia, struct ViaMem *Mem, unsigned char *buf)
{
#ifdef DRM_VIA_DMA_BLIT
    if (pVia->UseDMACopy) {
        if (!ViaSwovCopyDMA(pVia, Mem, buf)) {
            xf86DrvMsg(pVia->scrnIndex, X_ERROR, "%s: DMA Copy failed. "
                       "Disabling.\n", __FUNCTION__);
            pVia->UseDMACopy = FALSE;
            memcpy(pVia->FBBase + Mem->Base, buf, Mem->Size);
            return FALSE;
        }
    } else
#endif
        memcpy(pVia->FBBase + Mem->Base, buf, Mem->Size);

    return TRUE;
}

/*
 * Another bloody ugly mess.
 *
 *
 */
#include <math.h>

/*
 * This function uses quadratic mapping to adjust the midpoint of the scaling.
 */
static float
rangeEqualize(float inLow,float inHigh,float outLow,float outHigh,float outMid,
              float inValue)
{
    float inRange = inHigh - inLow;
    float outRange = outHigh - outLow;
    float normIn = ((inValue - inLow) / inRange)*2.-1.;
    float  delta = outMid - outRange*0.5 - outLow;

    return (inValue - inLow) * outRange / inRange + outLow + (1. - normIn*normIn)*delta;
}

static unsigned
vPackFloat(float val, float hiLimit, float loLimit, float mult, int shift,
           Bool doSign)
{
    unsigned packed,mask,sign;

    val = (val > hiLimit) ? hiLimit : val;
    val = (val < loLimit) ? loLimit : val;
    sign = (val < 0) ? 1:0;
    val = (sign) ? -val : val;
    packed = ((unsigned)(val*mult + 1.)) >> 1;
    mask = (1 << shift) - 1;

    return (((packed >= mask) ? mask : packed) | ((doSign) ? (sign << shift) : 0));
}

/*
 * HORRIBLE
 *
 * This function is a partial rewrite of the overlay.c file of the original VIA drivers,
 * which was extremely nasty and difficult to follow. Coefficient for new chipset models should
 * be added in the table above and, if needed, implemented in the model switch below. -- Thomas
 */
static void
viaCalculateVideoColor(VIAPtr pVia, CARD32 *col1, CARD32 *col2)
{
    struct ViaXvPort *Port = pVia->Swov->Port;

    float fA,fB1,fC1,fD,fB2,fC2,fB3,fC3;
    float fPI,fContrast,fSaturation,fHue,fBrightness;

    float ColorCoefficientVT3122Ax[5] = {1.164, 1.596, 0.54, 0.45, 2.2};
    float ColorCoefficientOther[5] = {1.1875, 1.625, 0.875, 0.375, 2.0};
    const float *mCoeff;

    unsigned long dwA,dwB1,dwC1,dwD,dwB2,dwC2,dwB3,dwC3,dwS;
    unsigned long dwD_Int,dwD_Dec;
    int intD;

    fPI = (float)(M_PI/180.);

    if ((pVia->Chipset == VT3122) && VT3122_REV_IS_AX(pVia->HostRev)) {
        fBrightness = rangeEqualize(0.,10000.,-128.,128.,-12.,(float) Port->brightness);
        fContrast = rangeEqualize(0.,20000.,0.,1.6645,1.1,(float) Port->contrast);
        fSaturation = rangeEqualize(0.,20000,0.,2.,1.15,(float) Port->saturation);
        mCoeff = ColorCoefficientVT3122Ax;
    } else {
        fBrightness = rangeEqualize(0.,10000.,-128.,128.,-16.,(float) Port->brightness);
        fContrast = rangeEqualize(0.,20000.,0.,1.6645,1.0,(float) Port->contrast);
        fSaturation = rangeEqualize(0.,20000,0.,2.,1.,(float) Port->saturation);
        mCoeff = ColorCoefficientOther;
    }
    fHue = (float)Port->hue;

    fA  = (float)(mCoeff[0]*fContrast);
    fB1 = (float)(-mCoeff[1]*fContrast*fSaturation*sin(fHue*fPI));
    fC1 = (float)(mCoeff[1]*fContrast*fSaturation*cos(fHue*fPI));
    fD  = (float)(mCoeff[0]*(fBrightness));
    fB2 = (float)((mCoeff[2]*sin(fHue*fPI)-
        mCoeff[3]*cos(fHue*fPI))*fContrast*fSaturation);
    fC2 = (float)(-(mCoeff[2]*cos(fHue*fPI)+
        mCoeff[3]*sin(fHue*fPI))*fContrast*fSaturation);
    fB3 = (float)(mCoeff[4]*fContrast*fSaturation*cos(fHue*fPI));
    fC3 = (float)(mCoeff[4]*fContrast*fSaturation*sin(fHue*fPI));

    if ((pVia->Chipset == VT3122) && VT3122_REV_IS_AX(pVia->HostRev)) {
        dwA = vPackFloat(fA,1.9375,-0.,32,5,0);
        dwB1 = vPackFloat(fB1,0.75,-0.75,8.,2,1);
        dwC1 = vPackFloat(fC1,2.875,1.,16.,5,0);

        if (fD>=127)
            fD=127;

        if (fD<=-128)
            fD=-128;

        if (fD>=0) {
            dwS = 0;
        }
        else {
            dwS = 1;
            fD = fD+128;
        }

        dwD = ((unsigned long)(fD*2+1))>>1;
        if (dwD>=0x7f) {
            dwD = 0x7f|(dwS<<7);
        } else {
            dwD = (dwD&0x7f)|(dwS<<7);
        }

        dwB2 = vPackFloat(fB2,0.,-0.875,16.,3,0);
        dwC2 = vPackFloat(fC2,0.,-1.875,16.,4,0);
        dwB3 = vPackFloat(fB3,3.75,0.,8.,4,0);
        dwC3 = vPackFloat(fC3,1.25,-1.25,8.,3,1);
        *col1 = (dwA<<24)|(dwB1<<18)|(dwC1<<9)|dwD;
        *col2 = (dwB2<<25)|(dwC2<<17)|(dwB3<<10)|(dwC3<<2);
    } else {
        dwA = vPackFloat(fA,1.9375,0.,32.,5,0);
        dwB1 = vPackFloat(fB1,2.125,-2.125,16.,5,1);
        dwC1 = vPackFloat(fC1,2.125,-2.125,16.,5,1);

        if (fD>=0) {
            intD=(int)fD;
            if (intD>127)
                intD = 127;
            dwD_Int = ((unsigned long)intD)&0xff;
            dwD = ((unsigned long)(fD*16+1))>>1;
            dwD_Dec= dwD&0x7;
        } else {
            intD=(int)fD;
            if (intD< -128)
                intD = -128;
            intD = intD+256;
            dwD_Int = ((unsigned long)intD)&0xff;
            fD = -fD;
            dwD = ((unsigned long)(fD*16+1))>>1;
            dwD_Dec= dwD&0x7;
        }

        dwB2 = vPackFloat(fB2,1.875,-1.875,16,4,1);
        dwC2 = vPackFloat(fC2,1.875,-1.875,16,4,1);
        dwB3 = vPackFloat(fB3,3.875,-3.875,16,5,1);
        dwC3 = vPackFloat(fC3,3.875,-3.875,16,5,1);
        *col1 = (dwA<<24)|(dwB1<<16)|(dwC1<<8)|dwD_Int;
        *col2 = (dwD_Dec<<29|dwB2<<24)|(dwC2<<16)|(dwB3<<8)|(dwC3);
    }
}

/*
 *
 */
static void
VIASetColorSpace(VIAPtr pVia)
{
    CARD32 col1,col2;

    viaCalculateVideoColor(pVia, &col1, &col2);

    pVia->Swov->Video->Video3ColorSpace1 = col1;
    pVia->Swov->Video->Video3ColorSpace2 = col2;
}

/*
 *
 */
static void
ViaSwovColorKey(ScrnInfoPtr pScrn, CARD32 Key)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaSwov *Swov = VIAPTR(pScrn)->Swov;

    if (pScrn->bitsPerPixel == 8)
        Key &= 0x000000FF;
    else
        Key &= 0x00FFFFFF;

    Swov->Video->Compose &= ~0xFF;

    if ((pVia->Chipset == VT3122) && VT3122_REV_IS_CX(pVia->HostRev)) {
        Swov->Video->Video3ColorKey = Key;
        Swov->Video->Compose |= SELECT_VIDEO3_IF_COLOR_KEY;
    } else {
        Swov->Video->ColorKey = Key;
        Swov->Video->Compose |= SELECT_VIDEO_IF_COLOR_KEY;
    }
#ifdef USE_SECONDARY
    Swov->Video->ColorKeySecond = Key;
    Swov->Video->Compose |= SECOND_DISPLAY_COLOR_KEY_ENABLE;
#endif
}

#ifdef UNUSED
/*
 *
 */
static void
ViaSwovChromaKey(struct ViaSwov *Swov, CARD32 chromaLow, CARD32 chromaHigh)
{
    Swov->Video->ChromaKeyLow &= ~CHROMA_KEY_LOW;
    Swov->Video->ChromaKeyLow = (chromaLow & CHROMA_KEY_LOW) | V_CHROMAKEY_V3;

    Swov->Video->ChromaKeyHigh &= ~CHROMA_KEY_HIGH;
    Swov->Video->ChromaKeyHigh = chromaHigh & CHROMA_KEY_HIGH;

    /* Temporarily solve the H/W Interpolation error when using Chroma Key */
    Swov->Video->Video3Minify &= 0xFFFFFFF8;

    /* Modified by Scottie[2001.12.5] for select video if (color key & chroma key) */
    Swov->Video->Compose |= SELECT_VIDEO_IF_CHROMA_KEY;
}
#endif /* UNUSED */

/*
 * Eyecandy: alpha blended overlay.
 *
 * Since plain constant alpha looks pretty crappy, the alpha plane is
 * used. The implementation on the standard unichrome is pretty crappy,
 * as the Address can only be altered by 16byte offsets (truncates down
 * otherwise). We therefor are unable to have a full sized alpha plane
 * which we write the colorkey clippings to when they change. The FrameAdjust
 * through adapting the Address only happens at 32pixel offsets. We need the
 * startAddress at a given position and adapt the clipping all the time.
 * Since full screen alpha plane requires more memory bandwidth than window
 * only, the latter is chosen.
 */
#define VIA_ALPHAADDRESS_ALIGN 16

/*
 * We only have 4 bits of alpha to work with. Makes handling pretty crappy.
 */
static void
ViaSwovAlphaFillRectangle(struct ViaXvPort *Port, CARD8 *Buffer, BoxRec Box)
{
    register int i, j;
    int stride = Port->AlphaWidth / 2;
    register CARD8 *tmp;

    if (Box.x1 & 1) {
        tmp = Buffer + Box.y1 * stride + Box.x1 / 2;
        for (i = Box.y1; i < Box.y2; i++) {
            *tmp |= 0xF0;
            tmp += stride;
        }
        Box.x1++;
    }

    if (Box.x2 & 1) {
        tmp = Buffer + Box.y1 * stride + (Box.x2 - 1) / 2;
        for (i = Box.y1; i < Box.y2; i++) {
            *tmp |= 0x0F;
            tmp += stride;
        }
        Box.x2--;
    }

    tmp = Buffer + Box.y1 * stride + Box.x1 / 2;
    j = (Box.x2 - Box.x1) / 2;
    for (i = Box.y1; i < Box.y2; i++) {
        memset(tmp, 0xFF, j);
        tmp += stride;
    }
}

/*
 *
 */
static void
ViaSwovAlphaFillBuffer(ScrnInfoPtr pScrn, CARD8 *Buffer, unsigned long Size)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaXvPort *Port = pVia->Swov->Port;
    BoxPtr Boxes = REGION_RECTS(&(Port->clip));
    int num = REGION_NUM_RECTS(&(Port->clip));
    int i;

    memset(Buffer, ((Port->Opacity) << 4) | (Port->Opacity), Size);

    for (i = 0; i < num; i++) {
        BoxRec Box = Boxes[i];

        /* Boxes are always in the visible frame, yet offset against 0,0 */
        Box.x1 -= Port->Win_X + pScrn->frameX0;
        Box.x2 -= Port->Win_X + pScrn->frameX0;
        Box.y1 -= Port->Win_Y + pScrn->frameY0;
        Box.y2 -= Port->Win_Y + pScrn->frameY0;

        if (Box.x1 < 0)
            Box.x1 = 0;
        /* clipping might still be outdated */
        else if (Box.x1 > Port->AlphaWidth)
            continue;

        if (Box.x2 > Port->AlphaWidth)
            Box.x2 = Port->AlphaWidth;
        else if (Box.x2 < 0)
            continue;

        if (Box.y1 < 0)
            Box.y1 = 0;
        else if (Box.y1 > Port->AlphaHeight)
            continue;

        if (Box.y2 > Port->AlphaHeight)
            Box.y2 = Port->AlphaHeight;
        else if (Box.y2 < 0)
            continue;

        ViaSwovAlphaFillRectangle(Port, Buffer, Box);
    }
}

/*
 * You can't count on setting the Window parameters one direction at a time.
 * You have to set them both from scratch. This rules out inclusion in
 * SwovVisibleH/V.
 *
 * We depend on Port->Win_*. Make sure that you call this after VisibleV/H.
 */
static void
ViaSwovAlphaPosition(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaSwov *Swov = pVia->Swov;
    struct ViaXvPort *Port = Swov->Port;
    CARD32 Temp;

    ViaSwovAlphaFillBuffer(pScrn, Port->AlphaLocal, Port->AlphaFB->Size);

    ViaSwovCopy(pVia, Port->AlphaFB, Port->AlphaLocal);

    Temp = Swov->Video->Video3AlphaFetch & 0xFFFFFC00;
    Swov->Video->Video3AlphaFetch = Temp | (ALIGN_TO(Port->Win_W, 32) >> 5);

    Swov->Video->AlphaWindowEnd = (((Port->Win_X + Port->Win_W - 1) << 16) |
                                   (Port->Win_Y + Port->Win_H - 1));
    Swov->Video->AlphaWindowStart = (Port->Win_X << 16) | Port->Win_Y;
}

/*
 * Do we need to reallocate the alpha plane?
 */
static Bool
ViaSwovAlphaSize(ScrnInfoPtr pScrn, short drw_w, short drw_h)
{
    struct ViaXvPort *Port = VIAPTR(pScrn)->Swov->Port;

    if (drw_w > pScrn->currentMode->HDisplay)
        drw_w = pScrn->currentMode->HDisplay;

    drw_w = ALIGN_TO(drw_w, 32);

    if (drw_h > pScrn->currentMode->VDisplay)
        drw_h = pScrn->currentMode->VDisplay;

    if ((Port->AlphaWidth != drw_w) ||
        (Port->AlphaHeight != drw_h)) {
        Port->AlphaWidth = drw_w;
        Port->AlphaHeight = drw_h;
        return TRUE;
    } else
        return FALSE;
}

/*
 *
 */
static Bool
ViaSwovAlphaSurface(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaSwov *Swov = pVia->Swov;
    struct ViaXvPort *Port = Swov->Port;

    if (Port->AlphaFB)
        ViaMemFree(pScrn, Port->AlphaFB);
    if (Port->AlphaLocalUnaligned)
        xfree(Port->AlphaLocalUnaligned);

    Port->AlphaFB = ViaMemAlloc(pScrn, (Port->AlphaWidth / 2) * Port->AlphaHeight,
                                 VIA_ALPHAADDRESS_ALIGN);
    if (!Port->AlphaFB) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Alpha Surface allocation failed.\n",
                   __func__);
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Disabling Alpha.\n", __func__);
        return FALSE;
    }

    Port->AlphaLocalUnaligned = xnfcalloc(Port->AlphaFB->Size + 0x0F, 1);
    Port->AlphaLocal = (CARD8 *) (((int) Port->AlphaLocalUnaligned + 0x0F) & ~0x0F);

    Swov->Video->AlphaStride = Port->AlphaWidth / 2;

    ViaSwovAlphaPosition(pScrn);

    Swov->Video->Video3AlphaFifo &= 0x0000FFFF;
    Swov->Video->Video3AlphaFifo |= 0x04070000; /* depth: 8, threshold 4 */

    Swov->Video->Video3AlphaPreFifo &= 0x8000007F;
    Swov->Video->Video3AlphaPreFifo |= 0x00040000; /* prethreshold 4 */

    Swov->Video->AlphaAddress = Port->AlphaFB->Base;
    return TRUE;
}

/*
 * Handle colorkey too
 */
static void
ViaSwovAlphaEnable(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaSwov *Swov = pVia->Swov;
    struct ViaXvPort *Port = Swov->Port;

    ViaSwovAlphaSize(pScrn, Port->Drw_W, Port->Drw_H);

    if (ViaSwovAlphaSurface(pScrn)) {
        /* enable alpha */
        Swov->Video->AlphaControl = 0x0000F001;

        /* disable colorkey */
        Swov->Video->Compose &= ~0xFF;
    } else {
        Port->Opacity = 0;
        Swov->Video->AlphaControl = 0x001FF000;
    }
}

/*
 *
 */
static void
ViaSwovAlphaDisable(ScrnInfoPtr pScrn)
{
    struct ViaSwov *Swov = VIAPTR(pScrn)->Swov;
    struct ViaXvPort *Port = Swov->Port;

    if (Port->AlphaFB)
        ViaMemFree(pScrn, Port->AlphaFB);
    Port->AlphaFB = NULL;

    if (Port->AlphaLocalUnaligned)
        xfree(Port->AlphaLocalUnaligned);
    Port->AlphaLocalUnaligned = NULL;
    Port->AlphaLocal = NULL;

    Swov->Video->Video3AlphaFifo &= 0x0000FFFF;
    Swov->Video->Video3AlphaPreFifo &= 0x8000007F;

    Swov->Video->AlphaWindowStart = 0;
    Swov->Video->AlphaWindowEnd = 0;
    Swov->Video->AlphaStride = 0;

    Swov->Video->Video3AlphaFetch &= 0xFFFFFC00;
    Swov->Video->AlphaAddress = 0;
    Swov->Video->AlphaStride = 0;

    Swov->Video->AlphaControl = 0x001FF000;

    /* enable colorkey */
    ViaSwovColorKey(pScrn, Port->ColorKey);

    if (Port->autoPaint)
        xf86XVFillKeyHelper(pScrn->pScreen, Port->ColorKey, &Port->clip);
}

/*
 *
 */
static void
ViaSwovAlphaSet(ScrnInfoPtr pScrn, INT32 value)
{
    struct ViaXvPort *Port = VIAPTR(pScrn)->Swov->Port;

    if (!Port->Opacity) {
        Port->Opacity = value;

        if (value)
            ViaSwovAlphaEnable(pScrn);
    } else {
        Port->Opacity = value;

        if (!value)
            ViaSwovAlphaDisable(pScrn);
        else /* alter level */
            ViaSwovAlphaPosition(pScrn);
    }
}

/*
 *
 */
static int
viaSetPortAttribute(ScrnInfoPtr pScrn, Atom attribute, INT32 value, pointer data)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    struct ViaXvPort *Port = (struct ViaXvPort *) data;

    if (attribute == xvColorKey) {
        Port->ColorKey = value;

        if (!Port->Opacity) {
            ViaSwovColorKey(pScrn, value);
            REGION_EMPTY(pScrn->pScreen, &Port->clip);
        }
        return Success;
    } else if (attribute == xvAutoPaint) {
        if (value)
            Port->autoPaint = TRUE;
        else
            Port->autoPaint = FALSE;
        return Success;
    } else if (attribute == xvBrightness) {
        Port->brightness = value;
        VIASetColorSpace(pVia);
        return Success;
    } else if (attribute == xvContrast) {
        Port->contrast = value;
        VIASetColorSpace(pVia);
        return Success;
    } else if (attribute == xvSaturation) {
        Port->saturation = value;
        VIASetColorSpace(pVia);
        return Success;
    } else if (attribute == xvHue) {
        Port->hue = value;
        VIASetColorSpace(pVia);
        return Success;
    } else if (attribute == xvOpacity) {
        if ((value < 0) || (value > 15))
            return BadValue;

        if (pVia->Swov->Active)
            ViaSwovAlphaSet(pScrn, value);
        else
            Port->Opacity = value;
        return Success;
    }

    xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unhandled attribute.\n", __func__);
    return BadMatch;
}

/*
 *
 */
static int
viaGetPortAttribute(ScrnInfoPtr pScrn, Atom attribute, INT32 *value, pointer data)
{
    struct ViaXvPort *Port = (struct ViaXvPort *) data;

    if (attribute == xvColorKey) {
        *value =(INT32) Port->ColorKey;
        return Success;
    } else if (attribute == xvAutoPaint) {
        if (Port->autoPaint)
            *value = 1;
        else
            *value = 0;
        return Success;
    } else if (attribute == xvBrightness) {
        *value = Port->brightness;
        return Success;
    } else if (attribute == xvContrast) {
        *value = Port->contrast;
        return Success;
    } else if (attribute == xvSaturation) {
        *value = Port->saturation;
        return Success;
    } else if (attribute == xvHue) {
        *value = Port->hue;
        return Success;
    } else if (attribute == xvOpacity) {
        *value = Port->Opacity;
        return Success;
    }

    xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unhandled attribute.\n", __func__);
    *value = 0;
    return BadMatch;
}

/*
 * BLOODY MESS!
 *
 * The whole bandwidth mess, both for primary and secondary, and for Xv and
 * whatnot, should get a big review and be made to dependant on each other.
 *
 *   Decide if the mode support video overlay. This depends on the bandwidth
 *   of the mode and the type of RAM available.
 */
#define SINGLE_7205_100 0.41
#define SINGLE_7205_133 0.70

#define VIDEO_BPP 2

/*
 *
 */
static Bool
ViaSwovBandwidth(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    DisplayModePtr mode = pScrn->currentMode;

    /* Small trick here. We keep the height in 16's of lines and width in 32's
       to avoid numeric overflow */

    if (pVia->Chipset == VT3122) {
	CARD32 bandwidth = (mode->HDisplay >> 4) * (mode->VDisplay >> 5) *
	    pScrn->bitsPerPixel * mode->VRefresh;

	switch (pVia->MemType) {
	case VIA_MEM_DDR200:
	    /* Basic limit for DDR200 is about this */
	    if(bandwidth > 1800000)
		return FALSE;
	    /* But we have constraints at higher than 800x600 */
	    if (mode->HDisplay > 800) {
		if(pScrn->bitsPerPixel != 8)
		    return FALSE;
		if(mode->VDisplay > 768)
		    return FALSE;
		if(mode->VRefresh > 60)
		    return FALSE;
	    }
	    return TRUE;
	case VIA_MEM_DDR266:
	    if(bandwidth > 7901250)
		return FALSE;
	    return TRUE;
	}
    } else {
	unsigned dClock;
	float mClock,memEfficiency,needBandWidth,totalBandWidth;

	switch(pVia->MemType) {
	case VIA_MEM_DDR200:
	    mClock = 100;
	    memEfficiency = (float)SINGLE_7205_100;
            break;
	case VIA_MEM_DDR266:
	    mClock = 133;
	    memEfficiency = (float)SINGLE_7205_133;
            break;
	case VIA_MEM_DDR333:
	    mClock = 166;
	    memEfficiency = (float)SINGLE_7205_133;
            break;
        case VIA_MEM_DDR400:
            mClock = 200;
	    memEfficiency = (float)SINGLE_7205_133;
            break;
	default:
	    /*Unknow DRAM Type*/
	    xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unhandled RAM type.\n", __func__);
	    mClock = 166;
	    memEfficiency = (float)SINGLE_7205_133;
            break;
	}

        dClock = (mode->HDisplay * mode->VDisplay * mode->VRefresh) / 680000;

        needBandWidth = (float)(((pScrn->bitsPerPixel >> 3) + VIDEO_BPP)*dClock);
        totalBandWidth = (float)(mClock*16.*memEfficiency);

        if (needBandWidth < totalBandWidth)
            return TRUE;
    }
    return FALSE;
}

/*
 * Only needs to be altered when the Front buffer size changes.
 */
static void
ViaHQVFetch(VIAPtr pVia)
{
    struct ViaSwov *Swov = pVia->Swov;
    struct ViaXvPort *Port = Swov->Port;
    CARD32 Fetch = Port->Width;
    CARD32 Height = Port->Height;

    switch (Port->FourCC) {
    case FOURCC_YV12:
    case FOURCC_I420:
        /* Fetch is ok */
        break;
    case FOURCC_YUY2:
    case FOURCC_RV15:
    case FOURCC_RV16:
        Fetch *= 2;
        break;
    case FOURCC_RV32:
        Fetch *= 4;
        break;
    default:
        xf86DrvMsg(Swov->scrnIndex, X_ERROR, "%s: Unhandled FourCC.\n", __func__);
    }

    if ((pVia->Chipset == VT3122) && VT3122_REV_IS_AX(pVia->HostRev))
        Fetch >>= 3;

    if ((Port->Deinterlace & VIA_DEINT_BOB) &&
        !(Port->Deinterlace & VIA_DEINT_INTERLEAVE))
        Height <<= 1;

    Swov->HQV->SrcFetch = ((Fetch - 1) << 16) | (Height - 1);
}

/*
 *
 */
static Bool
ViaAllocateFrontBuffer(ScrnInfoPtr pScrn, int FourCC, CARD16 Width, CARD16 Height)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaSwov *Swov = pVia->Swov;
    struct ViaXvPort *Port = Swov->Port;
    int Size;

    VIAFUNC(pScrn->scrnIndex);

    if (Port->Front[0])
        ViaMemFree(pScrn, Port->Front[0]);
    if (Port->Front[1])
        ViaMemFree(pScrn, Port->Front[1]);
    Port->Front[0] = NULL;
    Port->Front[1] = NULL;

    switch (FourCC) {
    case FOURCC_YV12:
    case FOURCC_I420:
        if (FourCC == FOURCC_I420) { /* YUV */
            Port->CbOffset = Width * Height;
            Port->CrOffset = Port->CbOffset + (Width >> 1) * (Height >> 1);
        } else { /* YVU */
            Port->CrOffset = Width * Height;
            Port->CbOffset = Port->CrOffset + (Width >> 1) * (Height >> 1);
        }

        Size = (Width * Height * 3) >> 1;
        break;
    case FOURCC_YUY2:
    case FOURCC_RV15:
    case FOURCC_RV16:
        Port->CrOffset = 0;
        Port->CbOffset = 0;

        Size = Width * Height * 2;
        break;
    case FOURCC_RV32:
        Port->CrOffset = 0;
        Port->CbOffset = 0;

        Size = Width * Height * 4;
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: unhandled FourCC: 0x%08X\n",
                   __func__, FourCC);
        return FALSE;
    }

    ViaDebug(pScrn->scrnIndex, "%s: %C%C%C%C %dx%d: %dB\n", __func__,
             FourCC & 0xFF, (FourCC >> 8) & 0xFF, (FourCC >> 16) & 0xFF,
             (FourCC >> 24) & 0xFF, Width, Height, Size);

    Port->FourCC = FourCC;
    Port->Width = Width;
    Port->Height = Height;

    /* Front buffer, where the HQV sources its video. */
    Port->Front[0] = ViaMemAlloc(pScrn, Size, 16);
    if (!Port->Front[0])
        return FALSE;

    /* This one may fail. */
    Port->Front[1] = ViaMemAlloc(pScrn, Size, 16);
    if (!Port->Front[1]) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Not enough memory available.\n",
                   __func__);
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Using only a single front buffer.\n",
                   __func__);

        /* single buffering -- address always remains the same so set it now */
        Swov->HQV->SrcAddressY = Port->Front[0]->Base;
        if (Port->CrOffset) {
            Swov->HQV->SrcAddressU = Port->Front[0]->Base + Port->CbOffset;
            Swov->HQV->SrcAddressV = Port->Front[0]->Base + Port->CrOffset;
        }
    }

    ViaHQVFetch(pVia);

    return TRUE;
}

/*
 * Only needs to be altered when the Back buffer size changes.
 */
static void
ViaSwovStride(struct ViaSwov *Swov, struct ViaXvPort *Port)
{
    switch (Port->FourCC) {
    case FOURCC_YV12:
    case FOURCC_I420:
        /* planar YUV */
        Swov->HQV->SrcStride = ((Port->Width >> 1) << 16) | Port->Width;
        Swov->HQV->DstStride = Port->Width << 1;
        Swov->V3Shadow->Stride = Port->Width << 1;
        break;
    case FOURCC_YUY2:
    case FOURCC_RV15:
    case FOURCC_RV16:
        Swov->HQV->SrcStride = Port->Width << 1;
        Swov->HQV->DstStride = Port->Width << 1;
        Swov->V3Shadow->Stride = Port->Width << 1;
        break;
    case FOURCC_RV32:
        Swov->HQV->SrcStride = Port->Width << 2;
        Swov->HQV->DstStride = Port->Width << 2;
        Swov->V3Shadow->Stride = Port->Width << 2;
        break;
    default:
        xf86DrvMsg(Swov->scrnIndex, X_ERROR, "%s: Unhandled FourCC.\n", __func__);
    }
}

/*
 *
 */
static Bool
ViaAllocateBackBuffers(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaSwov *Swov = pVia->Swov;
    struct ViaXvPort *Port = Swov->Port;
    int Size;

    VIAFUNC(pScrn->scrnIndex);

    if (Port->Back[0])
        ViaMemFree(pScrn, Port->Back[0]);
    if (Port->Back[1])
        ViaMemFree(pScrn, Port->Back[1]);
    if (Port->Back[2])
        ViaMemFree(pScrn, Port->Back[2]);
    Port->Back[0] = NULL;
    Port->Back[1] = NULL;
    Port->Back[2] = NULL;

    switch (Port->FourCC) {
    case FOURCC_YV12:
    case FOURCC_I420:
    case FOURCC_YUY2:
    case FOURCC_RV15:
    case FOURCC_RV16:
        Size = Port->Width * Port->Height * 2;
        break;
    case FOURCC_RV32:
        Size = Port->Width * Port->Height * 4;
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: unhandled FourCC: 0x%08X\n",
                   __func__, Port->FourCC);
        return FALSE;
    }

    ViaDebug(pScrn->scrnIndex, "%s: %dB\n", __func__, Size);

    /* Back Buffer: HQV destination and Video3 source. */
    Port->Back[0] = ViaMemAlloc(pScrn, Size, 16);
    Port->Back[1] = ViaMemAlloc(pScrn, Size, 16);
    if (!Port->Back[0] || !Port->Back[1])
        return FALSE;

    /* This one may fail too. */
    if ((pVia->Chipset != VT3122) || VT3122_REV_IS_CX(pVia->HostRev)) {
        /* use 3 Back buffers */
        Port->Back[2] = ViaMemAlloc(pScrn, Size, 16);
        if (!Port->Back[2]) {
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Not enough memory available.\n",
                       __func__);
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Using only 2 back buffers.\n",
                       __func__);
        }
    }

    if (Port->Back[2])
        Swov->HQV->Control |= HQV_TRIPLE_BUFF;
    else
        Swov->HQV->Control &= ~HQV_TRIPLE_BUFF;

    Swov->HQV->DstAddress0 = Port->Back[0]->Base;
    Swov->HQV->DstAddress1 = Port->Back[1]->Base;
    if (Port->Back[2])
        Swov->HQV->DstAddress2 = Port->Back[2]->Base;

    ViaSwovStride(Swov, Port);

    return TRUE;
}

/*
 *
 */
static void
ViaSwovSurfaceDestroy(ScrnInfoPtr pScrn)
{
    struct ViaSwov *Swov = VIAPTR(pScrn)->Swov;
    struct ViaXvPort *Port = Swov->Port;

    VIAFUNC(pScrn->scrnIndex);

    Port->FourCC = 0;
    Port->Width = 0;
    Port->Height = 0;

    if (Port->Front[0])
        ViaMemFree(pScrn, Port->Front[0]);
    if (Port->Front[1])
        ViaMemFree(pScrn, Port->Front[1]);
    if (Port->Back[0])
        ViaMemFree(pScrn, Port->Back[0]);
    if (Port->Back[1])
        ViaMemFree(pScrn, Port->Back[1]);
    if (Port->Back[2])
        ViaMemFree(pScrn, Port->Back[2]);
    if (Port->AlphaFB)
        ViaMemFree(pScrn, Port->AlphaFB);
    if (Port->AlphaLocalUnaligned)
        xfree(Port->AlphaLocalUnaligned);

    Port->Front[0] = NULL;
    Port->Front[1] = NULL;
    Port->Back[0] = NULL;
    Port->Back[1] = NULL;
    Port->Back[2] = NULL;
    Port->AlphaFB = NULL;
    Port->AlphaLocalUnaligned = NULL;
    Port->AlphaLocal = NULL;
}

/*
 * Prime candidate for an attribute system.
 */
static void
ViaSwovFIFO(VIAPtr pVia, struct ViaSwov *Swov)
{
    CARD8 depth, prethreshold, threshold;

    switch (pVia->Chipset) {
    case VT3122:
        if (VT3122_REV_IS_CX(pVia->HostRev)) {
            depth = 64;
            prethreshold = 56;
            threshold = 56;
            Swov->V3Shadow->Control |= V3_EXPIRE_NUM_F;
        } else { /* VT3122Ax */
            depth = 32;
            prethreshold = 24;
            threshold = 24;
            Swov->V3Shadow->Control |= V3_EXPIRE_NUM;
        }
        break;
    case VT7205:
        depth = 32;
        prethreshold = 29;
        threshold = 29;
        Swov->V3Shadow->Control |= V3_EXPIRE_NUM_7205;
        break;
    case VT3108:
        depth = 64;
        prethreshold = 60;
        threshold = 60;
        Swov->V3Shadow->Control |= V3_EXPIRE_NUM_7205 | V3_PREFETCH;
        break;
    default:
        depth = 1;
        prethreshold = 0;
        threshold = 0;
        break;
    }

    Swov->V3Shadow->Fifo = ((depth - 1) & 0xff) | ((threshold & 0xff) << 8);
    Swov->V3Shadow->PreFifo =  prethreshold & 0x7f;
}

/*
 *
 */
static void
ViaHQVFormat(struct ViaSwov *Swov)
{
    struct ViaXvPort *Port = Swov->Port;

    Swov->HQV->Control &= 0x0FFFFFFF;

    switch (Port->FourCC) {
    case FOURCC_YV12:
    case FOURCC_I420:
        Swov->HQV->Control |= HQV_YUV420;
        break;
    case FOURCC_YUY2:
        Swov->HQV->Control |= HQV_YUV422;
        break;
    case FOURCC_RV15:
        Swov->HQV->Control |= HQV_RGB15;
        break;
    case FOURCC_RV16:
        Swov->HQV->Control |= HQV_RGB16;
        break;
    case FOURCC_RV32:
        Swov->HQV->Control |= HQV_RGB32;
        break;
    default:
        xf86DrvMsg(Swov->scrnIndex, X_ERROR, "%s: Unhandled FourCC.\n", __func__);
    }
}

/*
 *
 */
static void
ViaHQVInit(struct ViaSwov *Swov)
{
    struct ViaXvPort *Port = Swov->Port;

    Swov->HQV->Control = HQV_SRC_SW;

    ViaHQVFormat(Swov);

    /* For DCT450 test-BOB INTERLEAVE */
    if ((Port->Deinterlace & VIA_DEINT_BOB)) {
        Swov->HQV->Control |= HQV_FIELD_2_FRAME | HQV_DEINTERLACE;
        if ((Port->Deinterlace & VIA_DEINT_INTERLEAVE))
            Swov->HQV->Control |= HQV_FRAME_2_FIELD;
    }

    Swov->HQV->Control |= HQV_ENABLE;
}

/*
 *
 */
static void
ViaVideo3Format(struct ViaSwov *Swov)
{
    struct ViaXvPort *Port = Swov->Port;

    Swov->V3Shadow->Control &= 0xFFFFFFF0;

    switch (Port->FourCC) {
    case FOURCC_YV12:
    case FOURCC_I420:
    case FOURCC_YUY2:
        Swov->V3Shadow->Control |= V3_YUV422;
        break;
    case FOURCC_RV15:
        Swov->V3Shadow->Control |= V3_RGB15;
        break;
    case FOURCC_RV16:
        Swov->V3Shadow->Control |= V3_RGB16;
        break;
    case FOURCC_RV32:
        Swov->V3Shadow->Control |= V3_RGB32;
        break;
    default:
        xf86DrvMsg(Swov->scrnIndex, X_ERROR, "%s: Unhandled FourCC.\n", __func__);
    }
}
/*
 *
 */
static void
ViaVideo3Init(struct ViaSwov *Swov)
{
    Swov->Video->Compose = ALWAYS_SELECT_VIDEO | COMPOSE_V3_TOP;

    Swov->V3Shadow->Control = V3_COLORSPACE_SIGN | V3_YUV422 | V3_SWAP_HW_HQV;

    ViaVideo3Format(Swov);
}

/*
 *
 */
static void
ViaHQVIdle(struct ViaSwov *Swov)
{
    int i;

    for (i = 0; i < 10; i++)
        if (Swov->HQV->Control & HQV_IDLE)
            break;
        else
            usleep(1);

    if (i > IDLELOOP_LEVEL) {
        ViaDebug(Swov->scrnIndex, "%s: %d loops.\n", __func__, i);
        if (!(Swov->HQV->Control & HQV_IDLE))
            xf86DrvMsg(Swov->scrnIndex, X_WARNING, "%s: HQV not idle.\n", __func__);
    }
}

/*
 *
 */
static void
ViaSwovFireWait(struct ViaSwov *Swov)
{
    int i;

    for (i = 0; i < 25; i++)
        if (!(Swov->Video->Compose & V3_COMMAND_FIRE))
            break;
        else
            usleep(1);

    if (i > IDLELOOP_LEVEL) {
        ViaDebug(Swov->scrnIndex, "%s: %d loops.\n", __func__, i);
        if ((Swov->Video->Compose & V3_COMMAND_FIRE))
            xf86DrvMsg(Swov->scrnIndex, X_WARNING,
                       "%s: Video3 engine still firing.\n", __func__);
    }
}

/*
 * Minify is done by the HQV entirely.
 * Zoom is done by V3.
 *
 */
static void
ViaSwovZoomH(struct ViaSwov *Swov, struct ViaXvPort *Port)
{
    unsigned long tmp;

    Swov->HQV->Minify &= 0xFFFF0000;
    Swov->HQV->Filter &= 0xFFFF0000;
    Swov->V3Shadow->Zoom &= 0x0000FFFF;
    Swov->V3Shadow->Minify &= 0xF8FFFFFD;

    if (Port->Src_W == Port->Drw_W) { /* No zoom */
        Swov->HQV->Filter |= HQV_H_HIPASS_F1_DEFAULT;
    } else if (Port->Src_W < Port->Drw_W) { /* Zoom in */

        tmp = Port->Src_W * 0x800 / Port->Drw_W;
        Swov->V3Shadow->Zoom |= ((tmp & 0x7ff) << 16) | V3_X_ZOOM_ENABLE;

        Swov->V3Shadow->Minify |= V3_X_INTERPOLY;
        Swov->HQV->Filter |= HQV_H_HIPASS_F1_DEFAULT;

    } else { /* Zoom out */
        tmp = Port->Drw_W * 0x800 * 0x400 / Port->Src_W;
        tmp = tmp / 0x400 + ((tmp & 0x3ff) ? 1 : 0);

        Swov->HQV->Minify |= (tmp & 0x7ff) | HQV_H_MINIFY_ENABLE;

        if (Port->Src_W > (4 * Port->Drw_W))
            Swov->HQV->Filter |= HQV_H_TAP8_12221;
        else
            Swov->HQV->Filter |= HQV_H_TAP4_121;
        Swov->HQV->Minify |= HQV_HDEBLOCK_FILTER;
    }
}

/*
 *
 */
static void
ViaSwovZoomV(struct ViaSwov *Swov, struct ViaXvPort *Port)
{
    unsigned long tmp;
    short Src_H = Port->Src_H;

    if ((Port->Deinterlace & VIA_DEINT_BOB) &&
        !(Port->Deinterlace & VIA_DEINT_INTERLEAVE))
        Src_H <<= 1;

    Swov->HQV->Minify &= 0x0000FFFF;
    Swov->HQV->Filter &= 0x0000FFFF;
    Swov->V3Shadow->Zoom &= 0xFFFF0000;
    Swov->V3Shadow->Minify &= 0xFFF8FFFA;

    /* Setup Y zoom factor */
    if (Src_H == Port->Drw_H) { /* No zoom */
        Swov->HQV->Filter |= HQV_V_TAP4_121;
    } else if (Src_H < Port->Drw_H) { /* Zoom in */
        ScrnInfoPtr pScrn = xf86Screens[Swov->scrnIndex];

        tmp = Src_H * 0x0400 / Port->Drw_H - 1;
        Swov->V3Shadow->Zoom |= ((tmp & 0x3ff) | V3_Y_ZOOM_ENABLE);

        if ((VIAPTR(pScrn)->Chipset == VT3122) && (pScrn->currentMode->HDisplay > 1024))
            /* Temporary fix for 2D bandwidth problem. 2002/08/01*/
            Swov->V3Shadow->Minify |= V3_YCBCR_INTERPOLY;
        else
            Swov->V3Shadow->Minify |= V3_Y_INTERPOLY | V3_YCBCR_INTERPOLY;

        Swov->HQV->Filter |= HQV_V_TAP4_121;
    } else { /* Zoom out */
        tmp = Port->Drw_H * 0x0800 * 0x400 / Src_H;
        tmp = tmp / 0x400 + ((tmp & 0x3ff) ? 1 : 0);

        Swov->HQV->Minify |= ((tmp & 0x7ff) << 16) | HQV_V_MINIFY_ENABLE;

        if (Src_H > (4 * Port->Drw_H))
            Swov->HQV->Filter |= HQV_V_TAP8_12221;
       else
            Swov->HQV->Filter |= HQV_V_TAP4_121;
        Swov->HQV->Minify |= HQV_VDEBLOCK_FILTER;
    }
}

/*
 * Returns FALSE when the overlay is not currently visible.
 */
static Bool
ViaSwovVisibleH(ScrnInfoPtr pScrn, struct ViaXvPort *Port)
{
    struct ViaSwov *Swov = VIAPTR(pScrn)->Swov;
    short limit;
    long temp;
    short Src_X, Src_W, Dst_X, Dst_W;

    /* Horizontal */
    if (pScrn->currentMode->HDisplay < 2047)
        limit = pScrn->currentMode->HDisplay;
    else
        limit = 2047;

    if (((Port->Drw_X + Port->Drw_W) < pScrn->frameX0) ||
        (Port->Drw_X > (limit + pScrn->frameX0)))
        return FALSE;

    Src_X = Port->Src_X;
    Dst_W = Port->Drw_W;

    if (Port->Drw_X < pScrn->frameX0) {
        if (Port->Src_W > Port->Drw_W) /* HQV scaling */
            Src_X += pScrn->frameX0 - Port->Drw_X;
        else { /* Video3 scaling */
            temp = (pScrn->frameX0 - Port->Drw_X) * Port->Src_W;
            temp += Port->Drw_W >> 1;
            Src_X += temp / Port->Drw_W;
        }

        Dst_W -= pScrn->frameX0 - Port->Drw_X;
        Dst_X = 0;
    } else
        Dst_X = Port->Drw_X - pScrn->frameX0;

    if ((Dst_X + Dst_W) > limit)
        Dst_W = limit - Dst_X;

    if (Dst_W != Port->Drw_W) {
        temp = Dst_W * Port->Src_W;
        temp += Port->Drw_W >> 1;
        Src_W = temp / Port->Drw_W;
    } else
        Src_W = Port->Src_W;

    /* set up window */
    Swov->V3Shadow->WindowEnd &= 0x0000FFFF;
    Swov->V3Shadow->WindowEnd |= (Dst_X + Dst_W - 1) << 16;

    Swov->V3Shadow->WindowStart &= 0x0000FFFF;
    Swov->V3Shadow->WindowStart |= Dst_X << 16;

    /* set up Fetch */
    if (Port->FourCC == FOURCC_RV32) {
        if (Dst_W >= Src_W)
            Swov->V3Shadow->Fetch = (ALIGN_TO(Src_W << 2, 16) >> 4) + 1;
        else
            Swov->V3Shadow->Fetch = (ALIGN_TO(Dst_W << 2, 16) >> 4) + 1;
    } else {
        if (Dst_W >= Src_W)
            Swov->V3Shadow->Fetch = (ALIGN_TO(Src_W << 1, 16) >> 4) + 1;
        else
            Swov->V3Shadow->Fetch = (ALIGN_TO(Dst_W << 1, 16) >> 4) + 1;
    }

    /* Fix planar mode problem. */
    if (Swov->V3Shadow->Fetch < 4)
        Swov->V3Shadow->Fetch = 4;

    Swov->V3Shadow->SrcWidth = Src_W - 1;

    /* for startAddress offset */
    Port->Offset_X = Src_X;

    /* for alpha */
    Port->Win_X = Dst_X;
    Port->Win_W = Dst_W;

    return TRUE;
}

/*
 *
 */
static Bool
ViaSwovVisibleV(ScrnInfoPtr pScrn, struct ViaXvPort *Port)
{
    struct ViaSwov *Swov = VIAPTR(pScrn)->Swov;
    short limit;
    long temp;
    short Src_Y, Dst_Y, Dst_H;

    if (pScrn->currentMode->VDisplay < 2047)
        limit = pScrn->currentMode->VDisplay;
    else
        limit = 2047;

    if (((Port->Drw_Y + Port->Drw_H) < pScrn->frameY0) ||
        (Port->Drw_Y > (limit + pScrn->frameY0)))
        return FALSE;

    Src_Y = Port->Src_Y;
    Dst_H = Port->Drw_H;

    if (Port->Drw_Y < pScrn->frameY0) {
        if (Port->Src_H > Port->Drw_H) /* HQV scaling */
            Src_Y += pScrn->frameY0 - Port->Drw_Y;
        else { /* Video3 Scaling */
            /* Round down. */
            temp = (pScrn->frameY0 - Port->Drw_Y) * Port->Src_H;
            Src_Y += temp / Port->Drw_H;
        }

        Dst_H -= pScrn->frameY0 - Port->Drw_Y;
        Dst_Y = 0;
    } else
        Dst_Y = Port->Drw_Y - pScrn->frameY0;

    if ((Dst_Y + Dst_H) > limit)
        Dst_H = limit - Dst_Y;

    /* set up window */
    Swov->V3Shadow->WindowEnd &= 0xFFFF0000;
    Swov->V3Shadow->WindowEnd |= Dst_Y + Dst_H - 1;

    Swov->V3Shadow->WindowStart &= 0xFFFF0000;
    Swov->V3Shadow->WindowStart |= Dst_Y;

    /* for startAddress offset */
    Port->Offset_Y = Src_Y;

    /* for alpha */
    Port->Win_Y = Dst_Y;
    Port->Win_H = Dst_H;

    return TRUE;
}

/*
 *
 */
static void
ViaSwovStartAddress(struct ViaSwov *Swov, struct ViaXvPort *Port)
{
    unsigned long Offset;

    if (!Port->Back[0])
        return; /* No buffer yet, will try again later! */

    Offset = Port->Offset_Y * Port->Width + Port->Offset_X;

    if (Port->FourCC == FOURCC_RV32)
        Offset <<= 2;
    else
        Offset <<= 1;

    /* Warning: hardware rounds down to the nearest 16bytes */

    Swov->V3Shadow->Address0 = Port->Back[0]->Base + Offset;
    Swov->V3Shadow->Address1 = Port->Back[1]->Base + Offset;
    if (Port->Back[2])
        Swov->V3Shadow->Address2 = Port->Back[2]->Base + Offset;
}

/*
 * Tracking visibility is not needed, as it is a remnant of the driverside
 * FrameAdjust hooking. But Xv needs quite a few changes with respect to
 * visibility, Xv_ON/PENDING/OFF and Exa integration. I'm keeping things
 * for now. -- Luc.
 */
static Bool
ViaSwovWindow(ScrnInfoPtr pScrn, short src_x, short src_y, short drw_x,
              short drw_y, short src_w, short src_h, short drw_w, short drw_h)
{
    struct ViaSwov *Swov = VIAPTR(pScrn)->Swov;
    struct ViaXvPort *Port = Swov->Port;
    Bool HorizontalTouched = FALSE, VerticalTouched = FALSE;
    Bool HVisible = TRUE, VVisible = TRUE;

    if ((Port->Src_W != src_w) || (Port->Drw_W != drw_w)) {
        HorizontalTouched = TRUE;
        Port->Src_W = src_w;
        Port->Drw_W = drw_w;
        ViaSwovZoomH(Swov, Port);
    }

    if (HorizontalTouched || (Port->Src_X != src_x) || (Port->Drw_X != drw_x)) {
        HorizontalTouched = TRUE;
        Port->Src_X = src_x;
        Port->Drw_X = drw_x;
        HVisible = ViaSwovVisibleH(pScrn, Port);
    }

    if ((Port->Src_H != src_h) || (Port->Drw_H != drw_h)) {
        VerticalTouched = TRUE;
        Port->Src_H = src_h;
        Port->Drw_H = drw_h;
        ViaSwovZoomV(Swov, Port);
    }

    if (VerticalTouched || (Port->Src_Y != src_y) || (Port->Drw_Y != drw_y)) {
        VerticalTouched = TRUE;
        Port->Src_Y = src_y;
        Port->Drw_Y = drw_y;
        VVisible = ViaSwovVisibleV(pScrn, Port);
    }

    if (HorizontalTouched || VerticalTouched ||
        !(Swov->V3Shadow->Control & V3_ENABLE)) { /* Also handle uncover */

        if (HVisible && VVisible) {
            Swov->V3Shadow->Control |= V3_ENABLE;

            if (HorizontalTouched || VerticalTouched)
                ViaSwovStartAddress(Swov, Port);
        } else
            Swov->V3Shadow->Control &= ~V3_ENABLE;

        return TRUE;
    }

    return FALSE;
}

/*
 * Newer X versions have this built in, but debian xfree86-4.3.0 doesn't.
 */
#ifndef REGION_EQUAL
static Bool
RegionsEqual(ScreenPtr pScreen, RegionPtr A, RegionPtr B)
{
    int *dataA, *dataB;
    int num;

    num = REGION_NUM_RECTS(A);
    if(num != REGION_NUM_RECTS(B))
        return FALSE;

    if((A->extents.x1 != B->extents.x1) ||
       (A->extents.x2 != B->extents.x2) ||
       (A->extents.y1 != B->extents.y1) ||
       (A->extents.y2 != B->extents.y2))
        return FALSE;

    dataA = (int*)REGION_RECTS(A);
    dataB = (int*)REGION_RECTS(B);

    while(num--) {
        if((dataA[0] != dataB[0]) || (dataA[1] != dataB[1]))
           return FALSE;
        dataA += 2;
        dataB += 2;
    }

    return TRUE;
}
#define REGION_EQUAL(_pScreen, _pReg1, _pReg2) RegionsEqual((_pScreen), (_pReg1), (_pReg2))
#endif /* !REGION_EQUAL */

/*
 *
 */
static void
ViaHQVFlipInitWait(struct ViaSwov *Swov)
{
    int i;

    for (i = 0; i < 15; i++)
        if ((Swov->HQV->Control & HQV_FLIP_STATUS))
            break;
        else
            usleep(1);

    if (i > IDLELOOP_LEVEL) {
        ViaDebug(Swov->scrnIndex, "%s: status wait: %d.\n", __func__, i);
        if (!(Swov->HQV->Control & HQV_FLIP_STATUS))
            xf86DrvMsg(Swov->scrnIndex, X_WARNING, "%s: Unable to initialise"
                       " the HQV. (Status still not set)\n", __func__);
    }
}

/*
 *
 */
static void
ViaHQVFlip(VIAPtr pVia, unsigned char* buf)
{
    struct ViaSwov *Swov = pVia->Swov;
    struct ViaXvPort *Port = Swov->Port;
    int i;

    for (i = 0; i < 15; i++)
        if (!(Swov->HQV->Control & HQV_SW_FLIP))
            break;
        else
            usleep(1);

    if (i > IDLELOOP_LEVEL) {
        ViaDebug(Swov->scrnIndex, "%s: flip wait: %d.\n", __func__, i);
        if ((Swov->HQV->Control & HQV_SW_FLIP))
            xf86DrvMsg(Swov->scrnIndex, X_WARNING,
                       "%s: HQV is still flipping.\n", __func__);
    }

    if (buf) { /* if NULL, then we need to flip the previous buffer again */
        if (!Port->Front[1]) /* single buffering */
            ViaSwovCopy(pVia, Port->Front[0], buf);
        else { /* We only need to set the address */
            Swov->HQV->SrcAddressY = Port->Front[0]->Base;
            if (Port->CrOffset) {
                Swov->HQV->SrcAddressU = Port->Front[0]->Base + Port->CbOffset;
                Swov->HQV->SrcAddressV = Port->Front[0]->Base + Port->CrOffset;
            }
        }
    }

    Swov->HQV->Control |= HQV_SW_FLIP | HQV_FLIP_STATUS;
}


/*
 * Copy the Video3 Shadow over to the actual Video3 Registers and fire off
 * through Compose.
 *
 * Since all these registers only get written here, (except Control with
 * V3_ENABLE and Compose for the FIRE), and none of these registers are
 * used by the engine to relay its state to us, the content of the shadow
 * always fully corresponds with the content of the real registers.
 *
 * Where V3 gets disabled seperately, Swov->Active will be set FALSE,
 * triggering a complete re-initialisation of the Shadow upon the next
 * PutImage.
 */
static void
ViaVideo3Fire(struct ViaSwov *Swov)
{
    /* Clear Firing for Source flip */
    CARD32 Compose = Swov->Video->Compose & 0x00FF00FF;

    Swov->Video->Video3Control = Swov->V3Shadow->Control;
    /* On VT3122 V3 can not display on secondary */
#ifdef USE_SECONDARY
    Swov->Video->Video3Control |= 0x80000000;
#endif

    Swov->Video->Video3Zoom = Swov->V3Shadow->Zoom;
    Swov->Video->Video3Minify = Swov->V3Shadow->Minify;

    Swov->Video->Video3AlphaFetch &= ~V3_FETCH_COUNT;
    Swov->Video->Video3AlphaFetch |= Swov->V3Shadow->Fetch << 20;

    if (!Swov->Active) {
        Swov->Video->Video3AlphaFifo &= ~V3_FIFO_MASK;
        Swov->Video->Video3AlphaFifo |= Swov->V3Shadow->Fifo;

        Swov->Video->Video3AlphaPreFifo &= ~V3_PREFIFO_MASK;
        Swov->Video->Video3AlphaPreFifo |= Swov->V3Shadow->PreFifo;
    }

    Swov->Video->Video3SrcWidth = Swov->V3Shadow->SrcWidth;
    Swov->Video->Video3Stride = Swov->V3Shadow->Stride;
    Swov->Video->Video3WindowStart = Swov->V3Shadow->WindowStart;
    Swov->Video->Video3WindowEnd = Swov->V3Shadow->WindowEnd;

    Swov->Video->Video3Address0 = Swov->V3Shadow->Address0;
    Swov->Video->Video3Address1 = Swov->V3Shadow->Address1;
    Swov->Video->Video3Address2 = Swov->V3Shadow->Address2;

    /* Chocks away! */
    Swov->Video->Compose = Compose | V3_COMMAND_FIRE;
}


/*
 *
 */
static int
viaPutImage(ScrnInfoPtr pScrn, short src_x, short src_y, short drw_x,
            short drw_y, short src_w, short src_h, short drw_w, short drw_h,
            int id, unsigned char* buf, short width, short height, Bool sync,
            RegionPtr clipBoxes, pointer data
#if GET_ABI_MAJOR(ABI_VIDEODRV_VERSION) >= 1
            , DrawablePtr pDraw
#endif
            )
{
    VIAPtr  pVia = VIAPTR(pScrn);
    struct ViaSwov *Swov = pVia->Swov;
    struct ViaXvPort *Port = (struct ViaXvPort *) data;
    Bool ReAlloc = FALSE, Video3Touched = FALSE, ClippingTouched = FALSE;

    if (!Swov->Active) { /* initial pass */
        ReAlloc = TRUE;
        Video3Touched = TRUE;

        /* If there is bandwidth issue, block the H/W overlay */
        if (!Swov->Allowed && !(Swov->Allowed = ViaSwovBandwidth(pScrn))) {
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Xv Overlay rejected due"
                       " to insufficient memory bandwidth.\n", __func__);
            return BadAlloc;
        }

        if (!ViaAllocateFrontBuffer(pScrn, id, width, height)) {
            ViaSwovSurfaceDestroy(pScrn);
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Front buffer allocation"
                       " failed. Giving up.\n", __func__);
            return BadAlloc;
        }

        if (Port->Front[1]) /* We're double buffering */
            ViaSwovCopy(pVia, Port->Front[0], buf);

        ViaHQVInit(Swov);
        ViaVideo3Init(Swov);

#ifdef UNUSED
        ViaSwovChromaKey(Swov, 0, 0);
#endif
        ViaSwovFIFO(pVia, Swov);

        /* track Frame position */
        Swov->frameX0 = pScrn->frameX0;
        Swov->frameY0 = pScrn->frameY0;

    } else if (Port->FourCC != id) { /* change to a different FOURCC */
        ReAlloc = TRUE;
        Video3Touched = TRUE;

        /* make sure engines are idle first */
        ViaHQVIdle(Swov);

        if (!ViaAllocateFrontBuffer(pScrn, id, width, height)) {
            ViaSwovSurfaceDestroy(pScrn);
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Front buffer allocation"
                       " failed. Giving up.\n", __func__);
            return BadAlloc;
        }

        ViaHQVFormat(Swov);
        ViaVideo3Format(Swov);

        if (Port->Front[1]) /* We're double buffering */
            ViaSwovCopy(pVia, Port->Front[0], buf);

    } else if ((Port->Width != width) || (Port->Height != height)) { /* resize */
        ReAlloc = TRUE;
        Video3Touched = TRUE;

        /* make sure engines are idle first */
        ViaHQVIdle(Swov);

        if (!ViaAllocateFrontBuffer(pScrn, id, width, height)) {
            ViaSwovSurfaceDestroy(pScrn);
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Front buffer allocation"
                       " failed. Giving up.\n", __func__);
            return BadAlloc;
        }

        if (Port->Front[1]) /* We're double buffering */
            ViaSwovCopy(pVia, Port->Front[0], buf);
    } else { /* normal case -- copy now, cut down on idle time */
        if (Port->Front[1]) { /* We're double buffering */
            struct ViaMem *Mem = Port->Front[0];
            Port->Front[0] = Port->Front[1];
            Port->Front[1] = Mem;
            ViaSwovCopy(pVia, Port->Front[0], buf);
        }
        ViaHQVIdle(Swov);
    }

    if (ViaSwovWindow(pScrn, src_x, src_y, drw_x, drw_y, src_w, src_h, drw_w, drw_h))
        Video3Touched = TRUE;

    if (!REGION_EQUAL(pScrn->pScreen, &Port->clip, clipBoxes)) {
        REGION_COPY(pScrn->pScreen, &Port->clip, clipBoxes);
        ClippingTouched = TRUE;
    }

    if (Port->Opacity) { /* alpha handling */
        if (!Port->AlphaFB)
            ViaSwovAlphaEnable(pScrn);
        else if (ViaSwovAlphaSize(pScrn, drw_w, drw_h)) {
            if (!ViaSwovAlphaSurface(pScrn)) {
                Port->Opacity = 0;
                Swov->Video->AlphaControl = 0x001FF000;
            }
        } else if (Video3Touched || ClippingTouched)
            ViaSwovAlphaPosition(pScrn);
    }

    /* Colorkey handling - Opacity may have been altered */
    if (!Port->Opacity && ClippingTouched) {
        if (Port->autoPaint)
            xf86XVFillKeyHelper(pScrn->pScreen, Port->ColorKey, clipBoxes);
        if (!Swov->Active)
            ViaSwovColorKey(pScrn, Port->ColorKey);
    }

    if (ReAlloc) {
        if (!ViaAllocateBackBuffers(pScrn)) {
            ViaSwovSurfaceDestroy(pScrn);
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Failed to allocate Back "
                       "buffers. Giving up.\n", __func__);
            return BadAlloc;
        }

        /* Adjust to new Back buffer offsets */
        ViaSwovStartAddress(Swov, Port);
    }

    /* Double init to avoid flashing old memory content */
    if (!Swov->Active && (pVia->Chipset == VT3122) &&
        VT3122_REV_IS_AX(pVia->HostRev)) {
        ViaHQVFlip(pVia, buf);

        ViaSwovFireWait(Swov);
        ViaVideo3Fire(Swov);

        ViaHQVFlipInitWait(Swov);
        ViaHQVFlip(pVia, NULL);
    } else {
        if (Video3Touched) {
            ViaSwovFireWait(Swov);
            ViaVideo3Fire(Swov);
        }
        ViaHQVFlip(pVia, buf);
    }

    Swov->Active = TRUE;
    return Success;
}

/*
 * This doesn't fix the case where the overlay window was not
 * in the visible frame when switching away. That one requires
 * some DIX poking.
 */
static int
ViaSwovReputImage(ScrnInfoPtr pScrn, short drw_x, short drw_y,
                  RegionPtr clipBoxes, pointer data
#if GET_ABI_MAJOR(ABI_VIDEODRV_VERSION) >= 1
                  , DrawablePtr pDraw
#endif
                  )
{
    struct ViaSwov *Swov = VIAPTR(pScrn)->Swov;
    struct ViaXvPort *Port = Swov->Port;
    Bool HVisible = TRUE, VVisible = TRUE;

    if (!Swov->Active) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Overlay not active.\n", __func__);
        return BadAccess;
    }

    /* redraw colorkey when needed -- does this ever get hit here? */
    if (!REGION_EQUAL(pScrn->pScreen, &Port->clip, clipBoxes)) {
        REGION_COPY(pScrn->pScreen, &Port->clip, clipBoxes);
        if (Port->autoPaint && !Port->Opacity)
            xf86XVFillKeyHelper(pScrn->pScreen, Port->ColorKey, clipBoxes);
    }

    ViaHQVIdle(Swov);

    /* FrameAdjust is usually in only a single direction */
    if ((pScrn->frameX0 != Swov->frameX0) || (drw_x != Port->Drw_X)) {
        Port->Drw_X = drw_x;
        HVisible = ViaSwovVisibleH(pScrn, Swov->Port);
    }

    if ((pScrn->frameY0 != Swov->frameY0) || (drw_y != Port->Drw_Y)) {
        Port->Drw_Y = drw_y;
        VVisible = ViaSwovVisibleV(pScrn, Swov->Port);
    }

    if (HVisible && VVisible) {
        ViaSwovStartAddress(Swov, Swov->Port);
        Swov->V3Shadow->Control |= V3_ENABLE;
    } else
        Swov->V3Shadow->Control &= ~V3_ENABLE;

    if (Port->Opacity)
        ViaSwovAlphaPosition(pScrn);

    ViaSwovFireWait(Swov);
    ViaVideo3Fire(Swov);
    ViaHQVFlip(VIAPTR(pScrn), NULL);

    Swov->frameX0 = pScrn->frameX0;
    Swov->frameY0 = pScrn->frameY0;

    return Success;
}

/*
 *
 */
static void
viaQueryBestSize(ScrnInfoPtr pScrn, Bool motion, short vid_w, short vid_h,
                 short drw_w, short drw_h, unsigned int *p_w, unsigned int *p_h,
                 pointer data)
{
    *p_w = drw_w;
    *p_h = drw_h;

    if (*p_w > 2048 )
        *p_w = 2048;
}

/*
 *
 */
static int
viaQueryImageAttributes(ScrnInfoPtr pScrn, int FourCC, CARD16 *Width,
                        CARD16 *Height, int *pitches, int *offsets)
{
    int Size;

    if (!Width || !Height)
        return 0;

    if (*Width > VIA_MAX_XVIMAGE_X)
        *Width = VIA_MAX_XVIMAGE_X;
    if (*Height > VIA_MAX_XVIMAGE_Y)
        *Height = VIA_MAX_XVIMAGE_Y;

    switch (FourCC) {
    case FOURCC_YV12:  /* YUV420 */
    case FOURCC_I420:
        *Width = ALIGN_TO(*Width, 32);
        *Height = ALIGN_TO(*Height, 2);

        Size = 3 * (*Width) * (*Height);
        Size /= 2;

        if (pitches) {
            pitches[0] = *Width;
            pitches[1] = *Width >> 1;
            pitches[2] = *Width >> 1;
        }

        if (offsets) {
            offsets[0] = 0;
            offsets[1] = (*Width) * (*Height);
            offsets[2] = offsets[1] + ((*Width >> 1) * (*Height >> 1));
        }

        break;
    case FOURCC_YUY2:  /* YUV422 */
    case FOURCC_RV15:  /* RGB555 */
    case FOURCC_RV16:  /* RGB565 */
        *Width = ALIGN_TO(*Width, 8);

        Size = (*Width << 1) * (*Height);

        if (pitches)
            pitches[0] = *Width << 1;

        if (offsets)
            offsets[0] = 0;

        break;
    case FOURCC_RV32:  /* RGB888 */
        *Width = ALIGN_TO(*Width, 4);

        Size = (*Width << 2) * (*Height);

        if (pitches)
            pitches[0] = *Width << 2;

        if (offsets)
            offsets[0] = 0;

        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unhandled FourCC: 0x%08X.\n",
                   __func__, FourCC);
        *Width = 0;
        *Height = 0;
        return 0;
    }

    return Size;
}

/*
 *
 */
static void
ViaHQVIdleWait(struct ViaSwov *Swov)
{
    int i;

    for (i = 0; i < 10; i++)
        if (!(Swov->HQV->Control & HQV_IDLE) ||
            (Swov->HQV->Control & HQV_FLIP_STATUS)) {
            Swov->HQV->Control |= HQV_FLIP_STATUS;
            usleep(1);
        } else
            break;

    if (i > IDLELOOP_LEVEL) {
        ViaDebug(Swov->scrnIndex, "%s: idle/status wait: %d.\n", __func__, i);
        if (!(Swov->HQV->Control & HQV_IDLE))
            xf86DrvMsg(Swov->scrnIndex, X_WARNING, "%s: HQV not idle.\n", __func__);
        if (Swov->HQV->Control & HQV_FLIP_STATUS)
            xf86DrvMsg(Swov->scrnIndex, X_WARNING, "%s: HQV flip status not"
                       " cleared.\n", __func__);
    }
}

/*
 *
 */
static void
ViaVideoReset(ScrnInfoPtr pScrn)
{
    struct ViaSwov *Swov = VIAPTR(pScrn)->Swov;

    VIAFUNC(Swov->scrnIndex);

    Swov->Video->Video1Control = 0;
    Swov->Video->Video3Control = 0;
    Swov->Video->ColorKey = 0x821;
    Swov->Video->ColorKeySecond = 0x00000000;
    Swov->Video->Video3ColorKey = 0x00000000;
    Swov->Video->AlphaControl = 0x001FF000;

    Swov->Video->Compose = V1_COMMAND_FIRE;
    Swov->Video->Compose = V3_COMMAND_FIRE;

    if (Swov->HQV->Control & HQV_ENABLE) {
        ViaHQVIdleWait(Swov);

        if (Swov->HQVDisablePatch) {
            vgaHWPtr hwp = VGAHWPTR(pScrn);

            ViaSeqMask(hwp, 0x2E, 0x00, 0x10);
            Swov->HQV->Control &= ~HQV_ENABLE;
            ViaSeqMask(hwp, 0x2E, 0x10, 0x10);
        } else
            Swov->HQV->Control &= ~HQV_ENABLE;
    }
}

/*
 *
 */
static void
ViaSwovStop(ScrnInfoPtr pScrn)
{
    struct ViaSwov *Swov = VIAPTR(pScrn)->Swov;
    vgaHWPtr hwp = VGAHWPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    ViaHQVIdleWait(Swov);

    if (Swov->HQVDisablePatch)
	ViaSeqMask(hwp, 0x2E, 0x00, 0x10);

    Swov->Video->Video3AlphaFifo = ALPHA_FIFO_THRESHOLD4 | ALPHA_FIFO_DEPTH8
        | 0x00002431; /* threshold 24, depth 32 */

    Swov->HQV->Control &= ~HQV_ENABLE;

    if (Swov->HQVDisablePatch)
	ViaSeqMask(hwp, 0x2E, 0x10, 0x10);

    /* disable alpha too */
    Swov->Video->AlphaControl = 0x001FF000;

    Swov->Active = FALSE;
}

/*
 * If we ever use Video1, remember that on VT3122Ax the FIFO needs to be
 * disabled before restoring VT.
 */
static void
ViaStopVideo(ScrnInfoPtr pScrn, pointer data, Bool exit)
{
    struct ViaSwov *Swov = VIAPTR(pScrn)->Swov;
    struct ViaXvPort *Port = (struct ViaXvPort *)data;

    VIAFUNC(pScrn->scrnIndex);

    if (!Port->Opacity)
        REGION_EMPTY(pScrn->pScreen, &Port->clip);

    /* Hide overlay */
    Swov->Video->Video3Control &= ~V3_ENABLE;
    Swov->V3Shadow->Control &= ~V3_ENABLE;

    /* Fire at next VBlank */
    Swov->Video->Compose |=
        V3_COMMAND_FIRE | V_COMMAND_LOAD_VBI | V3_COMMAND_LOAD_VBI;

    if (exit) {
        ViaDebug(pScrn->scrnIndex, "%s: Down she goes.\n", __func__);

	ViaSwovStop(pScrn);
	ViaSwovSurfaceDestroy(pScrn);

	Port->Drw_X = 0;
	Port->Drw_Y = 0;
	Port->Drw_W = 0;
	Port->Drw_H = 0;
    }
}

/*
 *
 */
static struct ViaSwov *
ViaSwovInit(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaSwov *Swov;

    VIAFUNC(pVia->scrnIndex);

    pVia->Swov = (struct ViaSwov *) xnfcalloc(1, sizeof(struct ViaSwov));

    Swov = pVia->Swov;

    Swov->scrnIndex = pVia->scrnIndex;

    Swov->Allowed = FALSE;
    Swov->Active = FALSE;

    /* Set up memory Map */
    Swov->Video = (struct ViaVideoRegs *) (pVia->MapBase + 0x200);
    Swov->HQV = (struct ViaHQVRegs *) (pVia->MapBase + 0x3D0);

    if (pVia->PrintSwovRegs)
        ViaSwovPrintRegs(Swov, __func__);

    /* old HWDiff stuff */
    switch(pVia->Chipset) {
    case VT3122:
        if (VT3122_REV_IS_AX(pVia->HostRev))
            Swov->HQVDisablePatch = FALSE;
        else
            Swov->HQVDisablePatch = TRUE;
        break;
    case VT7205:
    case VT3108:
        Swov->HQVDisablePatch = TRUE;
        break;
    case VT3118:
        Swov->HQVDisablePatch = FALSE;
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unhandled ChipSet.\n", __func__);
    }

    ViaVideoReset(pScrn);

    return Swov;
}

/*
 *
 */
static struct ViaXvPort *
ViaXvPortInit(ScreenPtr pScreen)
{
    struct ViaXvPort *Port = xnfcalloc(1, sizeof(struct ViaXvPort));

    Port->ColorKey = 0x821;
    Port->autoPaint = TRUE;
    Port->brightness = 5000.;
    Port->saturation = 10000;
    Port->contrast = 10000;

#ifdef X_USE_REGION_NULL
    REGION_NULL(pScreen, &Port->clip);
#else
    REGION_INIT(pScreen, &Port->clip, NullBox, 1);
#endif

    /* attribute? */
    Port->Deinterlace = VIA_DEINT_INTERLEAVE;

    return Port;
}

/*
 *
 */
static XF86VideoAdaptorPtr
ViaXvAdaptorInit(ScrnInfoPtr pScrn)
{
    XF86VideoAdaptorPtr Adaptor;

    VIAFUNC(pScrn->scrnIndex);

    Adaptor = xf86XVAllocateVideoAdaptorRec(pScrn);

    Adaptor->type = XvWindowMask | XvImageMask | XvInputMask;
    Adaptor->flags = VIDEO_OVERLAID_IMAGES | VIDEO_CLIP_TO_VIEWPORT;

    Adaptor->name = ViaXVAdaptorName;
    Adaptor->nEncodings = 1;
    Adaptor->pEncodings = DummyEncoding;
    Adaptor->nFormats = sizeof(FormatsG) / sizeof(FormatsG[0]);
    Adaptor->pFormats = FormatsG;

    Adaptor->nPorts = 1;
    Adaptor->pPortPrivates = (DevUnion *)xnfcalloc(1, sizeof(DevUnion));
    Adaptor->pPortPrivates->ptr = VIAPTR(pScrn)->Swov->Port;
    Adaptor->nAttributes = NUM_ATTRIBUTES_G;
    Adaptor->pAttributes = AttributesG;

    Adaptor->nImages = NUM_IMAGES_G;
    Adaptor->pImages = ImagesG;
    Adaptor->PutVideo = NULL;
    Adaptor->StopVideo = ViaStopVideo;
    Adaptor->QueryBestSize = viaQueryBestSize;
    Adaptor->GetPortAttribute = viaGetPortAttribute;
    Adaptor->SetPortAttribute = viaSetPortAttribute;
    Adaptor->PutImage = viaPutImage;
    Adaptor->ReputImage = ViaSwovReputImage;
    Adaptor->QueryImageAttributes = viaQueryImageAttributes;

    return Adaptor;
}

/*
 *
 */
void
ViaVideoInit(ScrnInfoPtr pScrn, ScreenPtr pScreen)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    struct ViaSwov *Swov;
    XF86VideoAdaptorPtr adaptor[1], *adaptors[1];
    int num_adaptors;

    VIAFUNC(pScrn->scrnIndex);

    num_adaptors = xf86XVListGenericAdaptors(pScrn, adaptors);

    xvBrightness = MAKE_ATOM("XV_BRIGHTNESS");
    xvContrast = MAKE_ATOM("XV_CONTRAST");
    xvColorKey = MAKE_ATOM("XV_COLORKEY");
    xvHue = MAKE_ATOM("XV_HUE");
    xvSaturation = MAKE_ATOM("XV_SATURATION");
    xvAutoPaint = MAKE_ATOM("XV_AUTOPAINT_COLORKEY");
    xvOpacity = MAKE_ATOM("XV_OPACITY");

    Swov = ViaSwovInit(pScrn);

    Swov->Port = ViaXvPortInit(pScreen);

    Swov->Adaptor = ViaXvAdaptorInit(pScrn);

    Swov->Adaptors =  xnfalloc((num_adaptors + 1) * sizeof(XF86VideoAdaptorPtr*));

    if (num_adaptors)
        memcpy(Swov->Adaptors, adaptors, num_adaptors * sizeof(XF86VideoAdaptorPtr));

    adaptor[0] = Swov->Adaptor;
    memcpy(Swov->Adaptors + num_adaptors, adaptor, sizeof(XF86VideoAdaptorPtr));

    xf86XVScreenInit(pScreen, Swov->Adaptors, num_adaptors + 1);
    VIASetColorSpace(pVia);
}

/*
 *
 */
void
ViaVideoDestroy(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaSwov *Swov = pVia->Swov;

    VIAFUNC(pScrn->scrnIndex);

    /* Make sure we're shut down first. */
    ViaStopVideo(pScrn, Swov->Port, TRUE);

    /* Now tear down the wallpaper */
    xfree(Swov->Adaptors);
    xfree(Swov->Adaptor->pPortPrivates);
    xf86XVFreeVideoAdaptorRec(Swov->Adaptor);
    xfree(Swov->Port);
    xfree(Swov);

    pVia->Swov = NULL;
}

#endif /* XvExtension */
