/*
 * Copyright 2005 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xf86.h"
#include "xf86_OSproc.h"

#include "xf86fbman.h"

#include "via_driver.h"
#include "via_memory.h"

#ifndef X_USE_LINEARFB

/*
 * Old Xfree86 4.3 support. This predates the fb manager so it requires a bit
 * more work. We have to make sure that we don't assign already used bits.
 *
 * We keep an array of ViaMem structs in pVia for this.
 */

/*
 *
 */
void
ViaMemInit(ScrnInfoPtr pScrn,  ScreenPtr pScreen)
{
    VIAPtr pVia = VIAPTR(pScrn);
    int i;

    VIAFUNC(pScrn->scrnIndex);

    ViaDebug(pScrn->scrnIndex, "FB from %p to %p is available for allocation.\n",
             pVia->FBFreeStart, pVia->FBFreeEnd);

    for (i = 0; i < VIA_MEM_SLOTS; i++)
        pVia->Mem[i] = NULL;
}

/*
 *
 */
static void
ViaMemFBSort(VIAPtr pVia)
{
    int i;
    Bool swapped = TRUE;

    VIAFUNC(pVia->scrnIndex);

    while (swapped) {
        swapped = FALSE;

        for (i = 1; i < VIA_MEM_SLOTS; i++) {
            if (pVia->Mem[i]) {
                if (!pVia->Mem[i - 1]) {
                    pVia->Mem[i - 1] = pVia->Mem[i];
                    pVia->Mem[i] = NULL;
                    swapped = TRUE;
                } else if (pVia->Mem[i]->Base < pVia->Mem[i - 1]->Base) {
                    struct ViaMem *tmp = pVia->Mem[i - 1];
                    pVia->Mem[i - 1] = pVia->Mem[i];
                    pVia->Mem[i] = tmp;
                    swapped = TRUE;
                }
            }
        }
    }
}

/*
 *
 */
static void
ViaMemFBFree(ScrnInfoPtr pScrn, struct ViaMem *Mem)
{
    VIAPtr pVia = VIAPTR(pScrn);
    int i;  

    VIAFUNC(pScrn->scrnIndex);

    for (i = 0; i < VIA_MEM_SLOTS; i++)
        if (pVia->Mem[i] == Mem) {
            /* actual free happens in ViaMemFree */
            pVia->Mem[i] = NULL;
            return;
        }
    xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unable to find memory block at"
               " %p.\n", __func__, Mem->Base);
}

/*
 * Pretty expansive, but the old X servers will soon
 * become extinct. Less wasteful than the old way of
 * dividing free FB by 4 and handing these chunks out.
 * 
 */
static struct ViaMem *
ViaMemFBAlloc(ScrnInfoPtr pScrn, unsigned long size, CARD8 alignment)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaMem *Mem;
    int i;

    VIAFUNC(pScrn->scrnIndex);

    /* general checks first */
    if ((pVia->FBFreeEnd - pVia->FBFreeStart) < (size + alignment)) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: not enough FB memory"
                   " available.\n", __func__);
        return NULL;
    }

    ViaMemFBSort(pVia);

    if (pVia->Mem[VIA_MEM_SLOTS - 1]) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: No free memory slots"
                   " available.\n", __func__);
        return NULL;
    }

    Mem = xnfcalloc(sizeof(struct ViaMem), 1);
    Mem->Size = size;
    Mem->Private = NULL;
    Mem->PrivateSize = 0;
    Mem->Free = ViaMemFBFree;

    if (alignment) {
        size += alignment;
        alignment -= 1;
    }

    /* do we have free reign? */
    if (!pVia->Mem[0]) {
        Mem->Base = pVia->FBFreeStart;
        Mem->Base = (Mem->Base + alignment) & ~alignment;

        pVia->Mem[0] = Mem;
        return Mem;
    }

    /* gap in front? */
    if ((pVia->Mem[0]->Base - pVia->FBFreeStart) >= size) {
        Mem->Base = pVia->FBFreeStart;
        Mem->Base = (Mem->Base + alignment) & ~alignment;

        pVia->Mem[VIA_MEM_SLOTS - 1] = Mem;
        return Mem;
    }

    /* gap in between? */
    for (i = 1; pVia->Mem[i]; i++) {
        if ((pVia->Mem[i]->Base - pVia->Mem[i - 1]->Base - pVia->Mem[i - 1]->Size) >= size) {
            Mem->Base = pVia->Mem[i - 1]->Base + pVia->Mem[i - 1]->Size;
            Mem->Base = (Mem->Base + alignment) & ~alignment;
            
            pVia->Mem[VIA_MEM_SLOTS - 1] = Mem;
            return Mem;
        }
    }

    i--;

    /* room to spare? */
    if ((pVia->FBFreeEnd - pVia->Mem[i]->Base - pVia->Mem[i]->Size) >= size) {
        Mem->Base = pVia->Mem[i]->Base + pVia->Mem[i]->Size;
        Mem->Base = (Mem->Base + alignment) & ~alignment;

        pVia->Mem[VIA_MEM_SLOTS - 1] = Mem;
        return Mem;
    }
    
    xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: not enough free FB memory"
               " available.\n", __func__);
    xfree(Mem);
    return NULL;
}

#else /* if X_USE_LINEARFB */

/*
 * Xfree86 4.4 and newer has FB manager.
 *
 */

/*
 *
 */
void
ViaMemInit(ScrnInfoPtr pScrn,  ScreenPtr pScreen)
{
    VIAPtr pVia = VIAPTR(pScrn);
    unsigned long offset, size;
    
    VIAFUNC(pScrn->scrnIndex);
    
    ViaDebug(pScrn->scrnIndex, "FB from %p to %p is available for allocation.\n",
             pVia->FBFreeStart, pVia->FBFreeEnd);

    /* Why doesn't the FB manager handle alignment itself? */
    offset = (pVia->FBFreeStart + pVia->Bpp - 1 ) / pVia->Bpp; 
    size = pVia->FBFreeEnd / pVia->Bpp - offset;
    
    if (size > 0)
        xf86InitFBManagerLinear(pScreen, offset, size);
}

/*
 *
 */
static void
ViaMemFBFree(ScrnInfoPtr pScrn, struct ViaMem *Mem)
{
    VIAFUNC(pScrn->scrnIndex);

    xf86FreeOffscreenLinear((FBLinearPtr) Mem->Private);

    /* sick bastards manually xfree this Private,
       normal people know better */
}

/*
 *
 */
static struct ViaMem *
ViaMemFBAlloc(ScrnInfoPtr pScrn, unsigned long size, CARD8 alignment)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaMem *Mem;
    FBLinearPtr linear;
    unsigned long tempsize;

    VIAFUNC(pScrn->scrnIndex);

    /* Make sure we don't truncate requested size */
    tempsize = (size + alignment + pVia->Bpp - 1) / pVia->Bpp;
    
    linear = xf86AllocateOffscreenLinear(pScrn->pScreen, tempsize, 32,
                                         NULL, NULL, NULL);
    if (!linear)
	return NULL;
    
    Mem = xnfcalloc(sizeof(struct ViaMem), 1);
    Mem->Base = linear->offset * pVia->Bpp;

    /* adjust for alignment */
    if (alignment) {
        alignment -= 1;
        Mem->Base = (Mem->Base + alignment) & ~alignment;
    }

    Mem->Size = size;

    Mem->Private = linear;
    Mem->PrivateSize = sizeof(FBLinear);

    Mem->Free = ViaMemFBFree;

    return Mem;
}

#endif /* X_USE_LINEARFB */

#ifdef XF86DRI
/*
 * DRM.
 *
 */

/* First some workarounds for the badly trackable drm changes.
 * Plainly using VIDEO as a macro is a pretty bad initial choice
 * though */
#ifndef VIDEO
#define VIDEO VIA_MEM_VIDEO
#endif

/*
 *
 */
static void
ViaMemDRMFree(ScrnInfoPtr pScrn, struct ViaMem *Mem)
{
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    if (drmCommandWrite(pVia->drmFD, DRM_VIA_FREEMEM, (drm_via_mem_t *) Mem->Private,
                        sizeof(drm_via_mem_t)) < 0)
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Failed to deallocate Memory.\n",
                   __func__);

    xfree((drm_via_mem_t *) Mem->Private);
}

/*
 * Trick: Allocate size + alignment, and then align Base.
 *
 */
static struct ViaMem *
ViaMemDRMAlloc(ScrnInfoPtr pScrn, unsigned long size, CARD8 alignment)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaMem *Mem;
    drm_via_mem_t * DrmMem;
    int ret;

    VIAFUNC(pScrn->scrnIndex);

    DrmMem = xnfcalloc(sizeof(drm_via_mem_t), 1);

    DrmMem->context = 1;
    DrmMem->size = size + alignment;
    DrmMem->type = VIDEO;

    ret = drmCommandWriteRead(pVia->drmFD, DRM_VIA_ALLOCMEM, DrmMem,
                              sizeof(drm_via_mem_t));

    if (ret || (DrmMem->size != (size + alignment))) {
        xfree(DrmMem);
        return NULL;
    }

    Mem = xnfcalloc(sizeof(struct ViaMem), 1);

    if (alignment) {
        alignment -= 1;
        Mem->Base = (DrmMem->offset + alignment) & ~alignment;
    } else
        Mem->Base = DrmMem->offset;
    Mem->Size = size;

    Mem->Private = DrmMem;
    Mem->PrivateSize = sizeof(drm_via_mem_t);

    Mem->Free = ViaMemDRMFree;

    return Mem;
}

#endif /* XF86DRI */


/*
 * General functions.
 *
 */

/*
 *
 */
struct ViaMem *
ViaMemAlloc(ScrnInfoPtr pScrn, unsigned long size, CARD8 alignment)
{
    struct ViaMem *Mem;

    VIAFUNC(pScrn->scrnIndex);

    if (!size) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: libv is an idiot"
                   " (IQ %ld).\n", __func__, size);
        return NULL;
    }

#ifdef XF86DRI
    if (VIAPTR(pScrn)->directRenderingEnabled)
        Mem = ViaMemDRMAlloc(pScrn, size, alignment);
    else
#endif /* XF86DRI */
        Mem = ViaMemFBAlloc(pScrn, size, alignment);

    if (!Mem)
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Allocation failed "
                   "(%ldbytes requested).\n", __func__, size);
    else
        ViaDebug(pScrn->scrnIndex, "Allocated %d bytes at %p.\n", Mem->Size,
                 Mem->Base);
    return Mem;
}

/*
 *
 */
void
ViaMemFree(ScrnInfoPtr pScrn, struct ViaMem *Mem)
{
    VIAFUNC(pScrn->scrnIndex);

    if (Mem) {
        ViaDebug(pScrn->scrnIndex, "Freeing %d bytes at %p.\n", Mem->Size, 
                 Mem->Base);
        
        if (Mem->Free)
            Mem->Free(pScrn, Mem);

        xfree(Mem);
    }
}
