/*
 *	"portscan" match extension for iptables
 *	Copyright © Jan Engelhardt <jengelh [at] medozas de>, 2006 - 2008
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License; either
 *	version 2 of the License, or any later version, as published by the
 *	Free Software Foundation.
 */
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>

#include <xtables.h>
#include <linux/netfilter/x_tables.h>
#include "xt_portscan.h"

static const struct option portscan_mt_opts[] = {
	{.name = "stealth", .has_arg = false, .val = 'x'},
	{.name = "synscan", .has_arg = false, .val = 's'},
	{.name = "cnscan",  .has_arg = false, .val = 'c'},
	{.name = "grscan",  .has_arg = false, .val = 'g'},
	{},
};

static void portscan_mt_help(void)
{
	printf(
		"portscan match options:\n"
		"(Combining them will make them match by OR-logic)\n"
		"  --stealth    Match TCP Stealth packets\n"
		"  --synscan    Match TCP SYN scans\n"
		"  --cnscan     Match TCP Connect scans\n"
		"  --grscan     Match Banner Grabbing scans\n");
}

static int portscan_mt_parse(int c, char **argv, int invert,
    unsigned int *flags, const void *entry, struct xt_entry_match **match)
{
	struct xt_portscan_mtinfo *info = (void *)((*match)->data);

	switch (c) {
	case 'c':
		info->match_cn = true;
		return true;
	case 'g':
		info->match_gr = true;
		return true;
	case 's':
		info->match_syn = true;
		return true;
	case 'x':
		info->match_stealth = true;
		return true;
	}
	return false;
}

static void portscan_mt_check(unsigned int flags)
{
}

static void portscan_mt_print(const void *ip,
    const struct xt_entry_match *match, int numeric)
{
	const struct xt_portscan_mtinfo *info = (const void *)(match->data);
	const char *s = "";

	printf("portscan ");
	if (info->match_stealth) {
		printf("STEALTH");
		s = ",";
	}
	if (info->match_syn) {
		printf("%sSYNSCAN", s);
		s = ",";
	}
	if (info->match_cn) {
		printf("%sCNSCAN", s);
		s = ",";
	}
	if (info->match_gr)
		printf("%sGRSCAN", s);
	printf(" ");
}

static void portscan_mt_save(const void *ip, const struct xt_entry_match *match)
{
	const struct xt_portscan_mtinfo *info = (const void *)(match->data);

	if (info->match_stealth)
		printf("--stealth ");
	if (info->match_syn)
		printf("--synscan ");
	if (info->match_cn)
		printf("--cnscan ");
	if (info->match_gr)
		printf("--grscan ");
}

static struct xtables_match portscan_mt_reg = {
	.version       = XTABLES_VERSION,
	.name          = "portscan",
	.revision      = 0,
	.family        = AF_INET,
	.size          = XT_ALIGN(sizeof(struct xt_portscan_mtinfo)),
	.userspacesize = XT_ALIGN(sizeof(struct xt_portscan_mtinfo)),
	.help          = portscan_mt_help,
	.parse         = portscan_mt_parse,
	.final_check   = portscan_mt_check,
	.print         = portscan_mt_print,
	.save          = portscan_mt_save,
	.extra_opts    = portscan_mt_opts,
};

void _init(void);
void _init(void)
{
	xtables_register_match(&portscan_mt_reg);
}
