/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <dirent.h>
#include <time.h>
#include <limits.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <signal.h>



#include <X11/Intrinsic.h>	/* Intrinsics Definitions */
#include <X11/StringDefs.h>	/* Standard Name-String definitions */
#include <X11/Shell.h>     	/* Shell Definitions */

#include <X11/Xaw/Command.h>	/* Athena Command Widget */
#include <X11/Xaw/Form.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Repeater.h>

#include "types.h"
#include "xwave.h"
#include "misc.h"
#include "button.h"
#include "xwave_widget.h"
#include "sample_settings.h"
#include "audio.h"
#include "audio_file.h"
#include "filebrowse.h"
#include "graphics.h"
#include "fileop.h"
#include "record_dialog.h"

extern Main_Data *MD;
extern AppResources app_resources;

static void cancel_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void recpop_down_CB(Widget w, XtPointer client_data, 
			   XtPointer call_data);
static void rec_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void props_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void fname_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void sample_return(Sample_Return *sr);
static void props_label(byte res,byte mode,int freq);
static void fname_label(char *fname,char *type,char *comp);
static void getini_return(Sample_Return *sr);
static void record_popup (Widget w);
static void popup_action(Widget w, XtPointer client_data, 
			 XtPointer call_data);
static void upopup_action(Widget w, XtPointer client_data, 
			  XtPointer call_data);
static void stoprecord_action(Widget w, XtPointer client_data, 
			      XtPointer call_data);
static void rec_it(Widget w, Main_Data *md);
static int  file_return(char *fname,int type,int comp);
static void inuse_return();
static void exists_return();
static Boolean update_level(XtPointer);
static Boolean update_record(XtPointer);


static Widget popup,u_popup,plabel,flabel,tlabel,label5;
static byte res,mode;
static int freq,count=0;
static char *fname=NULL;
static long start_clock;
static int pid=-1;
static GC gcl,gcr;
static Display *dpy;
static Window winl,winr;
static XtWorkProcId p_id;

/* Kouhia: stuff needed for the slowly varying recording level meter */
#define SLOWMETERFREQ 32
int slowmeter_l,slowmeter_r,slowmeter_n;
int slowmeterold_l,slowmeterold_r;


static void sample_return(Sample_Return *sr)
{
    res=sr->res;
    mode=sr->channels;
    freq=sr->freq;
    XtFree((char*)sr);
    props_label(res,mode,freq);
}

static void getini_return(Sample_Return *sr)
{
    
    res=sr->res;
    mode=sr->channels;
    freq=sr->freq;
    XtFree((char*)sr);
    if ((fname=malloc(strlen(app_resources.tdir)+
		      strlen(app_resources.default_recname)+
		      strlen(app_resources.default_ext)+10))==NULL) {
	warn_popup(MD->mw->form,app_resources.err_mem);
	return;
    }
    
    sprintf(fname,"%s/%s_%i%s",app_resources.tdir,
	    app_resources.default_recname,count,app_resources.default_ext+1);
    
    fname_label(fname,ft2string(app_resources.default_ftype),
		comp2string(app_resources.default_comp));
    
    popup_centered(popup);
}


void props_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
    sample_dialog(sample_return,NULL);
}

void fname_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
    file_dialog(file_return,app_resources.default_ext,CREATE);
}

int file_return(char *name,int type,int comp)
{
    XtFree(fname);
    fname=get_abs_name(name);
    if (fname==NULL) fname=XtNewString(name);
    fname_label(name,ft2string(type),comp2string(comp));
    
    return(0);
}

static void inuse_return()
{
    close_file(MD);
    XtPopdown(popup);
    rec_it(XtParent(popup),MD);
}

static void exists_return()
{
    XtPopdown(popup);
    rec_it(XtParent(popup),MD);
}

void rec_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget popup=(Widget) client_data;
    if (MD->no>0)
      if (strcmp(MD->wd->name,fname)==0) {
	  bool_popup(inuse_return,NULL,NULL,NULL,
		     NULL,NULL,NULL,NULL,
		     XtParent(popup),app_resources.err_overwrite);
	  return;
      }
    if (file_exists(fname)) {
	bool_popup(exists_return,NULL,NULL,NULL,
		   NULL,NULL,NULL,NULL,		   
		   XtParent(popup),app_resources.err_overwrite);
	return;
    }

    XtPopdown(popup);
    rec_it(XtParent(popup),MD);
}


void cancel_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget popup=(Widget) client_data;
    XtPopdown(popup);
}

void create_record_dialog (Widget w)
{
    Widget form,button,label,graphr,graphl;
    XGCValues xgcv;
    
    popup=XtVaCreatePopupShell("rd_shell",transientShellWidgetClass,w,NULL);
    XtAddCallback(popup,XtNpopupCallback,popup_action,(XtPointer) NULL);
    
    form  = MW ("rd_main_form",formWidgetClass,popup,
		XtNresizable,TRUE,
		XtNborderWidth,0,
		NULL);
    
    label = MW ("rd_level_label",labelWidgetClass,form,
		XtNborderWidth,0,
		XtNbottom, XtChainTop,XtNtop, XtChainTop,
		XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    graphl = MW ("rd_ch1_gc",simpleWidgetClass,form,
		 XtNfromVert,label,NULL);
    
    graphr = MW ("rd_ch2_gc",simpleWidgetClass,form,
		 XtNfromVert,graphl,NULL);
    
    label = MW ("rd_sep1_label",labelWidgetClass,form,
		XtNfromVert,graphr,XtNlabel,"",NULL);
    
    label = MW ("rd_set_label",labelWidgetClass,form,
		XtNfromVert,label,XtNborderWidth,0,
		XtNbottom, XtChainTop,XtNtop, XtChainTop,
		XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    plabel = MW ("rd_props_label",labelWidgetClass,form,
		 XtNfromVert,label,XtNborderWidth,0,
		 XtNbottom, XtChainTop,XtNtop, XtChainTop,
		 XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    flabel = MW ("rd_fname_label",labelWidgetClass,form,
		 XtNfromVert,plabel,XtNborderWidth,0,
		 XtNbottom, XtChainTop,XtNtop, XtChainTop,
		 XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    tlabel = MW ("rd_type_label",labelWidgetClass,form,
		 XtNfromVert,flabel,XtNborderWidth,0,
		 XtNbottom, XtChainTop,XtNtop, XtChainTop,
		 XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    button = MW ("rd_props_btn",commandWidgetClass,form,
		 XtNfromVert,tlabel,
		 XtNtop,XawChainTop,XtNbottom,XawChainTop,
		 XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    XtAddCallback (button, XtNcallback, props_CB,(XtPointer) NULL);
    
    button = MW ("rd_fname_btn",commandWidgetClass,form,
		 XtNfromVert,tlabel,XtNfromHoriz,button,
		 XtNtop,XawChainTop,XtNbottom,XawChainTop,
		 XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    XtAddCallback (button, XtNcallback, fname_CB,(XtPointer) NULL);
    
    label = MW ("rd_sep2_label",labelWidgetClass,form,
		XtNfromVert,button,XtNlabel,"",NULL);
    
    button = MW ("rd_ok_btn",commandWidgetClass,form,
		 XtNfromVert,label,
		 XtNtop,XawChainTop,XtNbottom,XawChainTop,
		 XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    XtAddCallback (button, XtNcallback, rec_CB, (XtPointer) popup);
    
    button = MW ("rd_ca_btn",commandWidgetClass,form,
		 XtNfromVert,label,XtNfromHoriz,button,
		 XtNtop,XawChainTop,XtNbottom,XawChainTop,
		 XtNleft,XawChainRight,XtNright,XawChainRight,NULL);
    
    XtAddCallback (button, XtNcallback, cancel_CB, (XtPointer) popup);
    
    label = MW ("rd_sep3_label",labelWidgetClass,form,
		XtNfromVert,button,XtNlabel,"",NULL);
    
    XtRealizeWidget(popup);
    
    XtAddCallback (popup, XtNpopdownCallback, recpop_down_CB,(XtPointer) NULL);
    
    xgcv.foreground=ConvertColor(graphl,app_resources.wavemainfgl);
    gcl=XtGetGC(graphl,GCForeground, &xgcv);
    winl=XtWindow(graphl);
    xgcv.foreground=ConvertColor(graphr,app_resources.wavemainfgr);
    gcr=XtGetGC(graphr,GCForeground, &xgcv);
    winr=XtWindow(graphr);
    dpy=XtDisplay(graphl);
}

void recpop_down_CB(Widget w, XtPointer client_data,XtPointer call_data)
{
    XtRemoveWorkProc(p_id);
    /* close the audio device */
    get_samples(NULL,AU_CLOSE);
}

void props_label(byte res,byte mode,int freq)
{
    char settings[]="Stereo, 16bit, 44100hz, 250kb/s";
    
    if (mode==MONO) strcpy(settings,"Mono, ");
    else strcpy(settings,"Stereo, ");
    if (res==8) strcat(settings,"8bit, ");
    else strcat(settings,"16bit, ");
    strcat(settings,itoa(freq));
    strcat(settings,"hz, ");
    strcat(settings,itoa(freq*mode*(res/8)/1024));
    strcat(settings,"kb/s");
    XtVaSetValues(plabel,XtNlabel,settings,NULL);
}

void fname_label(char *name,char *type,char *comp)
{
    static char ft[20];
    
    short_fname(flabel,fname);
    sprintf(ft,"%s (%s)",type,comp);
    XtVaSetValues(tlabel,XtNlabel,ft,NULL);
}


void record_dialog(Main_Data *md)
{
    if (md->no!=0) {
	if (md->wd->length==0) {
	    res=md->wd->res;
	    mode=md->wd->channels;
	    freq=md->wd->freq;
	    fname=XtNewString(md->wd->name);
	    fname_label(fname,
			ft2string(md->wd->type),comp2string(md->wd->comp));
	    popup_centered(popup);
	    return;
	}
    }
    sample_dialog(getini_return,NULL);
}

void popup_action(Widget w, XtPointer client_data, XtPointer call_data)
{
    int buf_size;
    
    props_label(res,mode,freq);
    
    /* open audio device, get buf_size */
    buf_size=get_samples(NULL,AU_OPEN);

    slowmeter_l = 0;
    slowmeter_r = 0;
    slowmeterold_l = 0;
    slowmeterold_r = 0;
    slowmeter_n = 0;
    p_id=XtAppAddWorkProc(XtWidgetToApplicationContext(w),update_level,
			  NULL);
    return;
}

Boolean update_level(XtPointer client_data)
{
    
    int i,count;
    short val_l,val_r;
    short *buf16;
    static unsigned short bigl,bigr;
    
    XClearArea(dpy, winl, 0, 0, bigl, 7,FALSE);
    XClearArea(dpy, winr, 0, 0, bigr, 7,FALSE);
    /* MD->mg->fbuf is really allocated, don't care */
    buf16=(short *) MD->mg->fbuf;
    count=get_samples((unsigned char*)buf16,AU_READ);
    bigl=bigr=0;

    for (i=0;i<count/2;) {
	val_l=buf16[i++];
	val_r=buf16[i++];
	if (abs(val_l)>bigl) bigl=abs(val_l);
	if (abs(val_r)>bigr) bigr=abs(val_r);
	/* val XOR 0x80 to get char value 
	 val_l=buf[i++]^0x80;
	 val_r=buf[i++]^0x80;
	 */
    }
    
    bigl/=256;
    bigr/=256;

    slowmeter_n++;
    if (slowmeter_n < SLOWMETERFREQ) {
      if (slowmeter_l < bigl) slowmeter_l = bigl;
      if (slowmeter_r < bigr) slowmeter_r = bigr;
    } else {
      XClearArea(dpy, winl, 0, 8, slowmeterold_l, 7,FALSE);
      XClearArea(dpy, winr, 0, 8, slowmeterold_r, 7,FALSE);
      XFillRectangle(dpy, winl, gcl, 0, 8, slowmeter_l, 7);
      XFillRectangle(dpy, winr, gcr, 0, 8, slowmeter_r, 7);
      slowmeterold_l = slowmeter_l;
      slowmeterold_r = slowmeter_r;
      slowmeter_l = 0;
      slowmeter_r = 0;
      slowmeter_n = 0;
    }
    XFillRectangle(dpy, winl, gcl, 0, 0, bigl, 7);
    XFillRectangle(dpy, winr, gcr, 0, 0, bigr, 7);
    return(False);
}

void rec_it(Widget w, Main_Data *md)
{
    void stop_record();
    void abort_record();
    pid= fork();
    if (!pid) {
	XtDestroyApplicationContext(XtWidgetToApplicationContext(w));
	rec_file(res,mode,freq,fname);
	exit(0);
    } else if (pid<=0) return;
    
    signal(SIGUSR1,stop_record);
    signal(SIGUSR2,abort_record);
    
    record_popup(XtParent(popup));
    return;
}

void record_popup (Widget w)
{
    Widget popup,form,label1,label2,label3,label4,button;
    Dimension width,width1,height;
    char name[35];
    
    popup=XtVaCreatePopupShell("ru_shell",transientShellWidgetClass,w,NULL);
    u_popup=None;
    XtAddCallback(popup,XtNpopupCallback,upopup_action,(XtPointer) popup);
    
    form  = MW ("ru_main_form",formWidgetClass,popup,NULL);
    
    label1 = MW ("ru_fname_label",labelWidgetClass,form,
		 XtNborderWidth,0,
		 XtNlabel,fname,
		 NULL);
    
    XtVaGetValues(label1,XtNwidth, &width, NULL);
    
    if (res==8) strcpy(name,"Res.: 8bit");
    else strcpy(name,"Res.: 16bit");
    label2 = MW ("ru_res_label",labelWidgetClass,form,
		 XtNfromVert,label1,
		 XtNborderWidth,0,
		 XtNlabel,name,
		 NULL);
    XtVaGetValues(label2,XtNwidth, &width1, NULL);
    if (width1>width) width=width1;
    
    if (mode==MONO) strcpy(name,"Mode: Mono");
    else strcpy(name,"Mode: Stereo");
    label3 = MW ("ru_chn_label",labelWidgetClass,form,
		 XtNfromVert,label2,
		 XtNborderWidth,0,
		 XtNlabel,name,
		 NULL);
    XtVaGetValues(label3,XtNwidth, &width1, NULL);
    if (width1>width) width=width1;
    
    strcpy(name,"Freq.: ");
    strcat(name,itoa(freq));
    strcat(name," hz");
    label4 = MW ("ru_freq_label",labelWidgetClass,form,
		 XtNfromVert,label3,
		 XtNborderWidth,0,
		 XtNlabel,name,
		 NULL);
    XtVaGetValues(label4,XtNwidth, &width1, NULL);
    if (width1>width) width=width1;
    
    strcpy(name,"      0,00 kbytes (0,00s)      ");
    label5 = MW ("ru_counter_label",labelWidgetClass,form,
		 XtNfromVert,label4,
		 XtNborderWidth,0,
		 XtNlabel,name,
		 NULL);
    XtVaGetValues(label5,XtNwidth, &width1, NULL);
    if (width1>width) width=width1;
    
    button = MW ("ru_stop_btn",commandWidgetClass,form,
		 XtNfromVert,label5,
		 XtNtop,XawChainTop,XtNbottom,XawChainTop,
		 XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    XtAddCallback(button,XtNcallback,stoprecord_action,(XtPointer) popup);
    XtVaGetValues(button,XtNwidth, &height, NULL);
    XtVaSetValues(label1,XtNwidth, width, NULL);
    XtVaSetValues(label2,XtNwidth, width, NULL);
    XtVaSetValues(label3,XtNwidth, width, NULL);
    XtVaSetValues(label4,XtNwidth, width, NULL);
    XtVaSetValues(label5,XtNwidth, width, NULL);
    XtVaSetValues(button,XtNhorizDistance, (width/2)-(height/2), NULL);
    
    popup_centered(popup);
}

void upopup_action(Widget w, XtPointer client_data, XtPointer call_data)
{
    u_popup=w;
    start_clock=clock();
    p_id=XtAppAddWorkProc(XtWidgetToApplicationContext(w),update_record,
			  NULL);
    start_clock=0;
}

Boolean update_record(XtPointer client_data)
{
    double recordtime;
    long playsize;
#define ADD_CLOCK 90000
    
    start_clock+=ADD_CLOCK;
    recordtime=(double) start_clock/ONE_SECOND;
    playsize=(long) (recordtime*(double)(freq*mode*(res/8)));
    sprintf(MD->mw->messages,"%.2lf kbytes (%.2lfs)",
	    (double)playsize/1024,recordtime);
    XtVaSetValues(label5,XtNlabel, MD->mw->messages, NULL);
#if defined(linux)||defined(FreeBSD)||defined(sun)
    usleep(80000);
#elif defined(sgi)
    sginap(CLK_TCK/(ONE_SECOND/80000));
#endif
    return(False);
}

void abort_record()
{
    void stop_record();

    signal(SIGUSR2,SIG_IGN);
    if (u_popup!=None) {
	XtRemoveWorkProc(p_id);
	XtDestroyWidget(u_popup);
	warn_popup(MD->mw->form,app_resources.err_bg);
	stop_record();
    }
}

void stop_record()
{
    int pid_status;
    
    waitpid((pid_t)pid, &pid_status, 0);
    signal(SIGUSR1,SIG_IGN);
    pid=-1;
    load_file(fname,AF_UNDEFINED,AF_COMP_UNDEFINED);
    update_display(MD);
    XtFree(fname);
    count++;
    return;
}

void stoprecord_action(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget popup=(Widget) client_data;
    
    signal(SIGUSR2,SIG_IGN);
    kill((pid_t)pid,SIGUSR1);
    XtRemoveWorkProc(p_id);
    XtDestroyWidget(popup);
    return;
}

