/****************************************************************************
 *
 *	  pathlight.cc: implementation of simplified pathtracing
 *      This is part of the yafray package
 *      Copyright (C) 2002 Alejandro Conty Estevez and Alfredo de Greef
 *
 *      This library is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU Lesser General Public
 *      License as published by the Free Software Foundation; either
 *      version 2.1 of the License, or (at your option) any later version.
 *
 *      This library is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *      Lesser General Public License for more details.
 *
 *      You should have received a copy of the GNU Lesser General Public
 *      License along with this library; if not, write to the Free Software
 *      Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include "pathlight.h"
using namespace std;

__BEGIN_YAFRAY

#define WARNING cerr<<"[pathLight]: "

hash3d_t<pathAccum_t> *hash;
mutex_t hash_mutex;


pathLight_t::pathLight_t(int nsam, CFLOAT pwr, int depth,int cdepth,bool uQ,
		bool ca,PFLOAT casiz,CFLOAT thr,bool recal,bool di,bool shows,bool useg,int grids)
		: samples(nsam), power(pwr), maxdepth(depth),maxcausdepth(cdepth),use_QMC(uQ),
cache(ca),cache_size(casiz*M_PI),half_cache(casiz*0.5*M_PI),
recalculate(recal),direct(di),show_samples(shows),use_gradient(useg),
gridsize(grids),threshold(thr)
{
  if (use_QMC) {
    depth = (depth+1)*2;
    HSEQ = new Halton[depth];
    int base = 2;
    for (int i=0;i<depth;i++) {
      HSEQ[i].setBase(base);
      base = nextPrime(base);
    }
  }
  else {
    // samples must be integer squared value for jittered sampling
    int g = int(sqrt((float)samples));
    g *= g;
    if (g!=samples) {
      cout << "Samples value changed from " << samples << " to " << g << endl;
      samples = g;
    }
    grid = int(sqrt((float)samples));
    gridiv = 1.0/PFLOAT(grid);
    HSEQ = NULL;
  }
  sampdiv = 1.0 / PFLOAT(samples);

	hash=NULL;
	tree=NULL;
	angle_threshold=0.8;
	shadow_threshold=0.1;
	search=9;
	devaluated=1.0;
}

pathLight_t::~pathLight_t() 
{ 
	if (HSEQ) delete[] HSEQ;  HSEQ=NULL; 
	if (cache && (hash!=NULL)) {delete hash;hash=NULL;};
	if (tree!=NULL) delete tree;
}

void pathLight_t::init(scene_t &scene)
{
	if(hash!=NULL) {delete hash;cache_size*=0.5;}
	if(tree!=NULL) {delete tree;tree=NULL;}
	if(cache)
	{
		hash=new hash3d_t<pathAccum_t>(cache_size,100000);
		scene.setRepeatFirst();
	}
	use_in_indirect=false;
	dist_to_sample=cache_size*0.1;
	scene.getPublishedData("globalPhotonMap",pmap);
	scene.getPublishedData("irradianceGlobalPhotonMap",imap);
	scene.getPublishedData("irradianceHashMap",irhash);
}


color_t pathLight_t::illuminate(renderState_t &state,const scene_t &sc, const surfacePoint_t sp,
				const vector3d_t &eye) const
{
	if(cache) 
	{
		if(tree!=NULL) return interpolate(state,sc,sp,eye);
		else return cached(state,sc,sp,eye);
	}
	else return normalSample(state,sc,sp,eye);
}

photonData_t *pathLight_t::getPhotonData(renderState_t &state)const
{
	photonData_t *data=NULL;
	if(imap!=NULL)
	{
		bool present;
		data=state.context.get(photonData,present);
		if(!present)
		{
			data=new photonData_t(imap->getMaxRadius(),new vector<foundPhoton_t>(5+1));
			state.context.store(photonData,data);
		}
	}
	return data;
}

color_t pathLight_t::getLight(renderState_t &state,const surfacePoint_t &sp,
		const scene_t &sc,const vector3d_t &eye,photonData_t *data)const
{
	vector3d_t N = FACE_FORWARD(sp.Ng(), sp.N(), eye);
	color_t total(0,0,0);
	if(imap!=NULL)
	{
		bool slowway=false;
		const globalPhotonLight_t::compPhoton_t *irr=irhash->findExistingBox(sp.P());
		if(irr!=NULL) 
		{
			CFLOAT factor=irr->N*N;
			if(factor>0.7)	total=irr->irr*factor;
			else slowway=true;
		} else slowway=true;
		if(slowway)
		{
			vector<foundPhoton_t> &found=*(data->found);
			imap->gather(sp.P(),N,found,5,data->radius,0.25);
			if(found.size())
			{
				PFLOAT farest;
				if(found.size()==1) farest=data->radius;
				else farest=found.front().dis;
				if(farest==0.0) farest=1.0;
				PFLOAT div=0.0;
				for(vector<foundPhoton_t>::iterator i=found.begin();i!=found.end();++i)
				{
					PFLOAT factor=i->photon->direction()*N*(1.0-i->dis/farest);
					if(factor>0)
					{
						total+=i->photon->color()*factor;
						div+=factor;
					}
				}
				if(div>0.0) total*=1.0/div;
			}
		}
		total = total * sp.getShader()->getDiffuse(state,sp,N);
		total += sc.light(state,sp,sp.P()+eye,true);
		total += sp.getShader()->fromRadiosity(state,sp,
					energy_t(N,color_t(0.0)),eye);
	}
	else
	{
			total = sc.light(state,sp,sp.P()+eye,true);
			total += sp.getShader()->fromRadiosity(state,sp,
					energy_t(N,color_t(0.0)),eye);
	}
	return total;
}

static bool followCaustic(vector3d_t &ray,color_t &raycolor,
		const vector3d_t &N,const vector3d_t &FN,object3d_t *obj)
{
	if(!obj->caustics()) return false;
	color_t caus_rcolor,caus_tcolor;
	PFLOAT caus_IOR;
	obj->getCaustic(caus_rcolor, caus_tcolor, caus_IOR);
	CFLOAT kr,kt;
	vector3d_t edir=-ray;
	fresnel(edir,N,caus_IOR,kr,kt);
	color_t ref=caus_rcolor*kr;
	color_t trans=caus_tcolor*kt;
	CFLOAT pref=ref.getRed()+ref.getGreen()+ref.getBlue();
	CFLOAT ptrans=trans.getRed()+trans.getGreen()+trans.getBlue();
	if( (pref==0.0) && (ptrans==0.0) ) return false;
	if((pref/(pref+ptrans))>ourRandom())
	{
			ray=reflect(FN,edir);
			raycolor*=ref;
	}
	else
	{
			ray=refract(N,edir,caus_IOR);
			raycolor*=trans;
	}
	return true;
}

color_t pathLight_t::takeSample(renderState_t &state,const vector3d_t &N,
		const surfacePoint_t &sp,const scene_t &sc,PFLOAT &avgD,PFLOAT &minD,
		bool caching)const
{
	PFLOAT maxdist=1000000*sc.getWorldResolution()*sp.Z();
	int oldlevel=state.rayDivision;
	if(!direct) state.rayDivision=oldlevel*samples;
	int localsamples;
	if(!caching)	localsamples=samples/oldlevel;
	else localsamples=samples;

	if(localsamples==0) localsamples=1;
	color_t total(0.0);
	PFLOAT HNUM=0,HD=0,H,M=0;

	photonData_t *data=getPhotonData(state);
	if(direct) {avgD=maxdist;minD=maxdist;return getLight(state,sp,sc,N,data);}
	hemiSampler_t *sampler=getSampler(state,sc);
	sampler->samplingFrom(state,sp.P(),N,sp.NU(),sp.NV());


	for(int i=0;i<samples;++i)
	{
		color_t raycolor(1.0);
		vector3d_t ray = sampler->nextDirection(sp.P(),N, sp.NU(), sp.NV(), i, 0,raycolor);
		vector3d_t startray = ray;
		point3d_t where = sp.P();
		if(caching) HNUM+=1;
		for(int j=0,cj=0;j<maxdepth;++j)
		{
			if (raycolor.energy()<0.05) break;
			surfacePoint_t tempsp;
			if (!sc.firstHit(state,tempsp, where, ray, true)) //background reached
			{
				total += (startray*N)*raycolor*sc.getBackground(ray,state);
				break;
			}
			if(caching)
			{
				if((j==0) && (tempsp.Z()>0)) HD+=1.0/tempsp.Z();
				if((j==0) && ((tempsp.Z()<M) || (M==0)) ) M=tempsp.Z();
			}
			color_t light = getLight(state,tempsp,sc,-ray,data);
		//	color_t light = sc.light(state,tempsp,where,true);
		//	if (light.energy()>0.05)
		//		total += (startray*N)*raycolor*light;
		//	light = tempsp.getShader()->fromRadiosity(state,tempsp,
		//			energy_t(startray,color_t(0.0)),-ray);
		//	if (light.energy()>0.05)
			total += (startray*N)*raycolor*light;
			vector3d_t HN = FACE_FORWARD(tempsp.Ng(), tempsp.N(), -ray);
			if(!followCaustic(ray,raycolor,tempsp.N(),HN,tempsp.getObject()))
			{
				raycolor *= tempsp.getShader()->getDiffuse(state, tempsp, -ray);
				ray = sampler->nextDirection(tempsp.P(),HN, tempsp.NU(), tempsp.NV(),
						i, j+1,raycolor);
			}
			else if(cj<maxcausdepth) {cj++;j--;}
			where = tempsp.P();
		}
	}
	if(caching)
	{
		if(HD>0) H=HNUM/HD; else H=maxdist;
		if(M<=0) M=maxdist;
		avgD=H;
		minD=M;
	}
	state.rayDivision=oldlevel;
	total*=sampler->multiplier();
	return total;
}
		

color_t pathLight_t::normalSample(renderState_t &state,const scene_t &sc, 
		const surfacePoint_t sp, const vector3d_t &eye) const
{
	color_t light;
	vector3d_t N = FACE_FORWARD(sp.Ng(), sp.N(), eye);
	const shader_t *sha = sp.getShader();
	color_t total(0.0);

	if (sha->getDiffuse(state, sp, eye).energy()<0.05f) return total;

	PFLOAT foo1,foo2;
	total=takeSample(state,N,sp,sc,foo1,foo2);
	return sp.getShader()->getDiffuse(state, sp, eye)*total*power;
}

//------------- CACHE


inline point3d_t toPolar(const point3d_t &P,const scene_t &sc)
{
	vector3d_t v=P-sc.getCenterOfView();
	PFLOAT dxy=sqrt(v.x*v.x+v.y*v.y);
	PFLOAT x=v.length();
	v.x/=dxy;
	v.y/=dxy;
	v.z/=x;
	PFLOAT y=(v.y>0) ? acos(v.x) : -acos(v.x);
	PFLOAT z=asin(v.z);
	x=log(x);
	y*=cos(z);
	return point3d_t(x,y,z);
}

inline point3d_t toRealPolar(const point3d_t &P,const scene_t &sc)
{
	vector3d_t v=P-sc.getCenterOfView();
	PFLOAT dxy=sqrt(v.x*v.x+v.y*v.y);
	PFLOAT x=v.length();
	v.x/=dxy;
	v.y/=dxy;
	v.z/=x;
	PFLOAT y=(v.y>0) ? acos(v.x) : -acos(v.x);
	PFLOAT z=asin(v.z);
	x=log(x);
	return point3d_t(x,y,z);
}

static PFLOAT polarDist(const point3d_t &a,const point3d_t &b)
{
  PFLOAT raddif = b.x-a.x;
	//PFLOAT coco = cos(a.z)*cos(b.z)*cos(a.y-b.y)+sin(a.z)*sin(b.z);
	PFLOAT temp=sin((a.y-b.y)*0.5);
	PFLOAT res = cos(a.z-b.z)-2*cos(a.z)*cos(b.z)*temp*temp;
	res = acos(res);
	return sqrt(res*res+raddif*raddif);
}


static bound_t path_calc_bound(const vector<const pathSample_t *> &v)
{
	int size=v.size();
	if(size==0) return bound_t(point3d_t(),point3d_t());
	PFLOAT maxx,maxy,maxz,minx,miny,minz;
	maxx=minx=v[0]->realPolar.x;
	maxy=miny=v[0]->realPolar.y;
	maxz=minz=v[0]->realPolar.z;
	for(int i=0;i<size;++i)
	{
		const point3d_t &p=v[i]->realPolar;
		if(p.x>maxx) maxx=p.x;
		if(p.y>maxy) maxy=p.y;
		if(p.z>maxz) maxz=p.z;
		if(p.x<minx) minx=p.x;
		if(p.y<miny) miny=p.y;
		if(p.z<minz) minz=p.z;
	}
	return bound_t(point3d_t(minx,miny,minz),point3d_t(maxx,maxy,maxz));
}

static bool path_is_in_bound(const pathSample_t * const & p,bound_t &b)
{
	return b.includes(p->realPolar);
}

static point3d_t path_get_pos(const pathSample_t * const & p)
{
	return p->realPolar;
}

struct circle_t
{
	circle_t(const point3d_t &pp,PFLOAT r):point(pp),radius(r) {};
	point3d_t point;
	PFLOAT radius;
};


struct pointCross_f
{
	bool operator()(const circle_t &c,const bound_t &b)const 
	{
		const point3d_t &p=c.point;
		PFLOAT dis=(p.y-b.centerY())*cos(p.z);
		PFLOAT dis2=(p.y-b.centerY() + 
									((p.y<0) ? 2*M_PI : -2*M_PI))*cos(p.z);
		point3d_t temp(p.x,b.centerY()+dis,p.z);
		point3d_t temp2(p.x,b.centerY()+dis2,p.z),a,g;
		b.get(a,g);
		a+=-c.radius;
		g+=c.radius;
		bound_t bt(a,g);
		return bt.includes(temp) || bt.includes(temp2);
	};
};

CFLOAT pathLight_t::weight(const pathSample_t &sample,const point3d_t &P,
		const vector3d_t &N,CFLOAT avgOclu)const
{
	vector3d_t PP=P - sample.P;
	vector3d_t avgN=N+sample.N;
	avgN.normalize();
	avgOclu=sample.adist;
	if(avgOclu==0.0) return 0;
	
	CFLOAT D=PP.normLen()-sample.precision*2.0;
	if(D<0) D=0;
	CFLOAT a=sample.devaluated*D/avgOclu;
	CFLOAT cinv=1.000001-(sample.N*N);
	CFLOAT b=sqrt(cinv);
	cinv*=cinv;
	cinv*=cinv;
	b+=4*cinv;
	CFLOAT c=10.0*fabs(avgN*PP);
	//CFLOAT d=(D>avgOclu) ? (D-avgOclu)/avgOclu : 0.0;
	//return a*b*exp(1.0-1.0/(c*c))*exp(1.0-1.0/(d*d));
	CFLOAT f=a+b+c;
	CFLOAT A=2.0/shadow_threshold;
	f*=A;
	f*=f;
	f*=f;
	return A/pow(1.0+f, 0.25);
	//if(f<0.01) f=0.01;
	//return sample.devaluated/f;
}

CFLOAT pathLight_t::weightNoPrec(const pathSample_t &sample,const point3d_t &P,
		const vector3d_t &N,CFLOAT avgOclu)const
{
	vector3d_t PP=P - sample.P;
	vector3d_t avgN=N+sample.N;
	avgN.normalize();
	avgOclu=sample.adist;
	if(avgOclu==0.0) return 0;
	
	CFLOAT D=PP.normLen()-sample.precision;
	if(D<0) D=0;
	CFLOAT a=sample.devaluated*D/avgOclu;
	CFLOAT cinv=1.000001-(sample.N*N);
	CFLOAT b=sqrt(cinv);
	cinv*=cinv;
	cinv*=cinv;
	b+=4*cinv;
	CFLOAT c=10.0*fabs(avgN*PP);
	//CFLOAT d=(D>avgOclu) ? (D-avgOclu)/avgOclu : 0.0;
	//return a*b*exp(1.0-1.0/(c*c))*exp(1.0-1.0/(d*d));
	CFLOAT f=a+b+c;
	CFLOAT A=2.0/shadow_threshold;
	f*=A;
	f*=f;
	f*=f;
	return A/pow(1.0+f, 0.25);
	//if(f<0.01) f=0.01;
	//return sample.devaluated/f;
}


CFLOAT pathLight_t::weightNoDist(const pathSample_t &sample,const point3d_t &P,
		const vector3d_t &N,CFLOAT avgOclu)const
{
	vector3d_t PP=P - sample.P;
	vector3d_t avgN=N+sample.N;
	avgN.normalize();
	
	CFLOAT cinv=1.000001-(sample.N*N);
	CFLOAT b=sqrt(cinv);
	cinv*=cinv;
	cinv*=cinv;
	b+=4*cinv;
	CFLOAT c=10.0*fabs(avgN*PP);
	CFLOAT f=b+c;
	CFLOAT A=2.0/shadow_threshold;
	f*=A;
	f*=f;
	f*=f;
	return A/pow(1.0+f, 0.25);
	//if(f<0.01) f=0.01;
	//return 1.0/f;
}

CFLOAT pathLight_t::weightNoDev(const pathSample_t &sample,const point3d_t &P,
		const vector3d_t &N,CFLOAT avgOclu)const
{
	vector3d_t PP=P - sample.P;
	vector3d_t avgN=N+sample.N;
	avgN.normalize();
	avgOclu=sample.adist;
	if(avgOclu==0.0) return 0;
	
	CFLOAT D=PP.normLen()-sample.precision;
	if(D<0) D=0;
	CFLOAT a=D/avgOclu;
	CFLOAT cinv=1.000001-(sample.N*N);
	CFLOAT b=sqrt(cinv);
	cinv*=cinv;
	cinv*=cinv;
	b+=4*cinv;
	CFLOAT c=10.0*fabs(avgN*PP);
	CFLOAT f=a+b+c;
	CFLOAT A=2.0/shadow_threshold;
	f*=A;
	f*=f;
	f*=f;
	return A/pow(1.0+f, 0.25);
	//if(f<0.01) f=0.01;
	//return 1.0/f;
}


struct compareFound_f
{
	bool operator () (const foundSample_t &a,const foundSample_t &b)
	{
		return a.weight>b.weight;
	}
};

CFLOAT pathLight_t::gatherSamples(const point3d_t &P,const point3d_t &pP,
		const vector3d_t &N,vector<foundSample_t> &found,unsigned int K,
		PFLOAT &radius,unsigned int minimun,
		CFLOAT (pathLight_t::*W)(const pathSample_t &,const point3d_t &,
			          const vector3d_t &,CFLOAT)const)const
{
	foundSample_t temp;
	compareFound_f cfound;
	found.reserve(K+1);
	CFLOAT best=0,max=0,adist=0;
	unsigned int reached=0;
	bool repeat=true;
		
	while(repeat)
	{
		reached=0;
		best=0;
		adist=0;
		found.clear();
		circle_t circle(pP,radius);
		for(gObjectIterator_t<const pathSample_t *,circle_t,pointCross_f> 
				i(tree,circle);!i;++i)
		{
			CFLOAT pD=polarDist(pP,(*i)->realPolar);
			if(pD>radius) continue;
			reached++;
			temp.S=*i;
			temp.dis=pD;
			temp.weight=/* (1.0 - pD/radius )*/(this->*W)(**i,P,N,0);
			//if(temp.weight<=(0.8/shadow_threshold)) continue;
			if((found.size()==K) && (temp.weight<found.front().weight)) continue;
			if(temp.weight>best) {best=temp.weight;adist=temp.S->adist;}
			//if((min<0) || (temp.weight<min)) min=temp.weight;
			if(found.size()==K)
			{
				found.push_back(temp);
				push_heap(found.begin(),found.end(),cfound);
				pop_heap(found.begin(),found.end(),cfound);
				found.pop_back();
			}
			else
			{
				found.push_back(temp);
				push_heap(found.begin(),found.end(),cfound);
			}
		}
		while((found.size()>minimun) && (found.front().weight<=(0.8/shadow_threshold)))
		{
			pop_heap(found.begin(),found.end(),cfound);
			found.pop_back();
		}
		
		PFLOAT rrad=(found.front().dis==0) ? (0.0001*adist) : 
			radius*(found.front().S->P-P).length()/found.front().dis;
		repeat=( (((adist/rrad)>(0.8/shadow_threshold)) && (reached<K)) || (best<=(0.5/shadow_threshold))
				) && (radius<cache_size);
		if(repeat) radius*=2;
		if(radius>cache_size) radius=cache_size;
	}
	if(reached>K)
	{
		PFLOAT f=(PFLOAT)K/(PFLOAT)reached;
		if(f<(0.7*0.7)) radius*=0.95;
	}
	if(radius>cache_size) radius=cache_size;
	for(vector<foundSample_t>::iterator i=found.begin();i!=found.end();++i)
		if(i->dis>max) max=i->dis;
	
	return found.front().weight;
}

CFLOAT pathLight_t::gatherSamples(const point3d_t &P,const point3d_t &pP,
		const vector3d_t &N,vector<foundSample_t> &found,PFLOAT radius,
		CFLOAT (pathLight_t::*W)(const pathSample_t &,const point3d_t &,
			          const vector3d_t &,CFLOAT)const)const
{
	foundSample_t temp;
	compareFound_f cfound;
		
	found.clear();
	circle_t circle(pP,radius);
	for(gObjectIterator_t<const pathSample_t *,circle_t,pointCross_f> 
			i(tree,circle);!i;++i)
	{
		CFLOAT pD=polarDist(pP,(*i)->realPolar);
		if(pD>radius) continue;
		temp.S=*i;
		temp.dis=pD;
		temp.weight=(this->*W)(**i,P,N,0);
		if(temp.weight<=(0.8/shadow_threshold)) continue;
			found.push_back(temp);
			push_heap(found.begin(),found.end(),cfound);
	}
	
	return found.front().weight;
}

inline CFLOAT smoothFilter(CFLOAT x)
{
	CFLOAT temp=-2.0*x+3.0;
	return x*x*temp;
}

color_t pathLight_t::interpolate(renderState_t &state,const scene_t &s,
		const surfacePoint_t sp, const vector3d_t &eye) const
{
	if (sp.getShader()->getDiffuse(state, sp, eye).energy()<0.05f) return color_t(0.0);
	vector3d_t N = FACE_FORWARD(sp.Ng(), sp.N(), eye);
	point3d_t pP=toRealPolar(sp.P(),s);
	PFLOAT radius;
	bool present;
	radius=state.context.get(lastRadius,present);
	if(!present) radius=cache_size;
	
	vector<foundSample_t> samples;
	CFLOAT farest;
	if(show_samples)
	{
		radius=dist_to_sample*0.5;
		farest=gatherSamples(sp.P(),pP,N,samples,1,radius);
		if(samples.size() && (radius==dist_to_sample*0.5)) return color_t(1,1,1);
		else return color_t(0,0,0);
	}
	else
		farest=gatherSamples(sp.P(),pP,N,samples,search,radius,3);
	
	state.context.store(lastRadius,radius);
	if(samples.size()==1) farest=0;
	else if(farest>0.8/shadow_threshold) farest=0.8/shadow_threshold;

	//CFLOAT max=0;
	for(vector<foundSample_t>::iterator i=samples.begin();i!=samples.end();++i)
	{
		if(i->weight>2.0/shadow_threshold) i->weight=2.0/shadow_threshold;
		i->weight=(i->weight-farest)*(1.0-i->dis/cache_size);
	}

	color_t total(0,0,0);
	CFLOAT amount=0;
	for(vector<foundSample_t>::iterator i=samples.begin();i!=samples.end();++i)
	{
		CFLOAT f=i->weight;
		//color_t temp=f*(i->S->color+i->S->gradient.getVariation(i->S->P-sp.P()));
		if(use_gradient)
			total+=f*i->S->gradient.getVariation(i->S->color,sp.P()-i->S->P);
		else
			total+=f*i->S->color;
		amount+=f;
	}
	/*
	float sx,sy;
	sp.getScreenPos(sx,sy);
	if( ((int)((sx+1.0)*0.5*1024.0+0.5)==334) && ((int)((1.0-sy)*0.5*768.0+0.5)==498)) 
	{
		cout<<endl;
		for(vector<foundSample_t>::iterator i=samples.begin();i!=samples.end();++i) cout<<i->S->color<<"\t";
		cout<<endl;
		for(vector<foundSample_t>::iterator i=samples.begin();i!=samples.end();++i) cout<<(i->weight)<<" ";
		cout<<"far "<<farest<<"\t";
		cout<<endl;
		cout<<total<<endl;
		return color_t(1,0,0);
	}
	*/
	if(amount!=0) amount=1.0/amount;
	else {cout<<".";cout.flush(); return normalSample(state,s,sp,eye);}
	total*=amount;
	return sp.getShader()->getDiffuse(state, sp, eye)*total*power;
}

void pathLight_t::setIrradiance(pathSample_t &sample,PFLOAT &radius)
{
	vector3d_t &N = sample.N;
	point3d_t &pP= sample.realPolar;
	point3d_t &P= sample.P;
	//static vector<foundSample_t> samples;
	stsamples.clear();
	CFLOAT farest;

	farest=gatherSamples(P,pP,N,stsamples,search,radius,3,&pathLight_t::weightNoDev);
	
	if(stsamples.size()==1) farest=0;
	else if(farest>0.8/shadow_threshold) farest=0.8/shadow_threshold;

	for(vector<foundSample_t>::iterator i=stsamples.begin();i!=stsamples.end();++i)
	{
		if(i->weight>2.0/shadow_threshold) i->weight=2.0/shadow_threshold;
		i->weight=(i->weight-farest)*(1.0-i->dis/cache_size);
	}

	color_t total(0,0,0);
	CFLOAT amount=0;
	for(vector<foundSample_t>::iterator i=stsamples.begin();i!=stsamples.end();++i)
	{
		total+=i->weight*i->S->color;
		amount+=i->weight;
	}
	if(amount!=0) amount=1.0/amount;
	else sample.mixed=color_t(0,0,0);
	sample.mixed=total*power*amount;
}

CFLOAT pathLight_t::getGradient(const pathAccum_t &acc)
{
	CFLOAT max=0;
	for(list<pathSample_t>::const_iterator i=acc.radiance.begin();
			i!=acc.radiance.end();++i)
	{
		CFLOAT ene1=maxAbsDiff(i->mixed,color_t(0,0,0));
		//PFLOAT norm=1.0/acc.maxdist;
		circle_t circle(i->realPolar,acc.maxdist);
		for(gObjectIterator_t<const pathSample_t *,circle_t,pointCross_f> 
				j(tree,circle);!j;++j)
		{
			CFLOAT pD=polarDist(i->realPolar,(*j)->realPolar);
			if(pD>acc.maxdist) continue;
			//pD*=norm;
			//if(pD<=0.0) pD=1.0;
			CFLOAT ene2=maxAbsDiff((*j)->mixed,color_t(0,0,0));
			if(ene1>ene2) ene2=ene1;
			if(ene2<1.0) ene2=1.0;
			
			CFLOAT cdiff=maxAbsDiff(i->mixed,(*j)->mixed)*(i->N*(*j)->N)/ene2;
			if(cdiff>max) max=cdiff;
		}
	}
	return max;
}

void pathLight_t::computeGradients()
{
	vector<foundSample_t> samples;
	PFLOAT radius=cache_size;
	for(hash3d_t<pathAccum_t>::iterator i=hash->begin();
			i!=hash->end();++i)
	{
		for(list<pathSample_t>::iterator j=(*i).radiance.begin();
				j!=(*i).radiance.end();++j)
		{
			vector3d_t dR,dG,dB;
			CFLOAT maxR,minR,maxG,minG,maxB,minB;
			maxR=minR=j->color.getRed();
			maxG=minG=j->color.getGreen();
			maxB=minB=j->color.getBlue();
			gatherSamples(j->P,j->realPolar,j->N,samples,4,radius,4);
			
			for(vector<foundSample_t>::iterator k=samples.begin();k!=samples.end();++k)
			{
				if(k->S==&(*j)) continue;
				vector3d_t dir=k->S->P-j->P;
				PFLOAT len=dir.normLen();
				if((len>j->M) || (len>k->S->M)) continue;
				CFLOAT factor=1.0-(k->weight-(1.0/shadow_threshold))/(1.0/shadow_threshold);
				if(factor<0.0) factor=0.0; else if(factor>1.0) factor=1.0;
				factor=(j->N*k->S->N)*(1.000001-j->N*dir)*factor;
				if(factor<0) factor=0;
				CFLOAT aR=k->S->color.getRed()-j->color.getRed();
				CFLOAT aG=k->S->color.getGreen()-j->color.getGreen();
				CFLOAT aB=k->S->color.getBlue()-j->color.getBlue();
				CFLOAT iR=aR*factor+j->color.getRed(),iG=aG*factor+j->color.getGreen(),
							 iB=aB*factor+j->color.getBlue();
				dir*=factor;
				vector3d_t actualRdir=dR;
				vector3d_t actualGdir=dG;
				vector3d_t actualBdir=dB;
				actualRdir.normalize();
				actualGdir.normalize();
				actualBdir.normalize();
				dR+=dir*(aR/len);
				dR*=1.0/(1.0+fabs(dir*actualRdir));
				if(iR>maxR) maxR=iR; else if(iR<minR) minR=iR;
				dG+=dir*(aG/len);
				dG*=1.0/(1.0+fabs(dir*actualGdir));
				if(iG>maxG) maxG=iG; else if(iG<minG) minG=iG;
				dB+=dir*(aB/len);
				dB*=1.0/(1.0+fabs(dir*actualBdir));
				if(iB>maxB) maxB=iB; else if(iB<minB) minB=iB;
			}
			j->gradient.dR=dR;
			j->gradient.dG=dG;
			j->gradient.dB=dB;
			j->gradient.minR=minR;
			j->gradient.minG=minG;
			j->gradient.minB=minB;
			j->gradient.maxR=maxR;
			j->gradient.maxG=maxG;
			j->gradient.maxB=maxB;
		}
	}
}

bool pathLight_t::testRefinement(const scene_t &sc)
{
	if(threshold>=1.0) return false;
	if(devaluated>2)
	{
		for(hash3d_t<pathAccum_t>::iterator i=hash->begin();
				i!=hash->end();++i)
			for(list<pathSample_t>::iterator j=(*i).radiance.begin();
					j!=(*i).radiance.end();++j)
			{
				if(j->devaluated==1.0) continue;
				CFLOAT div=j->M/(j->precision*j->devaluated);
				if(div<1.0) div=1.0;
				j->devaluated=1.0+(j->devaluated-1.0)/div;
			}
		return false;
	}
	devaluated*=2;
	bool changed=false;
	int change=0;
	int num=0;

	cache_size*=2.0;
	PFLOAT mixradius=cache_size;
	for(hash3d_t<pathAccum_t>::iterator i=hash->begin();
			i!=hash->end();++i)
		for(list<pathSample_t>::iterator j=(*i).radiance.begin();
				j!=(*i).radiance.end();++j)
			setIrradiance(*j,mixradius);

	vector<foundSample_t> samples;
	mixradius=cache_size;
	for(hash3d_t<pathAccum_t>::iterator i=hash->begin();
			i!=hash->end();++i)
		for(list<pathSample_t>::iterator j=(*i).radiance.begin();
				j!=(*i).radiance.end();++j,++num)
		{
			CFLOAT minR=1,minG=1,minB=1;
			CFLOAT maxR=0,maxG=0,maxB=0;
			samples.clear();
			gatherSamples(j->P,j->realPolar,j->N,samples,35,mixradius,2,&pathLight_t::weightNoDist);
			for(vector<foundSample_t>::iterator i=samples.begin();i!=samples.end();++i)
			{
				if(i->S->mixed.getRed()  >maxR) maxR=i->S->mixed.getRed();
				if(i->S->mixed.getGreen()>maxG) maxG=i->S->mixed.getGreen();
				if(i->S->mixed.getBlue() >maxB) maxB=i->S->mixed.getBlue();
				if(i->S->mixed.getRed()  <minR) minR=i->S->mixed.getRed();
				if(i->S->mixed.getGreen()<minG) minG=i->S->mixed.getGreen();
				if(i->S->mixed.getBlue() <minB) minB=i->S->mixed.getBlue();
			}
			
			color_t min(minR,minG,minB),max(maxR,maxG,maxB);
			//min*=power;
			//max*=power;
			sc.adjustColor(min);
			sc.adjustColor(max);
			min.clampRGB01();
			max.clampRGB01();
			if((maxAbsDiff(max,min))>threshold)
			{
				j->devaluated=devaluated;
				changed=true;
				change++;
			}
		}
	cout<<"\nRefinement:"<<change<<"/"<<num<<"   "<<endl;
	cache_size*=0.5;
	return changed;
}

/*
bool pathLight_t::couldBeUseful(const vector3d_t &N,const point3d_t &P,const point3d_t &rP,
		const pathAccum_t *a)const
{
	if(a==NULL) return false;
	if(! a->valid ) return false;
	for(list<pathSample_t>::const_iterator i=a->radiance.begin();
			i!=a->radiance.end();++i)
	{
		if(weight(*i,P,N)<(1.0/shadow_threshold)) continue;
		return true;
	}
	return false;
}
*/

bool pathLight_t::couldBeUseful(const vector3d_t &N,const point3d_t &P,const point3d_t &rP,
		int cx,int cy,int cz)const
{
	pathAccum_t *a;

	for(int i=cx;i<=(cx+1);i+=(i==cx) ? -1 : ((i<cx) ? 2 : 1) )
		for(int j=cy;j<=(cy+1);j+=(j==cy) ? -1 : ((j<cy) ? 2 : 1) )
			for(int k=cz;k<=(cz+1);k+=(k==cz) ? -1 : ((k<cz) ? 2 : 1) )
			{
				a=hash->findExistingBox(i,j,k);
				if((a==NULL) || ! a->valid) continue;
				for(list<pathSample_t>::iterator l=a->radiance.begin();
							l!=a->radiance.end();++l)
				{
					PFLOAT pD=polarDist(rP,l->realPolar);
					if(pD>cache_size) continue;
					if((weightNoPrec(*l,P,N))<(1.0/shadow_threshold)) continue;
		/*
		vector3d_t PP=P - i->P;
		//PFLOAT D=PP.normLen();
		PFLOAT pD=polarDist(rP,i->realPolar);
		vector3d_t avgN=N+i->N;
		avgN.normalize();
		if((N*i->N)<angle_threshold) continue;
		if(sqrt(fabs(avgN*PP))>(1.0-angle_threshold)) continue;
		if((i->adist / (i->adist + D)) < shadow_threshold) continue;
		//if(pD>a->maxdist) continue;
		*/
					a->radiance.push_front(*l);
					a->radiance.erase(l);
					return true;
				}
			}
	return false;
}
		
color_t pathLight_t::cached(renderState_t &state,const scene_t &sc,
		const surfacePoint_t sp, const vector3d_t &eye) const
{
	if (sp.getShader()->getDiffuse(state, sp, eye).energy()<0.05f) return color_t(0.0);
	int cx,cy,cz;
	point3d_t pP=toPolar(sp.P(),sc);
	//point3d_t rP(pP.x+half_cache,pP.y+half_cache,pP.z+half_cache);
	point3d_t rP(pP);
	hash->getBox(rP,cx,cy,cz);
	//const pathAccum_t *A=hash->findExistingBox(cx,cy,cz);
	vector3d_t N = FACE_FORWARD(sp.Ng(), sp.N(), eye);

	hash_mutex.wait();
	if(!couldBeUseful(N,sp.P(),toRealPolar(sp.P(),sc),cx,cy,cz))
	{
		hash_mutex.signal();
		PFLOAT H,M;
		color_t total=takeSample(state,N,sp,sc,H,M,true);
		hash_mutex.wait();
		//A=hash->findExistingBox(cx,cy,cz);
		if(!couldBeUseful(N,sp.P(),toRealPolar(sp.P(),sc),cx,cy,cz))
		{
			pathAccum_t &nuevo=hash->findBox(rP);
			if(!nuevo.valid) 
			{
				nuevo.radiance.clear();
				nuevo.maxdist=2*cache_size;
				nuevo.maxgradient=shadow_threshold;
			}
			nuevo.radiance.push_front(pathSample_t(N,total,H, sp.P(),
						pP,M,state.traveled*sc.getWorldResolution(),devaluated));
			nuevo.valid=true;
			//A=&nuevo;
		}
		hash_mutex.signal();
	} else hash_mutex.signal();

	return color_t(0,0,0);
}

void pathLight_t::postInit(scene_t &scene)
{
	if(!cache) return;
	vector<const pathSample_t *> pointers;

	for(hash3d_t<pathAccum_t>::iterator i=hash->begin();
			i!=hash->end();++i)
	{
		for(list<pathSample_t>::const_iterator j=(*i).radiance.begin();
				j!=(*i).radiance.end();++j)
			pointers.push_back(&(*j));
	}
	if(tree!=NULL) delete tree;
	tree=buildGenericTree(pointers,path_calc_bound,path_is_in_bound,
			path_get_pos,1);
	if(use_gradient) computeGradients();
	if(!direct && testRefinement(scene))
	{
		scene.setRepeatFirst();
		delete tree;
		tree=NULL;
	}
	else
	{
		cache_size*=2;
		cout<<pointers.size()<<" samples took\n";
	}
}

hemiSampler_t *pathLight_t::getSampler(renderState_t &state,const scene_t &sc)const
{
	bool present;
	hemiSampler_t *sam=state.context.get(_sampler,present);
	if(!present)
	{
		//const globalPhotonMap_t *pmap;
		//sc.getPublishedData("globalPhotonMap",pmap);
		if((pmap!=NULL) && (samples>96)) sam=new photonSampler_t(samples,maxdepth,*pmap,gridsize);
		else 
		if(use_QMC) sam=new haltonSampler_t(maxdepth,samples);
		else sam=new randomSampler_t(samples);
		state.context.store(_sampler,sam);
	}
	return sam;
}


light_t *pathLight_t::factory(paramMap_t &params,renderEnvironment_t &render)
{
	CFLOAT power = 1.0,thr=0.1;
	int samples = 16,depth=3,cdepth=4,search=50,grid=36;
	bool useqmc = false;
	bool cache = false;
	bool recalculate =true;
	bool direct=false;
	bool show_samples=false;
	bool useg=false;
	PFLOAT cache_size = 0.01,angt=0.2,shadt=0.3;

	params.getParam("power", power);
	params.getParam("depth", depth);
	params.getParam("caus_depth", cdepth);
	params.getParam("samples", samples);
	params.getParam("use_QMC", useqmc);
	params.getParam("cache", cache);
	params.getParam("direct", direct);
	params.getParam("grid", grid);

	if (samples<1) {
		WARNING << "Samples value too low, minimum is one\n";
		samples = 1;
	}
	if(cache)
	{
		params.getParam("cache_size", cache_size);
		params.getParam("angle_threshold",angt);
		params.getParam("threshold",thr);
		params.getParam("shadow_threshold",shadt);
		params.getParam("search",search);
		params.getParam("recalculate",recalculate);
		params.getParam("show_samples",show_samples);
		params.getParam("gradient",useg);
		if(search<3) search=3;
		//render.repeatFirstPass();
	}
	pathLight_t *path=new pathLight_t(samples, power, depth,cdepth, useqmc,
			cache,cache_size,thr,recalculate,direct,show_samples,useg,grid);
	if(cache) path->setCacheThreshold(angt,shadt,search);
	return path;
}

pluginInfo_t pathLight_t::info()
{
	pluginInfo_t info;

	info.name="pathlight";
	info.description="Montecarlo raytracing indirect lighting system";

	info.params.push_back(buildInfo<FLOAT>("power",0,10000,1.0,"Power of the indirect light"));
	info.params.push_back(buildInfo<INT>("depth",1,50,3,"Light bounces, set it to \
				1 if globalphotonmap present"));
	info.params.push_back(buildInfo<INT>("caus_depth",0,50,4,"Extra bounces when inside glass"));
	info.params.push_back(buildInfo<INT>("samples",1,5000,16,"Light samples, the \
			higher, the less noise and slower"));
	info.params.push_back(buildInfo<BOOL>("use_QMC","Whenever to use quasi montecarlo sampling"));
	info.params.push_back(buildInfo<BOOL>("cache","Whenever to cache iradiance"));
	info.params.push_back(buildInfo<BOOL>("direct","Shows the photonmap directly, use this for \
				tunning a globalphotonlight"));
	info.params.push_back(buildInfo<INT>("grid",36,36,36,"only for development"));
	info.params.push_back(buildInfo<FLOAT>("cache_size",0.000001,2*M_PI,0.01,
				"Cache mode: Size of the cache cells, at least 1 sample per cell (polar coords)"));
	info.params.push_back(buildInfo<FLOAT>("threshold",0.000001,1000,0.3,
				"Cache mode: Threshold used to know when to resample a cached value"));
	info.params.push_back(buildInfo<FLOAT>("shadow_threshold",0.000001,1000,0.3,
				"Cache mode: Quality of the shadows/lighting, the lower, the better"));
	info.params.push_back(buildInfo<INT>("search",3,1000,50,"Cache mode: Maximun \
				number of values to do interpolation"));
	info.params.push_back(buildInfo<BOOL>("show_samples","Show the sample \
				distribution instead of lighting"));
	info.params.push_back(buildInfo<BOOL>("gradient","Activates the use of \
				gradients. Not working fine, but can solve some artifacts"));

	return info;
			
}

extern "C"
{
	
void registerPlugin(renderEnvironment_t &render)
{
	render.registerFactory("pathlight",pathLight_t::factory);
	cout<<"Registered pathlight\n";
}

}
__END_YAFRAY
