/****************************************************************************
 *
 *		  pathlight.h simplified pathtracing interface
 *      This is part of the yafray package
 *      Copyright (C) 2002 Alejandro Conty Estevez
 *
 *      This library is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU Lesser General Public
 *      License as published by the Free Software Foundation; either
 *      version 2.1 of the License, or (at your option) any later version.
 *
 *      This library is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *      Lesser General Public License for more details.
 *
 *      You should have received a copy of the GNU Lesser General Public
 *      License along with this library; if not, write to the Free Software
 *      Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#ifndef __PATHLIGHT_H
#define __PATHLIGHT_H

#ifdef HAVE_CONFIG_H
#include<config.h>
#endif

#include "light.h"
#include "mcqmc.h"
#include "hash3d.h"
#include "ccthreads.h"
#include "params.h"
#include "photon.h"
#include <vector>
#include <list>
#include <utility>
#include "pathtools.h"
#include "globalphotonlight.h"

__BEGIN_YAFRAY

struct gradient_t
{
	color_t getVariation(const color_t &c,const vector3d_t &v)const
	{
		CFLOAT r=c.getRed()+v*dR;
		CFLOAT g=c.getGreen()+v*dG;
		CFLOAT b=c.getBlue()+v*dB;
		if(r<minR) r=minR; else if(r>maxR) r=maxR;
		if(g<minG) g=minG; else if(g>maxG) g=maxG;
		if(b<minB) b=minB; else if(b>maxB) b=maxB;
		return color_t(r,g,b);
	};

	vector3d_t dR,dG,dB;
	CFLOAT maxR,minR,maxG,minG,maxB,minB;
};

struct pathSample_t
{
	pathSample_t(const vector3d_t &n,const color_t &c, PFLOAT a,const point3d_t &p,
			const point3d_t &pp, PFLOAT m, PFLOAT prec,CFLOAT dev=1.0):
		N(n),color(c),adist(a),M(m),precision(prec),P(p),pP(pp),deval(false),devaluated(dev)
		{
			realPolar=pp;
			PFLOAT corr=cos(pp.z);
			if(corr>0) realPolar.y/=corr;
		};
	vector3d_t N;
	color_t color,mixed;
	PFLOAT adist,M,precision;
	point3d_t P;
	point3d_t pP;
	point3d_t realPolar;
	gradient_t gradient;
	bool deval;
	CFLOAT devaluated;
};

struct foundSample_t
{
	const pathSample_t *S;
	PFLOAT dis;
	PFLOAT weight;
};

struct pathAccum_t
{
	pathAccum_t():valid(false),resample(true) {maxdist=-1;subdivision=1;};
	std::list<pathSample_t> radiance;
	PFLOAT maxdist;
	CFLOAT maxgradient;
	int subdivision;
	bool valid,resample;
};

class photonData_t : public context_t::destructible
{
	public:
		photonData_t(PFLOAT r,std::vector<foundPhoton_t> *f) {radius=r;found=f;};
		virtual ~photonData_t() {delete found;};
		bool valid()const {return found!=NULL;};

		PFLOAT radius;
		std::vector<foundPhoton_t> *found;
};


class pathLight_t : public light_t
{
	friend struct photonData_t;
	public:
		pathLight_t(int nsam, CFLOAT pwr, int depth, int cdepth,bool uQ,
				bool ca=false,PFLOAT casiz=1.0,CFLOAT thr=0.1,bool recal=true,
				bool di=false, bool shows=false,bool useg=false,int grid=36);

		void setCacheThreshold(PFLOAT a,PFLOAT s,int se) 
		{angle_threshold=1.0-a;shadow_threshold=s;search=se;};
		virtual color_t illuminate(renderState_t &state,const scene_t &s,
				const surfacePoint_t sp, const vector3d_t &eye) const;
		color_t normalSample(renderState_t &state,const scene_t &s,
				const surfacePoint_t sp, const vector3d_t &eye) const;
		color_t interpolate(renderState_t &state,const scene_t &s,
				const surfacePoint_t sp, const vector3d_t &eye) const;
		color_t cached(renderState_t &state,const scene_t &s,
				const surfacePoint_t sp, const vector3d_t &eye) const;
		// has no position, return origin
		virtual point3d_t position() const { return point3d_t(0, 0, 0); };
		virtual void init(scene_t &scene);
		virtual void postInit(scene_t &scene);
		virtual ~pathLight_t();

		static light_t *factory(paramMap_t &params,renderEnvironment_t &render);
		static pluginInfo_t info();
		
	protected:
		
		int samples, grid;	// number of samples, sqrt of samples
		PFLOAT gridiv, sampdiv;	// reciprocal of gridside & samples
		CFLOAT power;
		int maxdepth;
		int maxcausdepth;
		bool use_QMC;
		Halton* HSEQ;
		color_t takeSample(renderState_t &state,const vector3d_t &N,const surfacePoint_t &sp,
											const scene_t &sc,PFLOAT &avgD,PFLOAT &minD,bool caching=false)const;
		CFLOAT weight(const pathSample_t &sample,const point3d_t &P,
				const vector3d_t &N,CFLOAT avgOclu=0.0)const;
		CFLOAT weightNoPrec(const pathSample_t &sample,const point3d_t &P,
				const vector3d_t &N,CFLOAT avgOclu=0.0)const;
		CFLOAT weightNoDist(const pathSample_t &sample,const point3d_t &P,
				const vector3d_t &N,CFLOAT avgOclu=0.0)const;
		CFLOAT weightNoDev(const pathSample_t &sample,const point3d_t &P,
				const vector3d_t &N,CFLOAT avgOclu=0.0)const;
		CFLOAT gatherSamples(const point3d_t &P,const point3d_t &pP,
				const vector3d_t &N,std::vector<foundSample_t> &found,
				unsigned int K,PFLOAT &radius,unsigned int minimun=1,
				CFLOAT (pathLight_t::*W)(const pathSample_t &,const point3d_t &,
					const vector3d_t &,CFLOAT)const=&pathLight_t::weight )const;
		CFLOAT gatherSamples(const point3d_t &P,const point3d_t &pP,
				const vector3d_t &N,std::vector<foundSample_t> &found,PFLOAT radius,
				CFLOAT (pathLight_t::*W)(const pathSample_t &,const point3d_t &,
			          const vector3d_t &,CFLOAT)const)const;
		//bool couldBeUseful(const vector3d_t &N,const point3d_t &P,
		//		const point3d_t &rP,const pathAccum_t *a)const;
		bool couldBeUseful(const vector3d_t &N,const point3d_t &P,
				const point3d_t &rP,int cx,int cy,int cz)const;
		void setIrradiance(pathSample_t &sample,PFLOAT &radius);

		color_t getLight(renderState_t &state,const surfacePoint_t &sp,
				const scene_t &sc,const vector3d_t &eye,photonData_t *data)const;
		void computeGradients();
		CFLOAT getGradient(const pathAccum_t &acc);
		bool testRefinement(const scene_t &sc);

		hemiSampler_t *getSampler(renderState_t &state,const scene_t &sc)const;
		photonData_t *getPhotonData(renderState_t &state)const;

		bool cache;
		PFLOAT cache_size,half_cache,dist_to_sample;
		gBoundTreeNode_t<const pathSample_t *> *tree;
		PFLOAT angle_threshold,shadow_threshold;
		bool recalculate,direct,show_samples,use_gradient;
		int search,gridsize;
		float lastRadius;
		hemiSampler_t *_sampler;
		const globalPhotonMap_t *pmap;
		const globalPhotonMap_t *imap;
		const globalPhotonLight_t::irHash_t *irhash;
		photonData_t *photonData;
		CFLOAT threshold,devaluated;
		std::vector<foundSample_t> stsamples;
};

__END_YAFRAY

#endif
