/****************************************************************************
 *
 * 			mesh.cc: Mesh object implementation 
 *      This is part of the yafray package
 *      Copyright (C) 2002  Alejandro Conty Estvez
 *
 *      This library is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU Lesser General Public
 *      License as published by the Free Software Foundation; either
 *      version 2.1 of the License, or (at your option) any later version.
 *
 *      This library is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *      Lesser General Public License for more details.
 *
 *      You should have received a copy of the GNU Lesser General Public
 *      License along with this library; if not, write to the Free Software
 *      Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

extern int pcount;

#include "mesh.h"
using namespace std;
#include <iostream>

#define BOX_MIN 0.00001
//#include <stdio.h>
//
__BEGIN_YAFRAY

bound_t face_calc_bound(const vector<triangle_t *> &v)
{
	int size=v.size();
	if(size==0) return bound_t(point3d_t(),point3d_t());
	PFLOAT maxx,maxy,maxz,minx,miny,minz;
	maxx=minx=v[0]->a->x;
	maxy=miny=v[0]->a->y;
	maxz=minz=v[0]->a->z;
	for(int i=0;i<size;++i)
	{
		point3d_t p=*(v[i]->a);
		if(p.x>maxx) maxx=p.x;
		if(p.y>maxy) maxy=p.y;
		if(p.z>maxz) maxz=p.z;
		if(p.x<minx) minx=p.x;
		if(p.y<miny) miny=p.y;
		if(p.z<minz) minz=p.z;
		p=*(v[i]->b);
		if(p.x>maxx) maxx=p.x;
		if(p.y>maxy) maxy=p.y;
		if(p.z>maxz) maxz=p.z;
		if(p.x<minx) minx=p.x;
		if(p.y<miny) miny=p.y;
		if(p.z<minz) minz=p.z;
		p=*(v[i]->c);
		if(p.x>maxx) maxx=p.x;
		if(p.y>maxy) maxy=p.y;
		if(p.z>maxz) maxz=p.z;
		if(p.x<minx) minx=p.x;
		if(p.y<miny) miny=p.y;
		if(p.z<minz) minz=p.z;
	}
	minx-=BOX_MIN;
	miny-=BOX_MIN;
	minz-=BOX_MIN;
	maxx+=BOX_MIN;
	maxy+=BOX_MIN;
	maxz+=BOX_MIN;
	return bound_t(point3d_t(minx,miny,minz),point3d_t(maxx,maxy,maxz));
}

bool face_is_in_bound(triangle_t * const & t,bound_t &b)
{
	if(b.includes(*(t->a))) return true;
	if(b.includes(*(t->b))) return true;
	if(b.includes(*(t->c))) return true;
	return false;
}

point3d_t face_get_pos(triangle_t * const & t)
{
	point3d_t r=*(t->a);
	r=r+ *(t->b);
	r=r+ *(t->c);
	r=r/3;
	return r;
}


meshObject_t::meshObject_t(vector<point3d_t> *ver, vector<vector3d_t> *nor,
				vector<triangle_t> *ts, vector<GFLOAT> *fuv, vector<CFLOAT> *fvcol)
{
	vertices=ver;
	normals=nor;
	triangles=ts;
	unt=true;
	hasorco=false;
	if ( (ver==NULL) || (ts==NULL))
		cout<<"Error null mesh\n";
	shader=NULL;
	if(ver!=NULL) recalcBound();
	facesuv = fuv;
	faces_vcol = fvcol;
	vector<triangle_t *> ltri(ts->size());
	for(vector<triangle_t>::iterator i=ts->begin();i!=ts->end();++i)
		ltri[i-ts->begin()]=&(*i);
	tree=buildGenericTree(ltri,face_calc_bound,face_is_in_bound,face_get_pos,10);
}

meshObject_t::meshObject_t(const matrix4x4_t &M, vector<point3d_t> *ver, vector<vector3d_t> *nor,
				vector<triangle_t> *ts, vector<GFLOAT> *fuv, vector<CFLOAT> *fvcol)
{
	vertices=ver;
	normals=nor;
	triangles=ts;
	unt=true;
	hasorco=false;
	if ( (ver==NULL) || (ts==NULL))
		cout<<"Error null mesh\n";
	shader=NULL;
	//if(ver!=NULL) recalcBound();
	facesuv = fuv;
	faces_vcol = fvcol;
	tree=NULL;
	transform(M);
}

void meshObject_t::autoSmooth(PFLOAT angle)
{
	vector<list<triangle_t * > > access(vertices->size());
	int ia,ib,ic;
	for(vector<triangle_t>::iterator ite=triangles->begin();ite!=triangles->end()
			;ite++)
	{
		ia=(*ite).a-(&(*vertices)[0]);
		ib=(*ite).b-(&(*vertices)[0]);
		ic=(*ite).c-(&(*vertices)[0]);
		access[ia].push_back(&(*ite));
		access[ib].push_back(&(*ite));
		access[ic].push_back(&(*ite));
	}
	if (normals!=NULL) delete normals;
	normals= new vector<vector3d_t>(triangles->size()*3);

	PFLOAT cosang;
	if (angle<0) cosang=-2;
	else cosang=cos(angle* (((PFLOAT)2)*M_PI)/((PFLOAT)360));
	vector3d_t norm,na,nb,nc;
	int itri=0;
	for(vector<triangle_t>::iterator ite=triangles->begin();ite!=triangles->end()
			;ite++)
	{
		norm=(*ite).N();
		ia=(*ite).a-(&(*vertices)[0]);
		ib=(*ite).b-(&(*vertices)[0]);
		ic=(*ite).c-(&(*vertices)[0]);
		na=vector3d_t(0,0,0);
		for(list<triangle_t *>::iterator ite2=access[ia].begin();
				ite2!=access[ia].end();ite2++)
		{
			if( (((*ite2)->N())*norm)>cosang ) na=na+(*ite2)->N();
		}
		na.normalize();
		(*normals)[itri]=na;
		(*ite).na=&(*normals)[itri];

		nb=vector3d_t(0,0,0);
		for(list<triangle_t *>::iterator ite2=access[ib].begin();
				ite2!=access[ib].end();ite2++)
		{
			if( (((*ite2)->N())*norm)>cosang ) nb=nb+(*ite2)->N();
		}
		nb.normalize();
		(*normals)[itri+1]=nb;
		(*ite).nb=&(*normals)[itri+1];

		nc=vector3d_t(0,0,0);
		for(list<triangle_t *>::iterator ite2=access[ic].begin();
				ite2!=access[ic].end();ite2++)
		{
			if( (((*ite2)->N())*norm)>cosang ) nc=nc+(*ite2)->N();
		}
		nc.normalize();
		(*normals)[itri+2]=nc;
		(*ite).nc=&(*normals)[itri+2];
		itri+=3;
	}

}

void meshObject_t::recalcBound()
{
	PFLOAT minx,miny,minz;
	PFLOAT maxx,maxy,maxz;

	maxx=minx=(*vertices)[0].x;
	maxy=miny=(*vertices)[0].y;
	maxz=minz=(*vertices)[0].z;
	for(vector<point3d_t>::iterator ite=vertices->begin();ite!=vertices->end();
			ite++)
	{
		if( (*ite).x>maxx ) maxx=(*ite).x;
		if( (*ite).y>maxy ) maxy=(*ite).y;
		if( (*ite).z>maxz ) maxz=(*ite).z;

		if( (*ite).x<minx ) minx=(*ite).x;
		if( (*ite).y<miny ) miny=(*ite).y;
		if( (*ite).z<minz ) minz=(*ite).z;
	}

	maxx+=BOX_MIN;
	maxy+=BOX_MIN;
	maxz+=BOX_MIN;
	minx-=BOX_MIN;
	miny-=BOX_MIN;
	minz-=BOX_MIN;
	bound.set(point3d_t(minx,miny,minz),point3d_t(maxx,maxy,maxz));
}

meshObject_t::~meshObject_t()
{
	if (vertices!=NULL) delete vertices;
	if (normals!=NULL) delete normals;
	if (triangles!=NULL) delete triangles;
	if (facesuv!=NULL) delete facesuv;
	if (faces_vcol!=NULL) delete faces_vcol;
	if (tree!=NULL) delete tree;
}

void meshObject_t::transform(const matrix4x4_t &m)
{
	matrix4x4_t mnotras=m;

	if(!unt)
	{
		if(vertices!=NULL)
			for(vector<point3d_t>::iterator ite=vertices->begin();
					ite!=vertices->end();ite++)
				(*ite)=back*(*ite);
		if(normals!=NULL)
			for(vector<vector3d_t>::iterator ite=normals->begin();
					ite!=normals->end();ite++)
				(*ite)=back*(*ite);
	}

	back=m;
	back.inverse();
	// backRot is rotation only (only 3x3 used of 4x4),
	// so remove scaling by normalizing the matrix vectors.
	// and since inverse needed, use transpose of back directly.
	backRot.identity();
	vector3d_t mv(back[0][0], back[0][1], back[0][2]);
	mv.normalize();
	backRot.setRow(0, mv, 0);
	mv.set(back[1][0], back[1][1], back[1][2]);
	mv.normalize();
	backRot.setRow(1, mv, 0);
	mv.set(back[2][0], back[2][1], back[2][2]);
	mv.normalize();
	backRot.setRow(2, mv, 0);

	// orco matrix backOrco is m*scale derived from bound
	backOrco = m;
	recalcBound();	// need untransformed bound here
	point3d_t p1, p2;
	bound.get(p1, p2);
	mv = p2-p1;
	backOrco.scale(0.5*mv.x, 0.5*mv.y, 0.5*mv.z);

	if(vertices!=NULL)
		for(vector<point3d_t>::iterator ite=vertices->begin();
				ite!=vertices->end();ite++)
			(*ite)=m*(*ite);
	if(normals!=NULL)
		for(vector<vector3d_t>::iterator ite=normals->begin();
				ite!=normals->end();ite++)
			(*ite)=m*(*ite);
	if(triangles!=NULL)
		for(vector<triangle_t>::iterator ite=triangles->begin();
				ite!=triangles->end();ite++)
			(*ite).recNormal();
	unt=false;

	vector<triangle_t *> ltri(triangles->size());
	for(vector<triangle_t>::iterator i=triangles->begin();
			i!=triangles->end();++i)
		ltri[i-triangles->begin()]=&(*i);
	if(tree!=NULL) delete tree;
	tree=buildGenericTree(ltri,face_calc_bound,face_is_in_bound,face_get_pos,4);
	recalcBound();

	// backOrco, replace translation with (transformed!) bound center
	bound.get(p1, p2);
	mv = toVector(0.5*(p1+p2));
	backOrco.setColumn(3, mv, backOrco[3][3]);
	backOrco.inverse();

}


point3d_t meshObject_t::toObject(const point3d_t &p)const
{
	if(unt) return p;
	return (back*p);
}


vector3d_t meshObject_t::toObjectRot(const vector3d_t &v) const
{
	if (unt) return v;
	return (backRot*v);
}


point3d_t meshObject_t::toObjectOrco(const point3d_t &p) const
{
	if (unt) return p;
	return (backOrco*p);
}



bool meshObject_t::shoot(renderState_t &state,surfacePoint_t &where,
		const point3d_t &from, const vector3d_t &ray,bool shadow,PFLOAT dis) const
{
	triangle_t *hitt=NULL;
	object3d_t * lasto=state.lastobject;
	gBoundTreeNode_t<triangle_t*> * lastb=
		(gBoundTreeNode_t<triangle_t*> *)state.lastobjectelement;
	mray_t mray;
	mray.from=from;
	mray.ray=ray;
	PFLOAT minZ=-1;

	if(shadow && (lasto==(object3d_t *)this) && (lastb!=NULL))
	{
		vector<triangle_t *>::const_iterator i=lastb->begin();
		vector<triangle_t *>::const_iterator end=lastb->end();
		for(;i!=end;++i)
		{
			if( (*i)->hit(from,ray))
			{
				PFLOAT Z=(*i)->intersect(from,ray);
				if((Z>0) && ((Z<dis) || (dis<0))) return true;
			}
		}
	}
	gObjectIterator_t<triangle_t *,mray_t,rayCross_f> ite(tree,mray);
	for(;!ite;++ite)
	{
		//pcount++;
		if(hitt==*ite) continue;
		if( (*ite)->hit(from,ray))
		{
			PFLOAT Z=(*ite)->intersect(from,ray);
			if(shadow && (Z>0) && ((Z<dis) || (dis<0)))
				{state.lastobjectelement=(void *)ite.currentNode();return true;}
			else
			if( ((minZ<0) && (Z>0)) || ( (Z>0) && (Z<minZ) ))
			{
				minZ=Z;
				hitt=(*ite);
			}
		}
	}
	if(shadow) return false;
	if(hitt==NULL) return false;
	point3d_t h=from+minZ*ray;
	surfacePoint_t temp=hitt->getSurface(h,minZ,hasorco);
	temp.setObject((object3d_t *)this);
	if(temp.getShader()==NULL) temp.setShader(shader);
	where=temp;
	return true;
}

__END_YAFRAY
