/****************************************************************************
 *
 * 			scene.h: Scene manipulation and rendering api 
 *      This is part of the yafray package
 *      Copyright (C) 2002 Alejandro Conty Estevez
 *
 *      This library is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU Lesser General Public
 *      License as published by the Free Software Foundation; either
 *      version 2.1 of the License, or (at your option) any later version.
 *
 *      This library is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *      Lesser General Public License for more details.
 *
 *      You should have received a copy of the GNU Lesser General Public
 *      License along with this library; if not, write to the Free Software
 *      Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *      
 */
#ifndef __SCENE_H
#define __SCENE_H

#ifdef HAVE_CONFIG_H
#include<config.h>
#endif

#if HAVE_PTHREAD
#include<pthread.h>
#include <semaphore.h>
#include<map>
#endif

#include <vector>
#include <list>

#include "tools.h"

__BEGIN_YAFRAY
class scene_t;
class object3d_t;

struct renderState_t
{
	renderState_t() :raylevel(0),depth(0),contribution(1.0),lastobject(NULL)
		,lastobjectelement(NULL),currentPass(0),rayDivision(1),traveled(0) {};
	int raylevel;
	CFLOAT depth;
	CFLOAT contribution;
	object3d_t * lastobject;
	void * lastobjectelement;
	int currentPass;
	int rayDivision;
	context_t context;
	PFLOAT traveled;
};

__END_YAFRAY
#include "object3d.h"
#include "vector3d.h"
#include "camera.h"
#include "output.h"
#include "light.h"
#include "buffer.h"
#include "filter.h"
#include "bound.h"
#include "background.h"
#include "mcqmc.h"

#define STEPS 40

#ifdef FORCE_FORKS
  #include <sys/types.h>
  #include <sys/wait.h>
  #include <stdio.h>
  #include <unistd.h>
#endif


__BEGIN_YAFRAY
class scene_t
{
	public:
		scene_t();
		~scene_t();
		void setCamera(camera_t *cam);
		void addObject(object3d_t *obj) {obj_list.push_back(obj);};
		void addLight(light_t *light) {light_list.push_back(light);};
		void addFilter(filter_t *filter) {filter_list.push_back(filter);};
		color_t raytrace(renderState_t &state,const point3d_t &from,
				const vector3d_t & ray)const;
		color_t raytrace_indir(renderState_t &state,const point3d_t &from, 
				const vector3d_t & ray, bool isdif=false) const;
		bool isShadowed(renderState_t &state,const surfacePoint_t &p,
				const point3d_t &l)const;
		bool isShadowed(renderState_t &state,const surfacePoint_t &p,
				const vector3d_t &dir)const;
		void setupLights();
		void postSetupLights();
		bool checkSampling();

		bool doOnePass(renderState_t &state,
				int thread,std::vector<color_t> &line,
				std::vector<CFLOAT> &dep,
				std::vector<CFLOAT> &alpha,int numline,int pass);
		bool doAllPasses(renderState_t &state,
				int thread,std::vector<color_t> &line,
				std::vector<CFLOAT> &dep,
				std::vector<CFLOAT> &alpha,int numline);
		bool doPass(renderState_t &state,
				int thread,std::vector<color_t> &line,
				std::vector<CFLOAT> &dep,
				std::vector<CFLOAT> &alpha,int numline,int pass)
		{
			if (AA_onepass_max)
				return doAllPasses(state,thread, line, dep, alpha, numline);
			else
				return doOnePass(state,thread, line, dep, alpha, numline, pass);
		};

		void renderPart(colorOutput_t &out, int curpass, int div, int off);
		void render(colorOutput_t &out, int cpus=1);
		void setMaxRayDepth(int a) {maxraylevel=a;};
		int getMaxRayDepth()const {return maxraylevel;};
		bool firstHit(renderState_t &state,surfacePoint_t &sp,const point3d_t &p,
											const vector3d_t &ray,bool shadow=false)const;
		//bool firstHitRad(surfacePoint_t &sp,const point3d_t &p,
		//									const vector3d_t &ray)const;
		color_t light(renderState_t &state,const surfacePoint_t &sp,
				const point3d_t &from, bool indirect=false)const;
		void setBias(PFLOAT b) {self_bias=b;};
		PFLOAT selfBias()const {return self_bias;};

		void setBackground(const background_t *b) {background=b;};
		const color_t  getBackground(const vector3d_t &dir,renderState_t &state)const
			{if(background==NULL) return color_t(0.0); else return (*background)(dir,state);};

		// indirect sampling
		void setIndirect(int indir_sam, PFLOAT indir_pow)
		{
			indirect_grid = int(sqrt((PFLOAT)indir_sam));
			indirect_gdiv = 1.0/(PFLOAT)indirect_grid;
			indirect_gdiv2pi = indirect_gdiv*M_PI*2.0;
			indirect_samples = indir_sam;
			indirect_power = indir_pow/PFLOAT(indir_sam);
		}

		// gamma & exposure
		void setGamma(CFLOAT g) { gamma_R=0.0;  if (g!=0.0) gamma_R=1.0/g; }
		void setExposure(CFLOAT e) { exposure = -e; }
		void adjustColor(color_t &c)const {c.expgam_Adjust(exposure, gamma_R);};

		// fog
		void setFog(CFLOAT dens, const color_t &col) { fog_density=dens;  fog_color=col; }
		void fog_addToCol(CFLOAT depth, color_t &curcol) const;

		// new AA
		void setAASamples(int ps, int ms, PFLOAT pw, PFLOAT th, int spmax)
		{
			AA_passes = ps;
			AA_minsamples = ms;
			AA_samdiv = 1.0/(PFLOAT)ms;
			AA_pixelwidth = pw;
			AA_threshold = th;
			AA_onepass_max  = spmax;
			HSEQ1.setBase(2);
			HSEQ2.setBase(3);
		}

		void setRepeatFirst() {repeatFirst=true;};
		bool getRepeatFirst()const {return repeatFirst;};
		PFLOAT getWorldResolution()const {return world_resolution;};
		point3d_t getCenterOfView()const {return render_camera->position();};

		typedef std::list<light_t *>::iterator light_iterator;
		typedef std::list<light_t *>::const_iterator const_light_iterator;

		light_iterator lightsBegin() {return light_list.begin();};
		const_light_iterator lightsBegin()const {return light_list.begin();};
		light_iterator lightsEnd() {return light_list.end();};
		const_light_iterator lightsEnd()const {return light_list.end();};

		template<class T>
		void publishData(const std::string &key,const T *data) {published[key]=(const void *)data;};
		template<class T>
		void getPublishedData(const std::string &key,const T *&data)const 
		{
			std::map<std::string,const void *>::const_iterator i=published.find(key);
			data=(i==published.end()) ? (const T *)NULL : (const T *)(i->second);
		};

		void setRegion(PFLOAT mix,PFLOAT max,PFLOAT miy, PFLOAT may)
		{ scxmin=mix;scxmax=max;scymin=miy;scymax=may;};
	protected:
		camera_t *render_camera;
		PFLOAT min_raydis;
		PFLOAT world_resolution;
		std::list<object3d_t *> obj_list;
		std::list<light_t *> light_list;
		std::list<filter_t *> filter_list;
		light_t *radio_light;
		int maxraylevel;
		cBuffer_t colorBuffer;
		fBuffer_t ZBuffer;
		fBuffer_t ABuffer;
		Buffer_t<char> oversample;
		boundTree_t *BTree;
		PFLOAT self_bias;
		const background_t *background;
		// indirect lighting parameters
		int indirect_samples, indirect_grid;
		CFLOAT indirect_power;
		PFLOAT indirect_gdiv, indirect_gdiv2pi;
		// exposure and gamma controls
		CFLOAT gamma_R, exposure;
		// simple fog, density & color
		CFLOAT fog_density;
		color_t fog_color;
		// new AA parameters
		int AA_onepass_max;
		int AA_passes, AA_minsamples;
		PFLOAT AA_pixelwidth, AA_threshold, AA_samdiv;
		Halton HSEQ1, HSEQ2;
		// used to keep track of the screen sampling position, for 'win' texmap mode
		point3d_t screenpos;
		PFLOAT scymin,scymax,scxmin,scxmax;
		bool repeatFirst;
		std::map<std::string,const void *> published;
};

__END_YAFRAY

#endif
