/**
 * yateaddressbook.cpp
 * This file is part of the YATE Project http://YATE.null.ro
 *
 * A Qt based universal telephony client
 * Author: Dorin Lazar <lazar@deuromedia.ro>
 *
 * Yet Another Telephony Engine - a fully featured software PBX and IVR
 * Copyright (C) 2004 Null Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "yateaddressbook.hpp"
#include "utils.hpp"

#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <alloca.h>

YateAddress::YateAddress () :
	address(NULL),
	alias(NULL)
{
}

YateAddress::~YateAddress()
{
	delete[] address; address = NULL;
	delete[] alias; alias = NULL;
}

void YateAddress::setAddress (const char *newAddress)
{
	delete[] address;
	address = CCstrdup (newAddress);
}

void YateAddress::setAlias (const char *newAlias)
{
	delete[] alias;
	alias = CCstrdup (newAlias);
}

YateAddressBook::YateAddressBook(const char *filename)
{
	if (!filename) {
		yab_filename = new char[2048];
		snprintf (yab_filename, 2047, "%s/.yateaddressbook", getenv("HOME"));
	}
	else {
		int length = strlen (filename);
		yab_filename = new char[length + 1];
		strcpy (yab_filename, filename);
	}
	loadAddresses ();
}

YateAddressBook::~YateAddressBook ()
{
	store();
	unsigned int n = addresses.size();
	for (unsigned int i=0; i<n ; i++)
		delete addresses[i];
	delete[] yab_filename; yab_filename = NULL;
}

void YateAddressBook::loadAddresses (void)
{
	FILE *fd = fopen (yab_filename, "r");
	if (!fd) {
		perror (yab_filename);
		fd = fopen (yab_filename, "w");
		if (!fd) {
			perror ("Cannot create file");
		}
		else
			fclose (fd);
		return;
	}

	char *buffer = (char *)alloca (2048);

	memset (buffer, 0, 2048);

	while (!feof (fd)) {
		fgets (buffer, 2047, fd);
		int length = strlen (buffer);
		if (buffer[length-1]=='\n')
			buffer[length-1] = '\0';
		char *address;
		if (!(address = strchr (buffer, '=')))
			continue;
		address[0] = '\0';
		address++;
		if (strlen (buffer)==0 || strlen (address)==0) /* invalid line, discarding */
			continue;
		YateAddress *newAddress = new YateAddress;
		newAddress->setAddress (address);
		newAddress->setAlias (buffer);
		addresses.push_back (newAddress);
	}
	fclose (fd);
}

bool YateAddressBook::store (void)
{
	FILE *fd = fopen (yab_filename, "w");
	if (!fd) {
		perror (yab_filename);
		return false;
	}

	unsigned int n = addresses.size();
	for (unsigned int i=0; i<n; i++) {
		fprintf (fd, "%s=%s\n", addresses[i]->getAlias(), addresses[i]->getAddress());
	}
	fclose (fd);
	return true;
}

void YateAddressBook::addAddress (const char *alias, const char *address)
{
	YateAddress *newAddress = new YateAddress;
	newAddress->setAddress (address);
	newAddress->setAlias (alias);
	addresses.push_back (newAddress);
}
