/* hsrp.h
 * Definitions for Cisco Hot Standby Router Protocol
 *
 * $Id: hsrp.h,v 1.4 2005/07/29 20:37:21 t0mac Exp $ 
 *
 * Yersinia
 * By David Barroso <tomac@wasahero.org> and Alfredo Andres <slay@wasahero.org>
 * Copyright 2005 Alfredo Andres and David Barroso
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef __HSRP_H__
#define __HSRP_H__

#include <libnet.h>

#include "terminal-defs.h"
#include "interfaces.h"

#define HSRP_PACKET_SIZE 20

/* Version */
#define HSRP_VERSION 0x0

/* Opcode */
#define HSRP_TYPE_HELLO 0x0
#define HSRP_TYPE_COUP 0x1
#define HSRP_TYPE_RESIGN 0x2

/* State */
#define HSRP_STATE_INITIAL 0x0
#define HSRP_STATE_LEARN   0x1
#define HSRP_STATE_LISTEN  0x2
#define HSRP_STATE_SPEAK   0x4
#define HSRP_STATE_STANDBY 0x8
#define HSRP_STATE_ACTIVE  0x10

#define HSRP_AUTHDATA_LENGTH  8

/*
 *  HSRP header
 *  Static header size: 20 bytes
 */
struct hsrp_data
{
    u_int8_t version;           /* Version of the HSRP messages */
    u_int8_t opcode;            /* Type of message */
    u_int8_t state;            /* Current state of the router */
    u_int8_t hello_time;       /* Period in seconds between hello messages */
    u_int8_t hold_time;        /* Seconds that the current hello message is valid */
    u_int8_t priority;         /* Priority for the election proccess */
    u_int8_t group;            /* Standby group */
    u_int8_t reserved;         /* Reserved field */
    char authdata[HSRP_AUTHDATA_LENGTH]; /* Password */
    u_int32_t virtual_ip;      /* Virtual IP address */
    /* UDP Data */
    u_int16_t sport;
    u_int16_t dport;
    /* IP Data */
    u_int32_t sip;
    u_int32_t dip;
    /* Ethernet Data */
    u_int8_t mac_source[ETHER_ADDR_LEN];
    u_int8_t mac_dest[ETHER_ADDR_LEN];
};

#define HSRP_TOTAL_FIELDS 17

#define HSRP_SMAC        0
#define HSRP_DMAC        1
#define HSRP_SIP         2
#define HSRP_DIP         3
#define HSRP_SPORT       4
#define HSRP_DPORT       5
#define HSRP_VER         6
#define HSRP_OPCODE      7 
#define HSRP_STATE       8
#define HSRP_HELLO_TIME  9
#define HSRP_HOLD_TIME   10
#define HSRP_PRIORITY    11
#define HSRP_GROUP       12
#define HSRP_RESERVED    13
#define HSRP_AUTHDATA    14
#define HSRP_VIRTUALIP   15

/* Default values */
#define HSRP_DFL_VERSION    HSRP_VERSION
#define HSRP_DFL_TYPE       HSRP_TYPE_HELLO
#define HSRP_DFL_STATE      HSRP_STATE_INITIAL
#define HSRP_DFL_HELLO_TIME 3
#define HSRP_DFL_HOLD_TIME  10
#define HSRP_DFL_PRIORITY   0xFF
#define HSRP_DFL_GROUP      0x00
#define HSRP_DFL_RESERVED   0x00
#define HSRP_DFL_AUTHDATA   "cisco"
#define HSRP_DFL_PORT       1985

static const struct tuple_type_desc hsrp_opcode[] = {
    { HSRP_TYPE_HELLO, "(HELLO)" },
    { HSRP_TYPE_COUP, "(COUP)" },
    { HSRP_TYPE_RESIGN, "(RESIGN)" },
    { 0, NULL }
};

static const struct tuple_type_desc hsrp_state[] = {
    { HSRP_STATE_INITIAL, "(INITIAL)" },
    { HSRP_STATE_LEARN, "(LEARN)" },
    { HSRP_STATE_LISTEN, "(LISTEN)" },
    { HSRP_STATE_SPEAK, "(SPEAK)" },
    { HSRP_STATE_STANDBY, "(STANDBY)" },
    { HSRP_STATE_ACTIVE, "(ACTIVE)" },
    { 0, NULL }
};

static struct proto_features hsrp_features[] = {
    { F_UDP_PORT, HSRP_DFL_PORT },
    { -1, 0 }
};


static struct position_fields hsrp_fields[] = {
   { HSRP_SMAC,       "Source MAC",      1,  1, 12, 17, 0, 0, FIELD_MAC, FIELD_FIRST, NULL},
   { HSRP_DMAC,       "Destination MAC", 1, 46, 62, 17, 0, 0, FIELD_MAC, FIELD_NORMAL, NULL},
   { HSRP_SIP,        "SIP",             2,  1,  5, 15, 1, 1, FIELD_IP, FIELD_NORMAL, NULL},
   { HSRP_DIP,        "DIP",             2, 22, 26, 15, 1, 17, FIELD_IP, FIELD_NORMAL, NULL},
   { HSRP_SPORT,      "SPort",           2, 43, 49, 5 , 0, 0, FIELD_DEC, FIELD_NORMAL, NULL},
   { HSRP_DPORT,      "DPort",           2, 56, 62, 5 , 0, 0, FIELD_DEC, FIELD_NORMAL, NULL},
   { HSRP_VER,        "Version",         3,  1,  9, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, NULL},
   { HSRP_OPCODE,     "Opcode",          3, 13, 20, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, hsrp_opcode},
   { HSRP_STATE,      "State",           3, 24, 30, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, hsrp_state},
   { HSRP_HELLO_TIME, "Hello",           3, 34, 40, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, NULL},
   { HSRP_HOLD_TIME,  "Hold",            3, 44, 49, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, NULL},
   { HSRP_PRIORITY,   "Priority",        3, 53, 62, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, NULL},
   { HSRP_GROUP,      "Group",           4,  1,  7, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, NULL},
   { HSRP_RESERVED,   "Reserved",        4, 11, 20, 2 , 0, 0, FIELD_HEX, FIELD_NORMAL, NULL},
   { HSRP_AUTHDATA,   "Auth",            4, 24, 29, 8 , 1, 49, FIELD_STR, FIELD_NORMAL, NULL},
   { HSRP_VIRTUALIP,  "VIP",             4, 39, 43, 15, 1, 33, FIELD_IP, FIELD_LAST, NULL},
   { 0,               NULL,              0,  0,  0, 0 , 0, 0, FIELD_NONE, FIELD_NORMAL, NULL}
};

/* true size + 1 extra element for '\0' */
struct hsrp_printable { /* HSRP and Ethernet fields*/
    u_int8_t version[3];
    u_int8_t opcode[3];
    u_int8_t state[3];
    u_int8_t hello_time[3];
    u_int8_t hold_time[3];
    u_int8_t priority[3];
    u_int8_t group[3];
    u_int8_t reserved[3];
    u_int8_t authdata[9];
    u_int8_t virtual_ip[16];
    /* UDP Data */
    u_int8_t sport[6];
    u_int8_t dport[6];
    /* IP Data */
    u_int8_t sip[16];
    u_int8_t dip[16];
    /* Ethernet Data */
    u_int8_t smac[18];
    u_int8_t dmac[18];
};

/* Attacks */
#define HSRP_ATTACK_SEND_RAW      0
#define HSRP_ATTACK_BECOME_ACTIVE 1
#define HSRP_ATTACK_MITM_BECOME_ACTIVE 2

#define HSRP_SOURCE_IP   0
static struct attack_param hsrp_active_params[] = {
  { NULL, "Source IP",    4, 0,    0, FIELD_IP,       15, NULL },
};

void hsrp_th_send_raw(void *);
void hsrp_th_send_raw_exit(struct attacks *);
void hsrp_th_become_active(void *);
void hsrp_th_become_active_exit(struct attacks *);

static struct attack hsrp_attack[] = {
  { HSRP_ATTACK_SEND_RAW,     "sending raw HSRP packet", NONDOS, SINGLE,    hsrp_th_send_raw, NULL, 0   },
  { HSRP_ATTACK_BECOME_ACTIVE, "becoming ACTIVE router", NONDOS, CONTINOUS, hsrp_th_become_active, hsrp_active_params, 
  SIZE_ARRAY(hsrp_active_params)    },
  { HSRP_ATTACK_MITM_BECOME_ACTIVE, "becoming ACTIVE router (MITM)", NONDOS, CONTINOUS, hsrp_th_become_active, NULL, 0   },
  { 0,                   NULL,                 0,      0,     NULL, NULL, 0                   }
};

void     hsrp_register(void);
void     hsrp_send_hellos(void *);
int8_t   hsrp_send_packet(struct attacks *);
char     **hsrp_get_printable_packet(struct pcap_data *);
char     **hsrp_get_printable_store(struct term_node *);
int8_t   hsrp_learn_packet(int8_t, u_int8_t *, void *, struct pcap_pkthdr *, struct pcap_data *);
int8_t   hsrp_load_values(struct pcap_data *, void *);
int8_t   hsrp_init_attribs(struct term_node *);
char     *hsrp_get_info(u_int8_t, char *);
int8_t   hsrp_update_field(int8_t, struct term_node *, void *);
int8_t   hsrp_update_data(int8_t, int8_t, int8_t, struct term_node *);
void     hsrp_help(void);
int8_t   hsrp_parser( struct term_node *, int8_t, char **, void *);

extern void   thread_libnet_error(char *, libnet_t *);
extern int8_t parser_vrfy_bridge_id(char *, u_int8_t * );
extern int8_t thread_create(pthread_t *, void *, void *);
extern void   write_log( u_int16_t mode, char *msg, ... );
extern int8_t attack_th_exit(struct attacks *);
extern void   attack_gen_mac(u_int8_t *);
extern int8_t interfaces_get_packet(u_int8_t *, struct pcap_pkthdr *, u_int8_t *, u_int16_t, time_t);
extern int8_t parser_vrfy_mac(char *, u_int8_t *);
extern int8_t parser_get_formated_inet_address(u_int32_t, char *, u_int16_t);
extern int8_t parser_get_inet_aton(char *, struct in_addr *);
extern int8_t parser_command2index(register const struct attack *, register int8_t);

extern struct interface_data interfaces[];
extern struct terminals *terms;
extern int8_t bin_data[];

#endif
/* vim:set tabstop=4:set expandtab:set shiftwidth=4:set textwidth=78: */
