/*
 * YH - Console Chinese Environment -
 * Copyright (C) 1999 Red Flag Linux (office@sonata.iscas.ac.cn)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */


/*****************---pimi.c----*********************/
/*
 * Platform Input Method Interface
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "imcommon.h"
#include "pimi.h"

#ifdef linux
#define MAXNIM		5
#else
#define MAXNIM		9
#endif

#define MAXIMNAMELEN	16

char	extcode[64];
char	cand[128];
char	result[128];

static int	commpipe[2];
static int	sktListen;
static pid_t	pid;

static int	curim;
static char	nameary[MAXNIM][MAXIMNAMELEN];
static char	ipary[MAXNIM][16];	
static int	portary[MAXNIM];
static int	sktary[MAXNIM];

static int	GetIMResponse(int idx);
static void	catch_usr1();				/* in parent process */
static void	catch_term();				/* in child process */


void	PIMIInit(port_fd)
int		port_fd;
{
	int			i;
	struct sockaddr_in	sockAddr;
	struct sigaction	act;

	for (i = 0; i < MAXNIM; i++)
		sktary[i] = -1;
	curim = -1;
	extcode[0] = '\0';
	cand[0] = '\0';
	result[0] = '\0';

	act.sa_handler = catch_usr1;
#if (defined(OPENSERVER) || defined(GEMINI))
	act.sa_sigaction = catch_usr1;
#endif
	act.sa_flags = 0;
	memset(&act.sa_mask, 0, sizeof(act.sa_mask));
	sigaction(SIGUSR1, &act, NULL);
	pipe(commpipe);

	sktListen = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	sockAddr.sin_family = PF_INET;
	sockAddr.sin_addr.s_addr = INADDR_ANY;
	sockAddr.sin_port = htons(0);
	bind(sktListen, (struct sockaddr *) &sockAddr, sizeof(sockAddr));
	listen(sktListen, 8);

	if ((pid = fork()) != 0)
	{
		close(port_fd);
		close(commpipe[1]);
	}
	else
	{
		pid_t			ppid;
#ifdef OPENSERVER
		int			len;
#else
		size_t			len;
#endif
		int			port;
		struct sigaction	act;

		act.sa_handler = catch_term;
#if (defined(OPENSERVER) || defined(GEMINI))
		act.sa_sigaction = catch_term;
#endif
		act.sa_flags = 0;
		memset(&act.sa_mask, 0, sizeof(act.sa_mask));
		sigaction(SIGTERM, &act, NULL);

		setpgrp();

		len = sizeof(sockAddr);
		getsockname(sktListen, (struct sockaddr *) &sockAddr, &len);
		port = ntohs(sockAddr.sin_port);

		close(commpipe[0]);
		ppid = getppid();

		write(port_fd, &port, sizeof(port));
		close(port_fd);

		while (1)
		{
			int		skt;
			char	*buf;

			len = sizeof(sockAddr);
			skt = accept(sktListen, (struct sockaddr *) &sockAddr, &len);
			buf = getresponse(skt, CMD_END);
			close(skt);
			write(commpipe[1], buf, strlen(buf));
			kill(ppid, SIGUSR1);
		}
	}
}


void	PIMIClearup()
{
	close(sktListen);
	close(commpipe[0]);

	for (curim = 0; curim < MAXNIM; curim++)
		PIMIQuit();

	kill(pid, SIGTERM);
}


void	PIMIQuit()
{
	if ((curim != -1) && (sktary[curim] != -1))
	{
		char	buf[128];

		sprintf(buf, "QUIT%s", CMD_END);
		send(sktary[curim], buf, strlen(buf), 0);
		close(sktary[curim]);
		sktary[curim] = -1;
	}
}


char	*PIMISelectIM(idx)
int	idx;
{
	char	*outbuf0 = "SELECT 0\r\n";
	char	*outbuf1 = "SELECT 1\r\n";
	char	*inbuf;

	if ((idx < 0) || (idx >= MAXNIM))
		return	NULL;

	if (curim != -1)
	{
		if (send(sktary[curim], outbuf0, strlen(outbuf0), 0) < 0)
			sktary[curim] = -1;
		else
		{
			inbuf = getresponse(sktary[curim], RESPONSE_END);
			if (inbuf == NULL)
				sktary[curim] = -1;
		}

		curim = -1;
	}

	if (sktary[idx] == -1)
		return NULL;

	if (send(sktary[idx], outbuf1, strlen(outbuf1), 0) < 0)
	{
		sktary[idx] = -1;
		return	NULL;
	}

	if (GetIMResponse(idx) == -1)
		return	NULL;

	return	nameary[curim = idx];
}


int	PIMIFilter(ch)
int	ch;
{
	char	outbuf[64];

	if ((curim == -1) || (sktary[curim] == -1))
	{
		PIMIClear();
		return	-1;
	}

	sprintf(outbuf, "FILTER %d\r\n", ch);
	if (send(sktary[curim], outbuf, strlen(outbuf), 0) < 0)
	{
		sktary[curim] = -1;
		return	-1;
	}
	
	return	GetIMResponse(curim);
}


static int	GetIMResponse(idx)
int	idx;
{
	char	*inbuf;
	char	*p, *q;

	inbuf = getresponse(sktary[idx], RESPONSE_END);

	if (inbuf == NULL)
	{
		sktary[curim] = -1;
		goto	onerror;
	}
	
	p = strchr(inbuf, '\r');
	if (p == NULL)
		goto	onerror;
	*p = '\0';
	if (memcmp(inbuf, "+OK", 3) != 0)
		goto	onerror;
	
	p += 2;
	q = strchr(p, '\r');
	if (q == NULL)
		goto	onerror;
	*q = '\0';
	strcpy(extcode, p);

	p = q + 2;
	q = strchr(p, '\r');
	if (q == NULL)
		goto	onerror;
	*q = '\0';
	strcpy(cand, p);

	p = q + 2;
	q = strchr(p, '\r');
	if (q == NULL)
		goto	onerror;

	/*
	 * Maybe the result is \r ...
	 */
	while (q[1] != '\n')
		q++;
	*q = '\0';
	strcpy(result, p);

	return	0;

onerror:
	PIMIClear();
	return	-1;
}


void	PIMIClear()
{
	extcode[0] = '\0';
	cand[0] = '\0';
	result[0] = '\0';
}


static void	catch_usr1()
{
	char			buf[512];
	int			n;
	int			b1, b2, b3, b4, b5, b6;
	char			name[MAXIMNAMELEN];
	char			ip[16];
	int			port;
	int			skt;
	struct sockaddr_in	sockAddr;

	n = read(commpipe[0], buf, sizeof(buf));
	if (n < 0)
		return;
	buf[n] = '\0';

	sscanf(buf, "%s%d%d%d%d%d%d", 
		name, &b1, &b2, &b3, &b4, &b5, &b6);
	sprintf(ip, "%d.%d.%d.%d", b1, b2, b3, b4);
	port = (b5 << 8) | b6;

	skt = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	sockAddr.sin_family = PF_INET;
	sockAddr.sin_port = htons(port);
	sockAddr.sin_addr.s_addr = inet_addr(ip);
	if (connect(skt, (struct sockaddr *) &sockAddr,	sizeof(sockAddr)) < 0)
		return;

	for (n = 0; n < MAXNIM; n++)
		if (sktary[n] == -1)
			break;

	if (n == MAXNIM)
	{
		int	oldcurim;

		oldcurim = curim;

		if (curim == 0)
			curim = n = 1;
		else
			curim = n = 0;
		PIMIQuit();

		curim = oldcurim;
	}

	strcpy(nameary[n], name);
	strcpy(ipary[n], ip);
	portary[n] = port;
	sktary[n] = skt;

	sprintf(buf, "Shift+F%d%s", 3 + n, CMD_END);
	send(skt, buf, strlen(buf), 0);
}


static void	catch_term()
{
	exit(0);
}

