#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>

#include "../include/Y2/Y.h"
#include "../include/Y2/Ylib.h"

#include "../include/string.h"

static void print_help(void);


#define MIN(a,b)        ((a) < (b) ? (a) : (b))
#define MAX(a,b)        ((a) > (b) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)     (((x) < 0) ? ((x) * -1) : (x))


/*
 *	Print help.
 */
static void print_help(void)
{
	printf("\
Usage: yaudiocd [track_number] [options]\n\
\n\
    [track_number] specifies the audio CD track number to play.\n\
\n\
    [options] can be any of the following:\n\
\n\
        --stop                       Stop audio CD.\n\
        --eject                      Eject audio CD.\n\
        --recorder <address:port>    Specify which Y server to connect to.\n\
\n\
    Return values:\n\
\n\
        0       Success.\n\
        1       General error.\n\
        2       Unable to connect to the Y server.\n\
        3       Systems error.\n\
\n"
	);
}


int main(int argc, char *argv[])
{
	int i;
	const char *arg;
	int play_track = 0;
	Boolean stop = False;
	Boolean eject = False;
	const char *con_arg = NULL;
	YConnection *con = NULL;

	/* Parse arguments. */
	for(i = 1; i < argc; i++)
	{
	    arg = argv[i];
	    if(arg == NULL)
		continue;

	    /* Help. */
	    if(strcasepfx(arg, "--h") ||
               strcasepfx(arg, "-h") ||
               !strcmp(arg, "?")
	    )
	    {
		print_help();
		return(0);
	    }
	    /* Play track. */
	    else if(strcasepfx(arg, "--p") ||
                    strcasepfx(arg, "-p")
            )
	    {
		i++;
		arg = (i < argc) ? argv[i] : NULL;
		if(arg != NULL)
		{
		    play_track = atoi(arg);
		}
		else
		{
		    fprintf(
			stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
		    );
		}
	    }
            /* Stop. */
            else if(strcasepfx(arg, "--s") ||
                    strcasepfx(arg, "-s")
            )
	    {
		stop = True;
	    }
            /* Eject. */
            else if(strcasepfx(arg, "--e") ||
                    strcasepfx(arg, "-e")
            )
            {
                eject = True;
            }
	    /* Connect address. */
	    else if(strcasepfx(arg, "--rec") ||
                    strcasepfx(arg, "-rec")
	    )
	    {
		i++;
                arg = (i < argc) ? argv[i] : NULL;
                if(arg != NULL)
		{
		    con_arg = arg;
		}
		else
		{
		    fprintf(
			stderr,
			"%s: Requires argument.\n",
			argv[i - 1]
		    );
		}
	    }
	    /* All else check if it is a track number? */
	    else if(isdigit(*arg) || (*arg == '-'))
	    {
		play_track = atoi(arg);
	    }
	}


        /* Connect to Y server. */
        con = YOpenConnection(
            NULL,               /* No start argument. */ 
            con_arg
        );
        if(con == NULL)
        {
            fprintf(stderr, "Unable to connect to the Y server");
            if(con_arg == NULL)
                con_arg = getenv("RECORDER");
            if(con_arg == NULL)
                fprintf(stderr, ".\n");
            else
                fprintf(stderr, ": %s\n", con_arg);
            return(2);
        }


	/* Play track? */
	if(play_track > 0)
	{
	    if(YPlayAudioCDTrack(con, play_track))
	    {
		fprintf(
		    stderr,
		    "Error playing audio CD track %i.\n",
		    play_track
		);
	    }
	    else
	    {
		Boolean got_response = False;
		YEvent event;
		YEventAudioCDStats *acds;

		/* Wait for audio CD play track response. */
		while(YGetNextEvent(con, &event, True) > 0)
		{
		    switch(event.type)
		    {
		      case YAudioCD:
		        acds = &event.audiocdstats;
		        if(acds->minor_op_code == YAudioCDPlayTrack)
			    got_response = True;
		        break;
		      case YDisconnect:
		      case YShutdown:
		        got_response = True;
		        break;
		    }
		    if(got_response)
			break;
		}
	    }
	}
        /* Stop? */
        else if(stop)
        {
            if(YStopAudioCD(con))
            {
                fprintf(
                    stderr,
                    "Error stopping audio CD\n"
                );
            }
            else
            {
                Boolean got_response = False;
                YEvent event;
                YEventAudioCDStats *acds;

                /* Wait for audio CD stop response. */
                while(YGetNextEvent(con, &event, True) > 0)
                {
                    switch(event.type)
                    {
                      case YAudioCD:
                        acds = &event.audiocdstats;
                        if(acds->minor_op_code == YAudioCDStop)
                            got_response = True;
                        break;
                      case YDisconnect:
                      case YShutdown:
                        got_response = True;
                        break;
                    }
                    if(got_response)
                        break;
                }
            }
        }
        /* Eject? */
        else if(eject)
        {
            if(YEjectAudioCD(con))
            {
                fprintf(
                    stderr,
                    "Error ejecting audio CD\n"
                );
            }
            else
            {
                Boolean got_response = False;
                YEvent event;
                YEventAudioCDStats *acds;

                /* Wait for audio CD eject response. */
                while(YGetNextEvent(con, &event, True) > 0)
                {
                    switch(event.type)
                    {
                      case YAudioCD:
                        acds = &event.audiocdstats;
                        if(acds->minor_op_code == YAudioCDEject)
                            got_response = True;
                        break;
                      case YDisconnect:
                      case YShutdown:
                        got_response = True;
                        break;
                    }
                    if(got_response)
                        break;
                }
            }
        }
	else	/* All else assume get audio CD tracks. */
	{
	    /* Get audio CD tracks. */
	    int i, total;
	    YAudioCDTrackStruct *t, **track = YGetAudioCDTracks(
		con, &total
	    );
	    if(track != NULL)
	    {
		if(total > 0)
		    printf("Total Tracks: %i\n", total);
		for(i = 0; i < total; i++)
		{
		    t = track[i];
		    if(t == NULL)
			continue;

		    printf(
"Track %i:\tStart: %.2i:%.2i:%.2i\tLength %.2i:%.2i\t%s\n",
			t->number,
			(int)(t->start_time / 3600),
			(int)(t->start_time / 60),
			(int)(t->start_time % 60),
			(int)(t->length / 60),
			(int)(t->length % 60),
			t->name
		    );
		}
		YFreeAudioCDTracksList(track, total);
	    }
	}

        /* Disconnect from Y server. */
        YCloseConnection(con, False);
        con = NULL;

	return(0);
}
