package Zim::Repository::Man;

use strict;
use File::Spec;
use Zim::Repository::Base;
use Zim::Page::Text;
use Zim::File;

our $VERSION = '0.15';
our @ISA     = 'Zim::Repository::Base';

my $null = File::Spec->devnull;

=head1 NAME

Zim::Repository::Man - Man page repository for zim

=head1 DESCRIPTION

This module can be used to read man pages in L<zim>.
It assumes that you have the GNU man program, if not it fails silently.

It derives from L<Zim::Repository>.

=head1 METHODS

=over 4

=cut

sub init {
	my $self = shift;
	open MAN, "man -w 2> $null |";
	my $path = join '', <MAN>;
	close MAN;
	$self->{path} = [grep length($_), split /:+/, $path];
	#warn "MANPATH: @{$self->{path}}\n";
}

=item C<list_pages(NAMESPACE)>

Lists all manpages in a certain section.

=cut

sub list_pages {
	my ($self, $namespace) = @_;
	$namespace =~ s/^$self->{namespace}:*//;
	return $self->list_sections unless length $namespace;
	return unless $namespace =~ /^(\d+\w*):*$/;
	my $section = $1;
	my @pages;
	for (@{$self->{path}}) {
		my $dir = File::Spec->catdir($_, "man$section");
		next unless -d $dir;
		#warn "Listing man pages in $dir\n";
		push @pages, map {s/\..*$//; $_} Zim::File->list_dir($dir);
	}
	$self->wipe_array(\@pages);
	return @pages;
}

=item C<list_sections()>

Used by C<list_pages()> when no section is given.

=cut

sub list_sections {
	my $self = shift;
	my @sections;
	for my $dir (@{$self->{path}}) {
		next unless -d $dir;
		#warn "Listing man sections in $dir\n";
		push @sections, grep s/^man(\d+\w*)/$1/, Zim::File->list_dir($dir);
	}
	$self->wipe_array(\@sections);
	return map "$_:", @sections;
}

=item C<get_page(NAME)>

Returns a L<Zim::Page::Text> object for man page NAME.

=cut

sub get_page {
	my ($self, $name) = @_;
	#warn "Get man page: $name\n";
	
	$name =~ s/^$self->{namespace}:*//;
	$name =~ s/^(\d+\w*):+//;
	my $sect = $1 || '';
	
	$ENV{MANWIDTH} = 80; # FIXME get window size (via Env ?)
	open MAN, "man -c $sect $name 2> $null |" or return undef;
	my ($block, @data);
	while (<MAN>) {
		# FIXME implement parsing algo like in Zim.pm
		# include bold and head2
		#s/((\S\cH\S)+)/<b>$1<\/b>/g;
		chomp;
		s/.\cH//g;
		if (/^[A-Z]+[A-Z\s]*$/) { # heading
			push @data, $block if length $block;
			push @data, ['head1', {}, $_];
			$block = '';
		}
		elsif (/\b[\w\:\.\-]+\(\w+\)/) { # links
			# FIXME namespace links per man section
			push @data, $block if length $block;
			while (s/(.*?)\b([\w\:\.\-]+\(\d+\w*\))//) {
				push @data, $1 if length($1);
				push @data, ['link', {to => $2}, $2];
			}
			$block = $_ . "\n";
		}
		else { $block .= $_ . "\n" }
	}
	push @data, $block if length $block;
	close MAN;
	return undef unless @data;
	
	$name = $self->{namespace}.($sect ? $sect.':' : '').$name;
	my $page = Zim::Page::Text->new($self, $name);
	$page->set_parse_tree(['Document', {}, @data]);
	
	return $page;
}

=item C<resolve_link(PAGE, LINK)>

Stub that calls C<resolve_page()>.

Man pages do not contain relative links or links to pages in other namespaces.
They can only link to other man pages.

=cut

sub resolve_link { $_[0]->resolve_page($_[2]) }

=item C<resolve_page(NAME)>

Case in-sensitive check whether a page exists or not.

=cut

sub resolve_page {
	my ($self, $name) = @_;
	$name = ':'.$name;
	$name =~ s/^:*$self->{namespace}:*(?:(\d+\w*):+)?//i;
	$name =~ s/^://;
	my $sect = lc($1) || '';
	$sect = lc($1) if $name =~ s/\((\d+\w*)\):*$//;

	#warn "Resolving $name in section $sect\n";
	open MAN, "man -w $sect $name 2> $null |";
	my $path = join '', <MAN>;
	close MAN;

	$path = undef unless $path =~ /\S/;
	$sect = lc($1) if ! $sect and $path =~ /man(\d+\w*)\W/;
	
	$name = $self->{namespace}.($sect ? $sect.':' : '').$name;
	return $path ? $self->get_page($name)        : ($name =~ /[A-Z]/)
	             ? $self->resolve_page(lc $name) : undef              ;

}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim::Repository>

=cut

