/*
    libzint - the open source barcode library
    Copyright (C) 2008-2019 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* vim: set ts=4 sw=4 et : */

#include "testcommon.h"

//#define TEST_RSS_BINARY_DIV_MODULO_DIVISOR_GENERATE_EXPECTED 1
//#define TEST_EXAMPLES_GENERATE_EXPECTED 1
//#define TEST_GENERAL_FIELD_GENERATE_EXPECTED 1
//#define TEST_BINARY_BUFFER_SIZE_GENERATE_EXPECTED 1

static void test_binary_div_modulo_divisor(void)
{
    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char* data;
        int ret_encode;
        float w;
        float h;
        int ret_vector;

        int expected_rows;
        int expected_width;
        unsigned char* expected;
    };
    struct item data[] = {
        /* 0*/ { BARCODE_RSS14, "1234567890123", 0, 100, 30, 0, 1, 96, "010111010010000001001110000000010100001011111010110100011001100101111111110001011011000111000101" },
        /* 1*/ { BARCODE_RSS14, "0000004537076", 0, 100, 30, 0, 1, 96, "010101001000000001000100000000010111111100101010111101111101010101111111110111010100100000000101" },
        /* 2*/ { BARCODE_RSS14, "0000004537077", 0, 100, 30, 0, 1, 96, "010101001000000001000111000000010111111101001010101010110000000101111100000111011111111011010101" },
        /* 3*/ { BARCODE_RSS14, "0000004537078", 0, 100, 30, 0, 1, 96, "010101001000000001000111000000010111111101001010101011010000000101111000000011011111111011010101" },
        /* 4*/ { BARCODE_RSS14, "0000000001596", 0, 100, 30, 0, 1, 96, "010101001000000001001111100000010111111100101010111101111101010101111100000111011111111011010101" },
        /* 5*/ { BARCODE_RSS14, "0000000001597", 0, 100, 30, 0, 1, 96, "010101001000000001011111000000010111111100101010101010110000000101111100000111011111110111010101" },
        /* 6*/ { BARCODE_RSS14, "0000000001598", 0, 100, 30, 0, 1, 96, "010101001000000001011111000000010111111100101010101011010000000101111000000011011111110111010101" },
        /* 7*/ { BARCODE_RSS_LTD, "1234567890123", 0, 100, 30, 0, 1, 74, "01001100111100101000100111010110101011001001010010101001010000011100011101" },
        /* 8*/ { BARCODE_RSS_LTD, "0000002013570", 0, 100, 30, 0, 1, 74, "01010101010100000010000001110100101101011001010111111110111111010101010101" },
        /* 9*/ { BARCODE_RSS_LTD, "0000002013571", 0, 100, 30, 0, 1, 74, "01010101010100000011000000110101011010100011010101010101000000100000011101" },
        /*10*/ { BARCODE_RSS_LTD, "0000002013572", 0, 100, 30, 0, 1, 74, "01010101010100000011000000110101010010111001010101010101000000110000001101" },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        int length = strlen(data[i].data);

        ret = ZBarcode_Encode(symbol, data[i].data, length);
        assert_equal(ret, data[i].ret_encode, "i:%d ZBarcode_Encode ret %d != %d\n", i, ret, data[i].ret_encode);

        #ifdef TEST_RSS_BINARY_DIV_MODULO_DIVISOR_GENERATE_EXPECTED
        printf("        /*%2d*/ { %s, \"%s\", %d, %.0f, %.0f, %d, %d, %d, ",
                i, testUtilBarcodeName(data[i].symbology), data[i].data, ret, data[i].w, data[i].h, data[i].ret_vector, symbol->rows, symbol->width);
        testUtilModulesDump(symbol, "", " },\n");
        #else

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d\n", i, symbol->rows, data[i].expected_rows);
        assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d\n", i, symbol->width, data[i].expected_width);

        int width, row;
        ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
        assert_zero(ret, "i:%d testUtilModulesCmp ret %d != 0 width %d row %d\n", i, ret, width, row);

        ret = ZBarcode_Buffer_Vector(symbol, 0);
        assert_equal(ret, data[i].ret_vector, "i:%d ZBarcode_Buffer_Vector ret %d != %d\n", i, ret, data[i].ret_vector);
        #endif

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

// Replicate examples from GS1 General Specifications 19.1 and ISO/IEC 24724:2011
static void test_examples(void)
{
    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char* data;
        int ret;

        int expected_rows;
        int expected_width;
        char* comment;
        unsigned char* expected;
    };
    // Verified manually against GS1 General Specifications 19.1 and ISO/IEC 24724:2011
    struct item data[] = {
        /* 0*/ { BARCODE_RSS14, "0950110153001", 0, 1, 96, "Figure 5.5.2.1.1-1. GS1 DataBar Omnidirectional",
                    "010000010100000101000111110000010111101101011100100011011101000101100000000111001110110111001101"
               },
        /* 1*/ { BARCODE_RSS_EXP, "[01]90614141000015[3202]000150", 0, 1, 151, "Figure 5.5.2.3.1-1. GS1 DataBar Expanded",
                    "0101100011001100001011111111000010100100010000111101110011100010100010111100000011100111010111111011010100000100000110001111110000101000000100011010010"
               },
        /* 2*/ { BARCODE_RSS_EXPSTACK, "[01]90614141000015[3202]000150", 0, 5, 102, "Figure 5.5.2.3.2-1. GS1 DataBar Expanded Stacked",
                    "010110001100110000101111111100001010010001000011110111001110001010001011110000001110011101011111101101"
                    "000001110011001111010000000010100101101110111100001000110001110101100100001010100001100010100000010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001011111011111001010000001010010111111011100100000000000000000000000000000000000000000000000000000"
                    "001010100000100000110001111110000101000000100011010010000000000000000000000000000000000000000000000000"
               },
        /* 3*/ { BARCODE_RSS14, "2001234567890", 0, 1, 96, "24724:2011 Figure 1 — GS1 DataBar Omnidirectional",
                    "010100011101000001001111111000010100110110111110110000010010100101100000000111000110110110001101"
               },
        /* 4*/ { BARCODE_RSS14, "0441234567890", 0, 1, 96, "24724:2011 Figure 2 — GS1 DataBar Omnidirectional",
                    "010010001000010001000111000000010101000001100110101100100100000101111110000011000010100011100101"
               },
        /* 5*/ { BARCODE_RSS14, "0001234567890", 0, 1, 96, "24724:2011 Figure 4 — GS1 DataBar Truncated",
                    "010101001000000001001111111000010111001011011110111001010110000101111111000111001100111101110101"
               },
        /* 6*/ { BARCODE_RSS14STACK, "0001234567890", 0, 3, 50, "24724:2011 Figure 5 — GS1 DataBar Stacked",
                    "01010100100000000100111111100001011100101101111010"
                    "00001010101011111010000000111010100011010010000000"
                    "10111001010110000101111111000111001100111101110101"
               },
        /* 7*/ { BARCODE_RSS14STACK_OMNI, "0003456789012", 0, 5, 50, "24724:2011 Figure 6 — GS1 DataBar Stacked Omnidirectional",
                    "01010100100000000100111110000001010011100110011010"
                    "00001011011111111010000001010100101100011001100000"
                    "00000101010101010101010101010101010101010101010000"
                    "00001000100010111010010101010000111101001101110000"
                    "10110111011101000101100000000111000010110010001101"
               },
        /* 8*/ { BARCODE_RSS_LTD, "1501234567890", 0, 1, 74, "24724:2011 Figure 7 — GS1 DataBar Limited",
                    "01000110011000110110101001110100101011010011010010010110001101110011001101"
               },
        /* 9*/ { BARCODE_RSS_LTD, "0031234567890", 0, 1, 74, "24724:2011 Figure 8 — (a) GS1 DataBar Limited",
                    "01010100000100100010000101110010101101101001010110000010100100101100000101"
               },
        /*10*/ { BARCODE_RSS_EXP, "[01]98898765432106[3202]012345[15]991231", 0, 1, 200, "24724:2011 Figure 10 — GS1 DataBar Expanded",
                    "01001000011000110110111111110000101110000110010100011010000001100010101111110000111010011100000010010100111110111001100011111100001011101100000100100100011110010110001011111111001110001101111010000101"
               },
        /*11*/ { BARCODE_RSS_EXP, "[01]90012345678908[3103]001750", 0, 1, 151, "24724:2011 Figure 11 — GS1 DataBar Expanded",
                    "0101110010000010011011111111000010111000010011000101011110111001100010111100000011100101110001110111011110101111000110001111110000101011000010011111010"
               },
        /*12*/ { BARCODE_RSS_EXPSTACK, "[01]98898765432106[3202]012345[15]991231", 0, 5, 102, "24724:2011 Figure 12 — GS1 DataBar Expanded Stacked symbol",
                    "010010000110001101101111111100001011100001100101000110100000011000101011111100001110100111000000100101"
                    "000001111001110010010000000010100100011110011010111001011111100111010100000010100001011000111111010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011101000010011100001000000001011100101100001110110110111110010001001010000001010011000100000110000"
                    "101000010111101100011100111111110100011010011110001001001000001101110100001111110001100111011111001010"
               },
        /*13*/ { BARCODE_RSS_EXPSTACK, "[01]95012345678903[3103]000123", 0, 5, 102, "24724:2011 Figure 13 — GS1 DataBar Expanded Stacked",
                    "010100010001111000101111111100001010111000001100010111000110001001101011110000001110010111000111011101"
                    "000011101110000111010000000010100101000111110011101000111001110110010100001010100001101000111000100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000001010000111001010000001010010111011011111100000000000000000000000000000000000000000000000000000"
                    "001011110101111000110001111110000101000100100000011010000000000000000000000000000000000000000000000000"
               },
        /*14*/ { BARCODE_RSS_LTD, "0009876543210", 0, 1, 74, "24724:2011 Figure F.2 — GS1 DataBar Limited",
                    "01010100100100110000110000010101101001011001010001000101000100000100100101"
               },
        /*15*/ { BARCODE_RSS_EXP, "[10]12A", 0, 1, 102, "24724:2011 Figure F.3 — GS1 DataBar Expanded",
                    "010100000110100000101111111100001010001000000010110101111100100111001011110000000010011101111111010101"
               },
        /*16*/ { BARCODE_RSS14STACK, "0000000000000", 0, 3, 50, "#183 GS1 DataBar Stacked separator alternation; verified manually against tec-it.com (bwipp differs)",
                    "01010100100000000100011111111001011111110010101010"
                    "00000101011111111010100000001010100000001101010000"
                    "10101010110000000101111111110111011111111011010101"
               },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        int length = strlen(data[i].data);

        ret = ZBarcode_Encode(symbol, data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d %s\n", i, ret, data[i].ret, symbol->errtxt);

        #ifdef TEST_EXAMPLES_GENERATE_EXPECTED
        if (ret == 0) {
            printf("        /*%2d*/ { %s, \"%s\", %d, %d, %d, \"%s\",\n",
                    i, testUtilBarcodeName(symbol->symbology), data[i].data, ret, symbol->rows, symbol->width, data[i].comment);
            testUtilModulesDump(symbol, "                    ", "\n");
            printf("               },\n");
        } else {
            printf("        /*%2d*/ { %s, \"%s\", %s, %d, %d, \"%s\", \"\" },\n",
                    i, testUtilBarcodeName(symbol->symbology), data[i].data, testUtilErrorName(ret), symbol->rows, symbol->width, data[i].comment);
        }
        #else

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
        assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

        if (ret == 0) {
            int width, row;
            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);
        }
        #endif

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

// Test general-purpose data compaction
static void test_general_field(void)
{
    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char* data;
        int ret;

        int expected_rows;
        int expected_width;
        char* comment;
        unsigned char* expected;
    };
    // Verified manually against bwipp and tec-it.com (bottom separator differs from tec-it.com where noted)
    struct item data[] = {
        /* 0*/ { BARCODE_RSS_EXP, "[91]1", 0, 1, 102, "Single numeric",
                    "010100000001000101101111111100001011001000010000010110111110101100001011110000000010101111100001011101"
               },
        /* 1*/ { BARCODE_RSS_EXP, "[91]12", 0, 1, 102, "2 numerics",
                    "010010000010000101101111111100001011001000010000010101111100101110001011110000000010101111100001011101"
               },
        /* 2*/ { BARCODE_RSS_EXP, "[91]123", 0, 1, 102, "Odd-numbered numeric",
                    "010100000110000100101111111100001011001000010000010100011111010111001011110000000010000011000110100101"
               },
        /* 3*/ { BARCODE_RSS_EXP, "[91]1234", 0, 1, 102, "Even-numbered numeric",
                    "010110010000001000101111111100001011001000010000010100111110001011101011110000000010001101111001011101"
               },
        /* 4*/ { BARCODE_RSS_EXP, "[91]A1234567C", 0, 1, 183, "Alphanumeric followed by 7 digits and alphanumeric",
                    "010100000111001001101111111100001011000001000101110101111110111101001011111100001110100110111110111100001111010110011000111111000010110001000001101101110111101111010010111111110011101"
               },
        /* 5*/ { BARCODE_RSS_EXP, "[91]A123456C", 0, 1, 151, "Alphanumeric followed by 6 digits and alphanumeric",
                    "0101100111001000001011111111000010110010000100000101011111101111010010111100000011101001101111101111000011110101100110001111110000101000100011000111010"
               },
        /* 6*/ { BARCODE_RSS_EXP, "[91]A12345B", 0, 1, 151, "Alphanumeric followed by 5 digits and alphanumeric",
                    "0101111001000001001011111111000010110010000100000101011111101111010010111100000011100000010111001001010000111101000010001111110000101100000001001010010"
               },
        /* 7*/ { BARCODE_RSS_EXP, "[91]A1234567", 0, 1, 151, "Alphanumeric followed by 7 digits, terminating",
                    "0101100100011100001011111111000010110010000100000101011111101111010010111100000011101001101111101111000011110110100110001111110000101101011110111100010"
               },
        /* 8*/ { BARCODE_RSS_EXP, "[91]A123456", 0, 1, 134, "Alphanumeric followed by 6 digits, terminating",
                    "01000101001100000010111111110000101100000100010111010111111011110100101111000000111010011011111011110000111101011001100011111100001010"
               },
        /* 9*/ { BARCODE_RSS_EXP, "[91]A12345", 0, 1, 134, "Alphanumeric followed by 5 digits, terminating",
                    "01000110010100000010111111110000101100000100010111010111111011110100101111000000111010011011111011110000101100111110100011111100001010"
               },
        /*10*/ { BARCODE_RSS_EXP, "[91]A1234", 0, 1, 134, "Alphanumeric followed by 4 digits, terminating",
                    "01011101000010000110111111110000101100000100010111010111111011110100101111000000111010011011111011110001101111100100100011111100001010"
               },
        /*11*/ { BARCODE_RSS_EXP, "[91]A123", 0, 1, 134, "Alphanumeric followed by 3 digits, terminating",
                    "01000010110010000010111111110000101100000100010111010111111011110100101111000000111000000101110010010001000010000101100011111100001010"
               },
        /*12*/ { BARCODE_RSS_EXP, "[91]a1234ABCDEFGb", 0, 1, 249, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 4 digits",
                    "010000100011100110101111111100001011001000010000010101101111110011101011111000000110000110110100011100001110000101011000111111000010100111000010110001000000100110110010111111110011101000001110010001010011011111100110001111000000101110111010011000010"
               },
        /*13*/ { BARCODE_RSS_EXP, "[91]a1234ABCDEFb", 0, 1, 249, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 4 digits",
                    "010111010001110001101111111100001011001000010000010101101111110011101011111000000110000110110100011100001110000101011000111111000010100111000010110001000000100110110010111111110011101000001110010001001100011011100110001111000000101101000111001110010"
               },
        /*14*/ { BARCODE_RSS_EXP, "[91]a1234ABCDEF", 0, 1, 232, "ISO-646 followed by 10 non-ISO-646 terminating, starting 4 digits",
                    "0100001011010000111011111111000010110000010001011101011011111100111010111110000001100001101101000111000011100001010110001111110000101001110000101100010000001001101100101111111100111010000011100100010011100111000101100011110000001010"
               },
        /*15*/ { BARCODE_RSS_EXP, "[91]a1234ABCDEb", 0, 1, 232, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 4 digits",
                    "0100001011011000011011111111000010110000010001011101011011111100111010111110000001100100011100100111011110010111110010001111110000101111100011001010010110001000011110101111111100111011100101111000110111001000011110100011110000001010"
               },
        /*16*/ { BARCODE_RSS_EXP, "[91]a1234ABCDE", 0, 1, 200, "ISO-646 followed by 9 non-ISO-646 terminating, starting 4 digits",
                    "01001000011000111010111111110000101100100001000001010110111111001110101111110000111000011011010001110000111000010101100011111100001010011100001011000100000010011011001011111111001110010011100000100101"
               },
        /*17*/ { BARCODE_RSS_EXP, "[91]aABCDEF12345b", 0, 1, 249, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 6 letters",
                    "010000100010100111101111111100001011001000010000010101101111110011101011111000000110100001110001011100010000010100011000111111000010111101000100100001100111010000110010111111110011101000001110011011000100000110101110001111000000101011110010001110010"
               },
        /*18*/ { BARCODE_RSS_EXP, "[91]aABCDEF1234b", 0, 1, 249, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 6 letters",
                    "010110111100110000101111111100001011001000010000010101101111110011101011111000000110100001110001011100010000010100011000111111000010111101000100100001100111010000110010111111110011101000001110011011000100001001110110001111000000101111110110110001010"
               },
        /*19*/ { BARCODE_RSS_EXP, "[91]aABCDE12345b", 0, 1, 249, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 5 letters",
                    "010000100010011011101111111100001011001000010000010101101111110011101011111000000110100001110001011100010000010100011000111111000010111101000100100001100100001110011010111111110011100110110010000111000101111000101110001111000000101111011001101000010"
               },
        /*20*/ { BARCODE_RSS_EXP, "[91]aABCDE1234", 0, 1, 200, "ISO-646 followed by 10 non-ISO-646 terminating, starting 5 letters",
                    "01000101100011100010111111110000101100100001000001010110111111001110101111110000111010000111000101110001000001010001100011111100001011110100010010000110000100101111001011111111001110010010001110011101"
               },
        /*21*/ { BARCODE_RSS_EXP, "[91]aABCDE1234b", 0, 1, 232, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 5 letters",
                    "0100010000110011011011111111000010110000010001011101011011111100111010111110000001100001000110110111010000010110000010001111110000101000011101001100011001000011100110101111111100111011001100100001110111001000011110100011110000001010"
               },
        /*22*/ { BARCODE_RSS_EXP, "[91]aABCDE1234", 0, 1, 200, "ISO-646 followed by 9 non-ISO-646 terminating, starting 5 letters",
                    "01000101100011100010111111110000101100100001000001010110111111001110101111110000111010000111000101110001000001010001100011111100001011110100010010000110000100101111001011111111001110010010001110011101"
               },
        /*23*/ { BARCODE_RSS_EXPSTACK, "[91]1", 0, 1, 102, "Single numeric",
                    "010100000001000101101111111100001011001000010000010110111110101100001011110000000010101111100001011101"
               },
        /*24*/ { BARCODE_RSS_EXPSTACK, "[91]12", 0, 1, 102, "2 numerics",
                    "010010000010000101101111111100001011001000010000010101111100101110001011110000000010101111100001011101"
               },
        /*25*/ { BARCODE_RSS_EXPSTACK, "[91]123", 0, 1, 102, "Odd-numbered numeric",
                    "010100000110000100101111111100001011001000010000010100011111010111001011110000000010000011000110100101"
               },
        /*26*/ { BARCODE_RSS_EXPSTACK, "[91]1234", 0, 1, 102, "Even-numbered numeric",
                    "010110010000001000101111111100001011001000010000010100111110001011101011110000000010001101111001011101"
               },
        /*27*/ { BARCODE_RSS_EXPSTACK, "[91]A1234567C", 0, 5, 102, "Alphanumeric followed by 7 digits and alphanumeric",
                    "010100000111001001101111111100001011000001000101110101111110111101001011111100001110100110111110111101"
                    "000011111000110110010000000010100100111110111010001010000001000010100100000010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000100000000101101000010000100010010011111011100100101000000011001100101000011000000000000000000000"
                    "101110011111111010010111101111011101101100000100011010000111111000110011010111100001000000000000000000"
               },
        /*28*/ { BARCODE_RSS_EXPSTACK, "[91]A123456C", 0, 5, 102, "Alphanumeric followed by 6 digits and alphanumeric",
                    "010110011100100000101111111100001011001000010000010101111110111101001011110000001110100110111110111101"
                    "000001100011011111010000000010100100110111101111101010000001000010100100001010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001100001010011001010000001010010111011100111000000000000000000000000000000000000000000000000000000"
                    "001000011110101100110001111110000101000100011000111010000000000000000000000000000000000000000000000000"
               },
        /*29*/ { BARCODE_RSS_EXPSTACK, "[91]A12345B", 0, 5, 102, "Alphanumeric followed by 5 digits and alphanumeric",
                    "010111100100000100101111111100001011001000010000010101111110111101001011110000001110000001011100100101"
                    "000000011011111011010000000010100100110111101111101010000001000010100100001010100001111110100011010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001111000010111101010000001010010011111110110100000000000000000000000000000000000000000000000000000"
                    "001010000111101000010001111110000101100000001001010010000000000000000000000000000000000000000000000000"
               },
        /*30*/ { BARCODE_RSS_EXPSTACK, "[91]A1234567", 0, 5, 102, "Alphanumeric followed by 7 digits, terminating **NOTE** bottom separator differs from tec-it.com, same as bwipp",
                    "010110010001110000101111111100001011001000010000010101111110111101001011110000001110100110111110111101"
                    "000001101110001111010000000010100100110111101111101010000001000010100100001010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001100001001011001010000001010010010100001000010000000000000000000000000000000000000000000000000000"
                    "001000011110110100110001111110000101101011110111100010000000000000000000000000000000000000000000000000"
               },
        /*31*/ { BARCODE_RSS_EXPSTACK, "[91]A123456", 0, 5, 102, "Alphanumeric followed by 6 digits, terminating **NOTE** ditto",
                    "010100001100111000101111111100001011001000010000010101111110111101001011110000001110100110111110111101"
                    "000011110011000111010000000010100100110111101111101010000001000010100100001010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001100001010011001010000001010010011101011111110000000000000000000000000000000000000000000000000000"
                    "001000011110101100110001111110000101100010100000001010000000000000000000000000000000000000000000000000"
               },
        /*32*/ { BARCODE_RSS_EXPSTACK, "[91]A12345", 0, 5, 102, "Alphanumeric followed by 5 digits, terminating **NOTE** ditto",
                    "010100100011000011101111111100001011001000010000010101111110111101001011110000001110100110111110111101"
                    "000011011100111100010000000010100100110111101111101010000001000010100100001010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001101001100000101010000001010010011101011111110000000000000000000000000000000000000000000000000000"
                    "001000010110011111010001111110000101100010100000001010000000000000000000000000000000000000000000000000"
               },
        /*33*/ { BARCODE_RSS_EXPSTACK, "[91]A1234", 0, 5, 102, "Alphanumeric followed by 4 digits, terminating **NOTE** ditto",
                    "010111000010010001101111111100001011001000010000010101111110111101001011110000001110100110111110111101"
                    "000000111101101110010000000010100100110111101111101010000001000010100100001010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001001000001101101010000001010010000111011101110000000000000000000000000000000000000000000000000000"
                    "001000110111110010010001111110000101111000100010001010000000000000000000000000000000000000000000000000"
               },
        /*34*/ { BARCODE_RSS_EXPSTACK, "[91]A123", 0, 5, 102, "Alphanumeric followed by 3 digits, terminating **NOTE** ditto",
                    "010110000100111000101111111100001011001000010000010101111110111101001011110000001110000001011100100101"
                    "000001111011000111010000000010100100110111101111101010000001000010100100001010100001111110100011010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001011110111101001010000001010010101111100001010000000000000000000000000000000000000000000000000000"
                    "001000100001000010110001111110000101010000011110100010000000000000000000000000000000000000000000000000"
               },
        /*35*/ { BARCODE_RSS_EXPSTACK, "[91]a1234ABCDEFGb", 0, 9, 102, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 4 digits **NOTE** ditto",
                    "010000100011100110101111111100001011001000010000010101101111110011101011111000000110000110110100011101"
                    "000011011100011001010000000010100100110111101111101010010000001100010100000101010001111001001011100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "101000100111000001011100111111110100110110010000001000110100001110010100001111110001101010000111000010"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011001000000110010100001010100100010001011001100000000000000000000000000000000000000000000000000000"
                    "010100110111111001100011110000001011101110100110000100000000000000000000000000000000000000000000000000"
               },
        /*36*/ { BARCODE_RSS_EXPSTACK, "[91]a1234ABCDEFb", 0, 9, 102, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 4 digits",
                    "010111010001110001101111111100001011001000010000010101101111110011101011111000000110000110110100011101"
                    "000000101110001110010000000010100100110111101111101010010000001100010100000101010001111001001011100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "101000100111000001011100111111110100110110010000001000110100001110010100001111110001101010000111000010"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000111001000110010100001010100100101110001100000000000000000000000000000000000000000000000000000000"
                    "010011000110111001100011110000001011010001110011100100000000000000000000000000000000000000000000000000"
               },
        /*37*/ { BARCODE_RSS_EXPSTACK, "[91]a1234ABCDEF", 0, 9, 102, "ISO-646 followed by 10 non-ISO-646 terminating, starting 4 digits **NOTE** ditto",
                    "010110111001000011101111111100001011001000010000010101101111110011101011111000000110000110110100011101"
                    "000001000110111100010000000010100100110111101111101010010000001100010100000101010001111001001011100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "101000100111000001011100111111110100110110010000001000110100001110010100001111110001101010000111000010"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000011000111010010100001010100101011111000010100000000000000000000000000000000000000000000000000000"
                    "010011100111000101100011110000001010100000111101000100000000000000000000000000000000000000000000000000"
               },
        /*38*/ { BARCODE_RSS_EXPSTACK, "[91]a1234ABCDEb", 0, 9, 102, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 4 digits **NOTE** ditto",
                    "010110111001100001101111111100001011001000010000010101101111110011101011111000000110010001110010011101"
                    "000001000110011110010000000010100100110111101111101010010000001100010100000101010001101110001101100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011100001011000100001000000001010000111101110010110101100111000001001010000001010110000010110000000"
                    "101100011110100111011100111111110101111000010001101001010011000111110100001111110001001111101001111010"
                    "000011100001011000100001000000001010000111101110010110101100111000001001010000001010110000010110000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000110111100001010100001010100101011111000010100000000000000000000000000000000000000000000000000000"
                    "010111001000011110100011110000001010100000111101000100000000000000000000000000000000000000000000000000"
               },
        /*39*/ { BARCODE_RSS_EXPSTACK, "[91]a1234ABCDE", 0, 5, 102, "ISO-646 followed by 9 non-ISO-646 terminating, starting 4 digits",
                    "010010000110001110101111111100001011001000010000010101101111110011101011111100001110000110110100011101"
                    "000001111001110001010000000010100100110111101111101010010000001100010100000010100001111001001011100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010111110001101100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "101001000001110010011100111111110100110110010000001000110100001110010100001111110001101010000111000010"
               },
        /*40*/ { BARCODE_RSS_EXPSTACK, "[91]aABCDEF12345b", 0, 9, 102, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 6 letters",
                    "010000100010100111101111111100001011001000010000010101101111110011101011111000000110100001110001011101"
                    "000011011101011000010000000010100100110111101111101010010000001100010100000101010001011110001110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010011000111110100001000000001011001111010001100111101101110100001001010000001010011101011111010000"
                    "101101100111000001011100111111110100110000101110011000010010001011110100001111110001100010100000100010"
                    "000010011000111110100001000000001011001111010001100111101101110100001001010000001010011101011111010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010111110010100010100001010100101000011011100000000000000000000000000000000000000000000000000000000"
                    "010001000001101011100011110000001010111100100011100100000000000000000000000000000000000000000000000000"
               },
        /*41*/ { BARCODE_RSS_EXPSTACK, "[91]aABCDEF1234b", 0, 9, 102, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 6 letters **NOTE** ditto",
                    "010110111100110000101111111100001011001000010000010101101111110011101011111000000110100001110001011101"
                    "000001000011001111010000000010100100110111101111101010010000001100010100000101010001011110001110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010011000111110100001000000001011001111010001100111101101110100001001010000001010011101011111010000"
                    "101101100111000001011100111111110100110000101110011000010010001011110100001111110001100010100000100010"
                    "000010011000111110100001000000001011001111010001100111101101110100001001010000001010011101011111010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010111101100010010100001010100100000010010011100000000000000000000000000000000000000000000000000000"
                    "010001000010011101100011110000001011111101101100010100000000000000000000000000000000000000000000000000"
               },
        /*42*/ { BARCODE_RSS_EXPSTACK, "[91]aABCDE12345b", 0, 9, 102, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 5 letters **NOTE** ditto",
                    "010000100010011011101111111100001011001000010000010101101111110011101011111000000110100001110001011101"
                    "000011011101100100010000000010100100110111101111101010010000001100010100000101010001011110001110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001111011001001100001000000001010011000111101100111101101110100001001010000001010011101011111010000"
                    "101110000100110110011100111111110101100111000010011000010010001011110100001111110001100010100000100010"
                    "000001111011001001100001000000001010011000111101100111101101110100001001010000001010011101011111010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010100001110100010100001010100100001001100101100000000000000000000000000000000000000000000000000000"
                    "010001011110001011100011110000001011110110011010000100000000000000000000000000000000000000000000000000"
               },
        /*43*/ { BARCODE_RSS_EXPSTACK, "[91]aABCDE1234", 0, 5, 102, "ISO-646 followed by 10 non-ISO-646 terminating, starting 5 letters",
                    "010001011000111000101111111100001011001000010000010101101111110011101011111100001110100001110001011101"
                    "000010100111000111010000000010100100110111101111101010010000001100010100000010100001011110001110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001100011101101100001000000001011000010110111100111101101110100001001010000001010011101011111010000"
                    "101110011100010010011100111111110100111101001000011000010010001011110100001111110001100010100000100010"
               },
        /*44*/ { BARCODE_RSS_EXPSTACK, "[91]aABCDE1234b", 0, 9, 102, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 5 letters **NOTE** ditto",
                    "010000100001110110101111111100001011001000010000010101101111110011101011111000000110000100011011011101"
                    "000011011110001001010000000010100100110111101111101010010000001100010100000101010001111011100100100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001111011001100100001000000001010011000111101100111001101000111101001010000001010111110010111110000"
                    "101110000100110011011100111111110101100111000010011000110010111000010100001111110001000001101000001010"
                    "000001111011001100100001000000001010011000111101100111001101000111101001010000001010111110010111110000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000110111100001010100001010100101011111000010100000000000000000000000000000000000000000000000000000"
                    "010111001000011110100011110000001010100000111101000100000000000000000000000000000000000000000000000000"
               },
        /*45*/ { BARCODE_RSS_EXPSTACK, "[91]aABCDE1234", 0, 5, 102, "ISO-646 followed by 9 non-ISO-646 terminating, starting 5 letters",
                    "010001011000111000101111111100001011001000010000010101101111110011101011111100001110100001110001011101"
                    "000010100111000111010000000010100100110111101111101010010000001100010100000010100001011110001110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001100011101101100001000000001011000010110111100111101101110100001001010000001010011101011111010000"
                    "101110011100010010011100111111110100111101001000011000010010001011110100001111110001100010100000100010"
               },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        int length = strlen(data[i].data);

        ret = ZBarcode_Encode(symbol, data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d %s\n", i, ret, data[i].ret, symbol->errtxt);

        #ifdef TEST_GENERAL_FIELD_GENERATE_EXPECTED
        if (ret == 0) {
            printf("        /*%2d*/ { %s, \"%s\", %d, %d, %d, \"%s\",\n",
                    i, testUtilBarcodeName(symbol->symbology), data[i].data, ret, symbol->rows, symbol->width, data[i].comment);
            testUtilModulesDump(symbol, "                    ", "\n");
            printf("               },\n");
        } else {
            printf("        /*%2d*/ { %s, \"%s\", %s, %d, %d, \"%s\", \"\" },\n",
                    i, testUtilBarcodeName(symbol->symbology), data[i].data, testUtilErrorName(ret), symbol->rows, symbol->width, data[i].comment);
        }
        #else

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
        assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

        if (ret == 0) {
            int width, row;
            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);
        }
        #endif

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_binary_buffer_size(void)
{
    testStart("");

    int ret;
    struct item {
        unsigned char* data;
        int ret;

        int expected_rows;
        int expected_width;
        char* comment;
    };
    struct item data[] = {
        /* 0*/ { "[91]1", 0, 1, 102, "Minimum digit" },
        /* 1*/ { "[91]+", 0, 1, 102, "Minimum ISO-646" },
        /* 2*/ { "[00]123456789012345678[00]123456789012345678[00]123456789012345678[91]12345678", 0, 1, 543, "70 == any AIs max" },
        /* 3*/ { "[00]123456789012345678[00]123456789012345678[00]123456789012345678[91]123456789", ZINT_ERROR_TOO_LONG, 0, 0, "71 > any AIs max" },
        /* 4*/ { "[01]12345678901234[00]123456789012345678[00]123456789012345678[91]1234567890123456", 0, 1, 543, "74 == 01 + other AIs max" },
        /* 5*/ { "[01]12345678901234[00]123456789012345678[00]123456789012345678[91]12345678901234567", ZINT_ERROR_TOO_LONG, 0, 0, "75 > 01 + other AIs max" },
        /* 6*/ { "[01]92345678901234[3920]123456789012345[00]123456789012345678[91]1234567890123456789", 0, 1, 543, "77 (incl. FNC1 after 3920) == 01 + 392x + other AIs max" },
        /* 7*/ { "[01]92345678901234[3920]123456789012345[00]123456789012345678[91]12345678901234567890", ZINT_ERROR_TOO_LONG, 0, 0, "78 > 01 + 392x + other AIs max" },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = BARCODE_RSS_EXP;
        int length = strlen(data[i].data);

        ret = ZBarcode_Encode(symbol, data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d %s\n", i, ret, data[i].ret, symbol->errtxt);

        #ifdef TEST_BINARY_BUFFER_SIZE_GENERATE_EXPECTED
        printf("        /*%2d*/ { \"%s\", %s, %d, %d, \"%s\" },\n",
                i, data[i].data, testUtilErrorName(ret), symbol->rows, symbol->width, data[i].comment);
        #else

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (%s)\n", i, symbol->rows, data[i].expected_rows, data[i].data);
        assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d (%s)\n", i, symbol->width, data[i].expected_width, data[i].data);
        #endif

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

int main()
{
    test_binary_div_modulo_divisor();
    test_examples();
    test_general_field();
    test_binary_buffer_size();

    testReport();

    return 0;
}
