##############################################################################
#
# Copyright (c) 2002 Ingeniweb SARL
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.0 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE
#
##############################################################################

"""Attachment mixin.
$Id: AttachmentMixin.py,v 1.6 2005/09/28 14:35:41 clebeaupin Exp $
"""

__author__ = ''
__docformat__ = 'restructuredtext'

# Python imports
from types import FunctionType as function

# Zope imports
from Globals import InitializeClass
from AccessControl import ClassSecurityInfo, getSecurityManager
from AccessControl.SecurityManagement import newSecurityManager
from AccessControl.SpecialUsers import emergency_user
from OFS.SimpleItem import SimpleItem
from OFS.CopySupport import _cb_encode, CopyError, cookie_path
from OFS import Moniker

# CMF imports
from Products.CMFCore import CMFCorePermissions
from Products.CMFCore.utils import getToolByName

# PloneArticle imports
from Products.PloneArticle.interfaces.IAttachment import IAttachmentContents, IAttachmentReferenceBrain
from config import *

def registerAttachmentSchema(klass, attachment_schema):
    """Register attachment schema and generate methods
    """
    for item in getattr(klass, attachment_schema):
        item.generateMethods(klass)

class methodGenerator:
    """Method generator
    """
    
    def make(self, from_klass, to_klass, from_method, to_method, permission):
        method = None
        
        def new_method(self, *args, **kwargs):
            field = from_klass
            return getattr(field, from_method)(self, *args, **kwargs)
        
        method = new_method
        method_name = to_method
        
        method = function(method.func_code,
                          method.func_globals,
                          method_name,
                          method.func_defaults,
                          method.func_closure,
                         )
        
        # Add function
        setattr(to_klass, method_name, method)
        
        # Use security
        if not to_klass.__dict__.has_key('security'):
            security = to_klass.security = ClassSecurityInfo()
        else:
            security = to_klass.security
        
        security.declareProtected(permission, method_name)

class AttachmentReferenceBrain(SimpleItem):
    
    # Standard security settings
    security = ClassSecurityInfo()
    
    __implements__ = (IAttachmentReferenceBrain,)
    
    security.declarePrivate('__init__')
    def __init__(self, title='', description=''):
        """
        """
        
        self.title = title
        self.description = description
    
    security.declarePublic('Title')
    def Title(self):
        """
        Get title.
        STORED IN REFERENCE.
        """
        
        return self.title
    
    security.declarePublic('Description')
    def Description(self):
        """
        Get description.
        STORED IN REFERENCE.
        """
        
        return self.description
    
    security.declarePublic('getObject')
    def getObject(self):
        """
        Returns object.
        """

        return self.getTargetObject()
    
    security.declarePrivate('setTitle')
    def setTitle(self, title):
        """
        Set title.
        STORED IN REFERENCE.
        """
        
        self.title = title
    
    security.declarePrivate('setDescription')
    def setDescription(self, description):
        """
        Set description.
        STORED IN REFERENCE.
        """
        
        self.description = description
        
InitializeClass(AttachmentReferenceBrain)

class OrderedList(SimpleItem):
    """Ordered list with unique elements"""
    
    security = ClassSecurityInfo()
    
    security.declarePrivate('__init__')
    def __init__(self):
        """Initialize list"""
        self.olist = []

    security.declarePrivate('add')
    def add(self, items):
        """Add new items"""
        for item in items:
            self.olist.append(item)
        
        self.olist = self.olist
    
    security.declarePrivate('remove')
    def remove(self, items):
        """Remove items"""
        for item in items:
            self.olist.remove(item)
            
        self.olist = self.olist
    
    security.declarePrivate('moveByDelta')
    def moveByDelta(self, items, delta):
        """Move items by delta"""
      
        for item in items:
            index = self.olist.index(item) - delta
            self.olist.remove(item)
            self.olist.insert(index, item)
            
        self.olist = self.olist

    security.declarePrivate('getItems')
    def getItems(self):
        """Get all items"""
        return self.olist

InitializeClass(OrderedList)
    
class AttachmentContents:
    """
    Base class for attachment features
    """
    
    # Standard security settings
    security = ClassSecurityInfo()
    
    __implements__ = (IAttachmentContents,)
    
    _content_brain_id = '__attachment_brain__'
    _ordered_refs_list_id = '__ordered_%s_refs__'
    _relationship_id = 'attachment_%s'
    _brain_klass = AttachmentReferenceBrain
    _view_permission = CMFCorePermissions.View
    _write_permission = CMFCorePermissions.ModifyPortalContent
    
    security.declarePrivate('__init__')
    def __init__(self, name):
        """
        initialize attachments
        """
    
        self.name = name
    
    security.declarePrivate('getContentType')
    def getContentType(self, instance): 
        """
        Returns portal type to add
        """
        raise NotImplementedError('AttachmentContents: getContentType')
        
    security.declarePrivate('getAllowedContentTypes')
    def getAllowedContentTypes(self, instance): 
        """
        Returns portal types you can paste as reference
        """
        return [self.getContentType(instance)]

    security.declarePrivate('getRelationShip')
    def getRelationShip(self): 
        """
        Returns relationship name used in references
        """
        return self._relationship_id % self.name
    
    security.declarePrivate('getOrderedRefsList')
    def getOrderedRefsList(self, instance):
        """Returns ordered references list
        """
        name = self._ordered_refs_list_id % self.name
        
        if not hasattr(instance, name):
            setattr(instance, name, OrderedList())
            
        return getattr(instance, name)

    security.declarePrivate('moveContentsByDelta')
    def moveContentsByDelta(self, instance, UIDs, delta):
        """
        Move contents by delta
        """
        
        refs = self.getContentReferences(instance)
        ref_UIDs = []
        
        # Get references UIDs
        for ref in refs:
            if ref.targetUID in UIDs:
                ref_UIDs.append(ref.UID())
        
        self.getOrderedRefsList(instance).moveByDelta(ref_UIDs, delta)
    
    security.declarePrivate('getContent')
    def getContent(self, instance, UID):
        """
        Parameters
            UID -> UID of content
            
        Returns a content object
        """
        
        at_tool = getToolByName(instance, 'archetype_tool')
        
        # Get object from UID and add content reference
        content = at_tool.getObject(UID)
        return content
    
    security.declarePrivate('getReference')
    def getReference(self, instance, UID):
        """
        Get reference from object having UID
        """
        
        ref_tool = getToolByName(instance, 'reference_catalog')
        refs = ref_tool.searchResults(sourceUID=instance.UID(), targetUID=UID, relationship=self.getRelationShip())
        
        if refs:
            return refs[0].getObject()
        else:
            return None
    
    security.declarePrivate('getContentBrain')
    def getContentBrain(self, instance, UID):
        """
        Parameters
            UID -> UID of content
            
        Returns a brain of content
        """
        
        ref = self.getReference(instance, UID)
        
        if ref:
            return getattr(ref, self._content_brain_id, None)
        else:
            return None
    
    security.declarePrivate('editContent')
    def editContent(self, instance, UID, title='', description=''):
        """
        Edit a content brain
        """
        
        # Modify brain
        brain = self.getContentBrain(instance, UID)
        brain.setTitle(title)
        brain.setDescription(description)
        
        # Modify object if in content
        content = self.getContent(instance, UID)
        parent = content.getParentNode()
        
        if hasattr(parent, 'UID') and parent.UID() == instance.UID():
            content.setTitle(title)
            content.setDescription(description)        
            
        # Reindex content
        instance.reindexObject()
        
    security.declarePrivate('editContents')
    def editContents(self, instance, content_infos):
        """
        Edit content brains
        infos is a dictionnary : uid -> (title, description)
        """
        
        for content_info in content_infos:
            UID = content_info['UID']
            title = content_info.get('title', '')
            description = content_info.get('description', '')
            self.editContent(instance, UID, title, description)
    
    security.declarePrivate('addContent')
    def addContent(self, instance, *args, **kwargs):
        """
        Attach a content to instance. Add content in instance.
        
        Parameters
            title -> title of content
            description -> description of content
        """
        
        # Create attachment in instance
        content_id = instance.generateUniqueId()
        instance.invokeFactory(self.getContentType(instance), \
                           id=content_id, \
                           **kwargs)
        
        # Get new attachment and add attachment reference
        content = getattr(instance, content_id)
        
        # Unindex contents in instance
        cat_tool = getToolByName(instance, 'portal_catalog')
        cat_tool.unindexObject(content)
        
        self.addContentFromObject(instance, content)
    
    security.declarePrivate('addContentFromObject')
    def addContentFromObject(self, instance, content):
        """
        Attach a content to instance. Just add a reference between content and
        instance.
        
        Parameters
            attachment -> content object to attach
        """
        
        content_type = content.portal_type
        allowed_content_type = self.getAllowedContentTypes(instance)
        
        if content_type not in allowed_content_type:
            Log(LOG_NOTICE, "Wrong attachment type. Expected : %s, given %s." % (allowed_content_type, content_type))
            raise TypeError, "%s cannot be added as %s. Use %s type" % (content_type, self.name, allowed_content_type)
        
        # Create brain
        brain = self._brain_klass(content.Title(), content.Description())
        
        # Add reference
        ref = instance.addReference(content, self.getRelationShip())
        setattr(ref, self._content_brain_id, brain)
        brain.__of__(ref)
        
        # Append ref UID to an ordered list
        self.getOrderedRefsList(instance).add([ref.UID()])
        
        # Reindex instance
        instance.reindexObject()
    
    security.declarePrivate('addContentFromUID')
    def addContentFromUID(self, instance, UID):
        """
        Attach a content to instance. Just add a reference between content and
        instance.
        
        Parameters
            UID -> UID of a content to attach
        """
        
        # Get object from UID and add content reference
        content = self.getContent(instance, UID)
        self.addContentFromObject(instance, content)
    
    security.declarePrivate('removeContent')
    def removeContent(self, instance, UID):
        """
        Remove an attachment from instance
        
        Parameters
            UID -> UID of content
        """
        
        # Get object from UID and add content reference
        content = self.getContent(instance, UID)
        content_id = content.getId()
        
        # Remove object if in instance
        # Just delete reference if not
        parent = content.getParentNode()
        ref = self.getReference(instance, UID)
        
        # Delete ref UID from the ordered list
        self.getOrderedRefsList(instance).remove([ref.UID()])
        
        if hasattr(parent, 'UID') and parent.UID() == instance.UID():
            instance.manage_delObjects([content_id])
        else:
            instance.deleteReference(UID, self.getRelationShip())
            
        # Reindex instance
        instance.reindexObject()
    
    security.declarePrivate('removeContents')
    def removeContents(self, instance, UIDs):
        """
        Remove list of contents
        
        Parameters
            UIDs -> List of UIDs
        """
        
        for UID in UIDs:
            self.removeContent(instance, UID)
    
    security.declarePrivate('getContentReferences')
    def getContentReferences(self, instance):
        """
        Returns all content references from instance
        """
        
        refs = instance.getReferenceImpl(self.getRelationShip())
        ref_UIDs = dict([(x.UID(), x) for x in refs])
        refs_list = []
        ordered_refs_list = self.getOrderedRefsList(instance)
        mtool = getToolByName(instance, 'portal_membership')
        request = instance.REQUEST
        
        # Update ordered list
        for ref_UID in ref_UIDs:
            if ref_UID not in ordered_refs_list.getItems():
                ordered_refs_list.add([ref_UID])
                
        for ref_UID in ordered_refs_list.getItems():
            if ref_UID not in ref_UIDs:
                ordered_refs_list.remove([ref_UID])
            else:
                # Check permission and avoid private contents 
                ref = ref_UIDs[ref_UID]
                content = ref.getTargetObject()
                
                if content is None:
                    continue
                
                parent = content.getParentNode()
                has_permission = mtool.checkPermission(CMFCorePermissions.View, content) \
                    and mtool.checkPermission(CMFCorePermissions.AccessContentsInformation, content)
                in_instance = hasattr(parent, 'UID') and parent.UID() == instance.UID()
                
                if in_instance and not has_permission:
                    # Add acquisition on Access Contents information and View permission
                    # Use manager role to change permissions
                    current_user = getSecurityManager().getUser()
                    newSecurityManager(request, emergency_user)
                    
                    # Add acquisition on View and Access content information
                    permissions = [CMFCorePermissions.AccessContentsInformation, CMFCorePermissions.View]
                    for permission in permissions:
                        roles = [x['name'] for x in content.rolesOfPermission(permission) if x['selected']]
                        content.manage_permission(permission, roles, 1)
                    
                    # Restore current user permission
                    newSecurityManager(request, current_user)
                
                if has_permission or in_instance:
                    refs_list.append(ref)
        
        return refs_list
    
    security.declarePrivate('copyContentsToClipboard')
    def copyContentsToClipboard(self, instance, UIDs):
        """Copy objects referenced by UIDs in clipboard
        """
        
        at_tool = getToolByName(instance, 'archetype_tool')
        oblist=[]
        request = instance.REQUEST
        
        for UID in UIDs:
            # Get object from UID and add content reference
            ob = at_tool.getObject(UID)
            
            if not ob.cb_isCopyable():
                Log(LOG_NOTICE, "Object in clipboard is not copyable.")
                raise CopyError, 'Copy Error from : ' % str(UID)
            m = Moniker.Moniker(ob)
            oblist.append(m.dump())
        
        cp = (0, oblist)
        cp = _cb_encode(cp)
        
        if request is not None:
            response = request.RESPONSE
            response.setCookie('__cp', cp, path='%s' % cookie_path(request))
            
        return cp
    
    security.declarePrivate('getContents')
    def getContents(self, instance):
        """
        Returns all content objects referenced in instance
        """
        
        return [x.getTargetObject() for x in self.getContentReferences(instance)]
    
    security.declarePrivate('getContentBrains')
    def getContentBrains(self, instance):
        """
        Returns all content brains referenced in instance
        """
        
        return [getattr(x, self._content_brain_id) for x in self.getContentReferences(instance) if hasattr(x, self._content_brain_id)]
    
    security.declarePrivate('getContentUIDs')
    def getContentUIDs(self, instance):
        """
        Returns all content uids referenced in instance
        """
        
        return [x.targetUID for x in self.getContentReferences(instance)]
    
    security.declarePrivate('isContentContained')
    def isContentContained(self, instance, UID):
        """
        Return true if content is stored in instance else return false.
        """
        
        content = self.getContent(instance, UID)
        parent = content.getParentNode()
        
        if hasattr(parent, 'UID') and parent.UID() == instance.UID():
            return True
        else:
            return False
        
        
    security.declarePrivate('generateMethods')
    def generateMethods(self, klass):
        """
        Generates dynamic methods
        """
        
        name = self.name.capitalize()
        
        gen_methods = (
            ('get%sUIDs', 'getContentUIDs', self._view_permission),
            ('get%sBrains', 'getContentBrains', self._view_permission),
            ('get%ss', 'getContents', self._view_permission),
            ('get%sReferences', 'getContentReferences', self._view_permission),
            ('get%sBrain', 'getContentBrain', self._view_permission),
            ('get%s', 'getContent', self._view_permission),
            ('is%sContained', 'isContentContained', self._view_permission),
            ('remove%ss', 'removeContents', self._write_permission),
            ('remove%s', 'removeContent', self._write_permission),
            ('edit%s', 'editContent', self._write_permission),
            ('edit%ss', 'editContents', self._write_permission),
            ('add%sFromUID', 'addContentFromUID', self._write_permission),
            ('add%sFromObject', 'addContentFromObject', self._write_permission),
            ('add%s', 'addContent', self._write_permission),
            ('move%ssByDelta', 'moveContentsByDelta', self._write_permission),
            ('copy%ssToClipboard', 'copyContentsToClipboard', self._write_permission),
            )
    
        for gen_method in gen_methods:
            to_method = gen_method[0] % name
            from_method = gen_method[1]
            permission = gen_method[2]
            methodGenerator().make(self, klass, from_method, to_method, permission)
        