# TrackerSupport mixin

import re
import DocumentTemplate
from OFS.DTMLDocument import DTMLDocument
from Regexps import url, bracketedexpr, wikiname1, wikiname2, simplewikilink, \
     wikilink, interwikilink, remotewikiurl, protected_line
from Utils import thunk_substituter, within_literal
from AccessControl import getSecurityManager, ClassSecurityInfo
import Permissions

ISSUE_FORM = DocumentTemplate.HTML ('''
<!-- start of issue property form -->
<form action="&dtml-page_url;/changeIssueProperties" method="post">
<table border="0" cellspacing="0" cellpadding="5"
<dtml-if "status == 'open'">
<dtml-if "severity == 'critical'">
bgcolor="#ff2222"
<dtml-elif "severity == 'serious'">
bgcolor="#ff6060"
<dtml-elif "severity == 'normal'">
bgcolor="#ffbbbb"
<dtml-elif "severity == 'minor'">
bgcolor="#ffdddd"
<dtml-elif "severity == 'wishlist'">
bgcolor="#ffe0e0"
<dtml-else>
bgcolor="#ffe0e0"
</dtml-if>
<dtml-elif "status == 'pending'">
bgcolor="#ffcc77"
<dtml-elif "status == 'closed'">
bgcolor="#e0f0e0"
<dtml-else>
bgcolor="#e0e0e0"
</dtml-if>
>
<tr><td>
**Description:**
<b><input type="text" name="title" value="<dtml-var "title[12:]" missing html_quote>" size="50" maxlength="200" style="font-weight:bold"></b>
<dtml-try>
<dtml-let
  highnumber="0#_.int(Catalog(page_type='issuedtml',sort_on='id')[-2].id[-4:])"
  thisnumber="_.int(id()[-4:])"
  previous="'IssueNo'+_.string.zfill(thisnumber-1,4)"
  next="'IssueNo'+_.string.zfill(thisnumber+1,4)"
>
<a href="&dtml-previous;">&lt;&lt;</a>&nbsp;<a href="&dtml-wiki_url;/ZwikiTracker?&dtml-QUERY_STRING;">^^</a>&nbsp;<a href="&dtml-next;">&gt;&gt;</a>
</dtml-let>
<dtml-except>
</dtml-try>
<br>
**Category:**
<select name="category">
<dtml-in issue_categories prefix=x>
<option <dtml-if "category==x_sequence_item">selected</dtml-if>>
&dtml-x_sequence_item;</option>
</dtml-in>
</select> 
**Severity:**
<select name="severity">
<dtml-in issue_severities prefix=x>
<option <dtml-if "severity==x_sequence_item">selected</dtml-if>>
&dtml-x_sequence_item;</option>
</dtml-in>
</select> 
**Status:**
<select name="status">
<dtml-in issue_statuses prefix=x>
<option <dtml-if "status==x_sequence_item">selected</dtml-if>>
&dtml-x_sequence_item;</option>
</dtml-in>
</select>
<br><b>Optional note:</b> <input type="text" name="log" value="" size="55" maxlength="55">&nbsp;
<input name="submit" type="submit" value="Change">
<br>
**Submitted by:**
<dtml-var "_.getattr(this(),'creator','') or '(unknown)'">
**at:**
<dtml-var creation_time missing=""><dtml-let
creation_time="_.getattr(this(),'creation_time','')"
creation="_.DateTime(_.getattr(this(),'creation_time','') or 1007105000)"
current="_.DateTime(ZopeTime().strftime('%Y/%m/%d %H:%M:%S'))"
elapsed="current-creation"
hourfactor="0.041666666666666664"
minutefactor="0.00069444444444444447"
secondsfactor="1.1574074074074073e-05"
days="_.int(_.math.floor(elapsed))"
weeks="days / 7"
months="days / 30"
years="days / 365"
hours="_.int(_.math.floor((elapsed-days)/hourfactor))"
minutes="_.int(_.math.floor((elapsed-days-hourfactor*hours)/minutefactor))"
seconds="_.int(_.round((elapsed-days-hourfactor*hours-minutefactor*minutes)/secondsfactor))"
>(<dtml-unless creation_time>unknown, ></dtml-unless><dtml-if years>
<dtml-var years> year<dtml-var "years > 1 and 's' or ''">
<dtml-elif months>
<dtml-var months> month<dtml-var "months > 1 and 's' or ''">
<dtml-elif weeks>
<dtml-var weeks> week<dtml-var "weeks > 1 and 's' or ''">
<dtml-elif days>
<dtml-var days> day<dtml-var "days > 1 and 's' or ''">
<dtml-elif hours>
<dtml-var hours> hour<dtml-var "hours > 1 and 's' or ''">
<dtml-elif minutes>
<dtml-var minutes> minute<dtml-var "minutes > 1 and 's' or ''">
<dtml-else>
<dtml-var seconds> second<dtml-var "seconds > 1 and 's' or ''">
</dtml-if> ago)
</dtml-let>
<br>
**Details & comments:**
</td></tr></table>
</form>
<!-- end of issue property form -->
''')

class TrackerSupport:
    """
    This mix-in class adds some methods to ZWikiPage to support the
    zwiki tracker (see http://zwiki.org/ZwikiTracker)
    """
    security = ClassSecurityInfo()

    security.declareProtected(Permissions.View, 'supportsIssueProperties')
    def supportsIssueProperties(self):
        """Does this page support issue properties management ?"""
        return re.search(r'(?i)issue',self.page_type) is not None

    security.declareProtected(Permissions.View, 'isIssue')
    def isIssue(self,client=None,REQUEST=None,RESPONSE=None,**kw):
        """
        Return true if this page is a tracker issue.

        In the past pages with a special page type were issues. Now, any
        page named "IssueNo.." is an issue. (and, whose type supports
        issue properties ? No never mind that)

        Flexibility will be useful here so this method may be overridden
        with a DTML method (XXX or python script).
        """
        if hasattr(self.folder(), 'isIssue'):
            return self.folder().isIssue(self,REQUEST)
        else:
            if (re.match(r'^IssueNo',self.title_or_id())
                or self.page_type == 'issuedtml'): # backwards compatibility
                return 1
            else:
                return 0

    def render_issuedtml(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        Render an "issue" page. This is an ordinary zwiki page with some
        extra properties appropriate for a bug/issue. These are displayed
        above the main text of the page in a sort of trouble ticket layout
        and can be edited.

        based on render_stxdtmllinkhtml

        deprecated - this is now supported in stxprelinkdtmlfitissuehtml        
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = self.applyLineEscapesIn(t)
            t = self.stxToHtml(t)
            self._prerendered = t or '\n'
            self.cook()
        if kw.get('pre_only',0): return
        # final render
        # dtml-evaluate both form and main page
        t = self.stxToHtml(apply(ISSUE_FORM.__call__,(self, REQUEST),kw)) + \
            apply(DTMLDocument.__call__,(self, client, REQUEST, RESPONSE), kw)
        t = self.renderLinksIn(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    def createIssue(self, pageid, text=None, title='',
                    category=None, severity=None, status=None, REQUEST=None):
        """
        Convenience method for creating a ZwikiTracker issue page.

        Security notes: create will check for page creation permission.
        Sets title/category/severity/status properties without requiring
        Manage properties permission.

        As of 0.17, issue pages are named "IssueNoNNNN issue description".
        These arguments should be cleaned up to reflect that, eg title
        should come from pageid and should be called description.  pageid
        should be called pagename.

        Also issues are now "ordinary" stxprelinkdtmlfitissuehtml pages.
        That page type is hardcoded here, if we add issue support to
        others we'd want to inherit from the parent page.

        We should be able to do without this method. 
        """
        self.create(pageid,text=text,REQUEST=REQUEST)
        issue = self.pageWithName(pageid)
        issue.manage_addProperty('category','issue_categories','selection')
        issue.manage_addProperty('severity','issue_severities','selection')
        issue.manage_addProperty('status','issue_statuses','selection')
        issue.manage_changeProperties(page_type='stxprelinkdtmlfitissuehtml',
                                      title=title,
                                      category=category,
                                      severity=severity,
                                      status=status
                                      )
        self.reindex_object()

    #def changeProperties(self, REQUEST=None, **kw):
    #    """
    #    Similar to manage_changeProperties, except redirects back to the
    #    current page. Also restores the issue number which we previously
    #    stripped from title.
    #    
    #    security issue: bypasses Manage properties permission
    #    
    #    Deprecated, useful for backwards compatibility or remove ?
    #    """
    #    if REQUEST is None:
    #        props={}
    #    else: props=REQUEST
    #    if kw:
    #        for name, value in kw.items():
    #            props[name]=value
    #    props['title'] = self.getId()[:11]+' '+props['title']
    #    propdict=self.propdict()
    #    for name, value in props.items():
    #        if self.hasProperty(name):
    #            if not 'w' in propdict[name].get('mode', 'wd'):
    #                raise 'BadRequest', '%s cannot be changed' % name
    #            self._updateProperty(name, value)
    #
    #    self._setLastEditor(REQUEST)
    #    self.reindex_object()
    #    if REQUEST:
    #        REQUEST.RESPONSE.redirect(self.page_url())
            
    def changeIssueProperties(self, title=None, category=None, severity=None, 
                              status=None, log=None, REQUEST=None):
        """
        Change an issue page's properties and redirect back there.

        Also, add a comment to the page describing what was done.

        It expects title to be the issue description, not the complete
        page name.  Changing this will trigger a page rename, which may be
        slow.
        
        XXX security: allows title/category/severity/status properties to
        be set without Manage properties permission.

        XXX upgrade issue: calling this before upgrading an issue to
        a 0.17-style page id will mess up the id/title.
        """
        comment = ''
        if title:
            title = self.getId()[:11]+' '+title
            if title != self.title_or_id():
                comment += "Title: '%s' => '%s' \n" % (self.title_or_id(),title)
            self.rename(title,updatebacklinks=1,sendmail=0,REQUEST=REQUEST)
        if category:
            if category != self.category:
                comment += "Category: %s => %s \n" % (self.category,category)
            self.manage_changeProperties(category=category)
        if severity:
            if severity != self.severity:
                comment += "Severity: %s => %s \n" % (self.severity,severity)
            self.manage_changeProperties(severity=severity)
        if status:
            if status != self.status:
                comment += "Status: %s => %s \n" % (self.status,status)
            self.manage_changeProperties(status=status)
        log = log or 'property change'
        self.comment(text=comment, use_heading=1, subject_heading=log, REQUEST=REQUEST)
        self._setLastEditor(REQUEST)
        self.reindex_object()
        if REQUEST:
            REQUEST.RESPONSE.redirect(self.page_url())

    def category_index(self):
        """helper method to facilitate sorting catalog results"""
        try:
            return 1 + self.issue_categories.index(self.category)
        except (AttributeError,ValueError):
            return 0
        
    def severity_index(self):
        """helper method to facilitate sorting catalog results"""
        try:
            return 1 + self.issue_severities.index(self.severity)
        except (AttributeError,ValueError):
            return 0

    def status_index(self):
        """helper method to facilitate sorting catalog results"""
        try:
            return 1 + self.issue_statuses.index(self.status)
        except (AttributeError,ValueError):
            return 0

