/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: proxy.c,v 1.66.2.10 2003/10/09 07:42:38 bazsi Exp $
 *
 * Author  : Bazsi
 * Auditor : kisza
 * Last audited version: 1.23
 * Notes:
 *
 ***************************************************************************/

#include <zorp/proxy.h>
#include <zorp/policy.h>
#include <zorp/proxyvars.h>
#include <zorp/pystream.h>
#include <zorp/pysockaddr.h>
#include <zorp/streamfd.h>
#include <zorp/log.h>
#include <zorp/io.h>

#include <stdarg.h>
#include <sys/socket.h>
#include <unistd.h>

/* a two-way connection between streams */

/* z_proxy_var_new() should be here, but is implemented in pyvars.c for now */

/*+

  Initialize a given proxy instance.

  Parameters:
    self       proxy instance to initialize
    session_id session_id
    client     client side Stream
    handler    policy object to send events to

  Returns:
    none

  +*/
void
z_proxy_init(ZProxy *self, 
             gchar *session_id, 
	     ZStream *client, 
	     ZPolicyObj *handler)
{
  ZPolicyThread *policy_thread;
  z_enter();
  if (client)
    {
      self->endpoints[EP_CLIENT] = client;
      z_stream_ref(client);
    }

  self->ref_cnt = 1;  
  g_strlcpy(self->session_id, session_id, sizeof(self->session_id));
  
  self->vars = z_proxy_vars_new();
  
  z_python_lock();
  self->handler = handler;
  z_policy_var_ref(handler);
  policy_thread = z_policy_thread_self();
  self->thread = z_policy_thread_new(policy_thread ? z_policy_thread_get_policy(policy_thread) : current_policy);
  z_python_unlock();

  self->setattr = z_proxy_vars_setattr;
  self->getattr = z_proxy_vars_getattr;
  
  z_proxy_leave(self);
}

/*+

  Called to free up a proxy instance. Frees up associated
  resources, closes streams, and calls the free virtual method.

  Parameters:
    self         proxy instance

  Returns:
    none

  +*/
void
z_proxy_destroy(ZProxy *self)
{
  int i;
  ZPolicyObj *handler;
  ZPolicyThread *thread;
  ZProxyVars *vars;

  z_proxy_enter(self);
  ZPROXY_SET_STATE(self, PS_DESTROYING);
  z_proxy_destroy_event(self);
  
  for (i = EP_CLIENT; i <= EP_SERVER; i++)
    {
      if (self->endpoints[i])
        {
          z_stream_shutdown(self->endpoints[i], SHUT_RDWR, NULL);
          z_stream_close(self->endpoints[i], NULL);
          z_stream_unref(self->endpoints[i]);
          self->endpoints[i] = NULL;
        }
    }
  
  z_policy_thread_acquire(self->thread);
  thread = self->thread;
  self->thread = NULL;
  
  vars = self->vars;
  self->vars = NULL;
  z_proxy_vars_destroy(vars);

  handler = self->handler;
  self->handler = NULL;
  z_policy_var_unref(handler);  

  z_policy_thread_release(thread);
  
  z_policy_thread_destroy(thread);
  
  z_leave();
}

void
z_proxy_free(ZProxy *self)
{
  z_enter();
  
  /* NOTE: z_proxy_fastpath_destroy is called here as the fastpath
   * information might not have been registered when z_proxy_destroy is
   * called. (fastpath info is set right after startup and the proxy might
   * exit while that code is running)
   */
  
  z_proxy_fastpath_destroy(&self->fastpath);
  if (self->free)
    (*self->free)(self);
    
  g_free(self);
  z_leave();
}

ZProxy *
z_proxy_ref(ZProxy *self)
{
  z_enter();
  g_assert(self->ref_cnt > 0);
  self->ref_cnt++;
  z_leave();
  return self;
}

void
z_proxy_unref(ZProxy *self)
{
  z_enter();
  g_assert(self->ref_cnt > 0);
  if (--self->ref_cnt == 0)
    {
      z_proxy_free(self);
    }
  z_leave();
}


void
z_proxy_quit(ZProxy *self)
{
  self->flags |= PF_QUIT;
}

/*+

  Send a config event to the associated policy object.

  Parameters:
    self      proxy instance

  Returns:
    TRUE if config is called, and ended succesfully

  Remarks:
    Locks the policy engine.

  +*/
gboolean
z_proxy_config_event(ZProxy *self)
{
  ZPolicyObj *res;
  gboolean called;
  
  z_proxy_enter(self);
  
  z_policy_thread_acquire(self->thread);
  self->flags |= PF_INITIALIZED;

  /*LOG
    This debug message is written when Zorp is about to call a user
    supplied __config__() event.
   */
  z_proxy_log(self, CORE_DEBUG, 7, "calling __config__() event;");
  res = z_policy_call(self->handler, "__config__", NULL, &called, self->session_id);
  if (res == NULL && called)
    {
      z_policy_thread_release(self->thread);
      z_proxy_leave(self);
      return FALSE;
    }
  z_policy_var_unref(res);

  /*LOG
    This debug message is written when Zorp is about to call a user
    supplied config() event.
   */
  z_proxy_log(self, CORE_DEBUG, 7, "calling config() event;");
  res = z_policy_call(self->handler, "config", NULL, &called, self->session_id);
  if (res == NULL && called)
    {
      z_policy_thread_release(self->thread);
      z_proxy_leave(self);
      return FALSE;
    }
  z_policy_var_unref(res);
  
  z_proxy_vars_dump_values(self->vars, self);
  
  z_policy_thread_release(self->thread);
  z_proxy_leave(self);
  return TRUE;
}

/*+

  Send a startup event to the associated policy object.

  Parameters:
    self      proxy instance

  Returns:
    TRUE if startup event be callable.

  Remarks:
    Locks the policy engine.

  +*/
gboolean
z_proxy_startup_event(ZProxy *self)
{
  ZPolicyObj *res;
  gboolean called;

  z_proxy_enter(self);
  
  z_policy_thread_acquire(self->thread);
  /*LOG
    This debug message is written when Zorp is about to call a user
    supplied __startup__() event.
   */
  z_proxy_log(self, CORE_DEBUG, 7, "calling __startup__() event;");
  res = z_policy_call(self->handler, "__startup__", NULL, &called, self->session_id);
  if (res == NULL && called)
    {
      z_policy_thread_release(self->thread);
      z_proxy_leave(self);
      return FALSE;
    }
  z_policy_var_unref(res);

  /*LOG
    This debug message is written when Zorp is about to call a user
    supplied startUp() event.
   */
  z_proxy_log(self, CORE_DEBUG, 7, "calling startUp() event;");
  res = z_policy_call(self->handler, "startUp", NULL, &called, self->session_id);
  if (res == NULL && called)
    {
      z_policy_thread_release(self->thread);
      z_proxy_leave(self);
      return FALSE;
    }
  z_policy_var_unref(res);
  z_policy_thread_release(self->thread);
  z_proxy_leave(self);
  return TRUE;
}

/*+

  Send a shutdown event to the associated policy object.

  Parameters:
    self      proxy instance

  Returns:
    none

  Remarks:
    Locks the policy engine.

  +*/
void 
z_proxy_shutdown_event(ZProxy *self)
{
  ZPolicyObj *res;
  gboolean called;

  z_proxy_enter(self);
  z_policy_thread_acquire(self->thread);

  /*LOG
    This debug message is written when Zorp is about to call a user
    supplied __shutdown__() event.
   */
  z_proxy_log(self, CORE_DEBUG, 7, "calling __shutdown__() event;");
  res = z_policy_call(self->handler, "__shutdown__", NULL, &called, self->session_id);
  z_policy_var_unref(res);

  /*LOG
    This debug message is written when Zorp is about to call a user
    supplied shutDown() event.
   */
  z_proxy_log(self, CORE_DEBUG, 7, "calling shutDown() event;");
  res = z_policy_call(self->handler, "shutDown", NULL, &called, self->session_id);
  z_policy_var_unref(res);
  z_policy_thread_release(self->thread);
  z_proxy_leave(self);
}

/*+

  Send a __destroy__ event to the associated policy object.

  Parameters:
    self      proxy instance

  Returns:
    none

  Remarks:
    Locks the policy engine.

  +*/
void 
z_proxy_destroy_event(ZProxy *self)
{
  ZPolicyObj *res;
  gboolean called;

  z_proxy_enter(self);
  /*LOG
    This debug message is written when Zorp is about to call a user
    supplied __destroy__() event.
   */
  z_proxy_log(self, CORE_DEBUG, 7, "calling __destroy__() event;");
  z_policy_thread_acquire(self->thread);
  res = z_policy_call(self->handler, "__destroy__", NULL, &called, self->session_id);
  z_policy_var_unref(res);
  z_policy_thread_release(self->thread);
  z_proxy_leave(self);
}

/*+

  Send a connectServer event to the associated policy object.

  Parameters:
    self      proxy instance
    host      host to connect to, used as a hint by the 
              policy layer but may as well be ignored
    port      port in host to connect to

  Returns:
    TRUE if the server-side connection is established, otherwise the
    connection should be closed.

  Remarks:
    Locks the policy engine.

  +*/
gint
z_proxy_connect_server_event(ZProxy *self, char *host, gint port)
{
  ZPolicyObj *res, *args;
  gint rc;
  gboolean called;
  
  z_proxy_enter(self);

  z_policy_thread_acquire(self->thread);
  self->endpoints[EP_SERVER] = NULL;
  if (host)
    {
      args = z_policy_var_build("(si)", host, port);
      res = z_policy_call(self->handler, "setServerAddress", args, &called, self->session_id);
      if (!res)
        {
          z_policy_thread_release(self->thread);
          z_proxy_leave(self);
          return FALSE;
        }
      if (!z_policy_var_parse(res, "i", &rc) || !rc)
        {
          z_policy_thread_release(self->thread);
          z_proxy_leave(self);
          return FALSE;
        }
      z_policy_var_unref(res);
    }

  res = z_policy_call(self->handler, "connectServer", NULL, &called, self->session_id);

  if (res && z_py_zorp_stream_check(res))
    {
      self->endpoints[EP_SERVER] = ((ZorpStream *) res)->stream;
      z_stream_ref(self->endpoints[EP_SERVER]);
      rc = 1;
    }
  else
    {
      rc = 0;
    }
  z_policy_var_unref(res);
  z_policy_thread_release(self->thread);
  z_proxy_leave(self);
  return rc;
}


gint
z_proxy_user_authenticated(ZProxy *self, gchar *entity)
{
  ZPolicyObj *res;
  gboolean called;
  gint rc = 1;
  
  z_proxy_enter(self);
  z_policy_thread_acquire(self->thread);

  res = z_policy_call(self->handler, "userAuthenticated", z_policy_var_build("(s)", entity), &called, self->session_id);
  if (!res)
    rc = 0;
  z_policy_var_unref(res);
  z_policy_thread_release(self->thread);
  z_proxy_leave(self);
  return rc;
}

ZStackedProxy *
z_proxy_stack_proxy(ZProxy *self, ZPolicyObj *proxy_class)
{
  int downpair[2], uppair[2];
  ZPolicyObj *res, *client_stream, *server_stream;
  ZStackedProxy *rc;
  ZStream *tmpstream;
  
  z_proxy_enter(self);
  if (proxy_class == z_policy_none)
    return NULL;
  if (socketpair(AF_UNIX, SOCK_STREAM, 0, downpair) == -1)
    {
      /*LOG
        This message indicates that stacking a child proxy failed, because
        creating an AF_UNIX domain socketpair failed on the client side.
       */
      z_proxy_log(self, CORE_ERROR, 1, "Error creating client socketpair for stacked proxy; error='%m'");
      z_policy_var_unref(proxy_class);
      z_proxy_leave(self);
      return NULL;
    }
  else if (socketpair(AF_UNIX, SOCK_STREAM, 0, uppair) == -1)
    {
      close(downpair[0]);
      close(downpair[1]);
      /*LOG
        This message indicates that stacking a child proxy failed, because
        creating an AF_UNIX domain socketpair failed on the server side.
       */
      z_proxy_log(self, CORE_ERROR, 1, "Error creating server socketpair for stacked proxy; error='%m'");
      z_policy_var_unref(proxy_class);
      z_proxy_leave(self);
      return NULL;
    }
  /*LOG
    This debug message indicates that Zorp is about to stack a proxy class
    with the given fds as communication channels.
   */
  z_proxy_log(self, CORE_DEBUG, 6, "Stacking subproxy; client='%d:%d', server='%d:%d'", downpair[0], downpair[1], uppair[0], uppair[1]);
  
  tmpstream = z_stream_new(downpair[1], "");
  client_stream = z_py_stream_new(tmpstream);
  z_stream_unref(tmpstream);
  
  tmpstream = z_stream_new(uppair[1], "");
  server_stream = z_py_stream_new(tmpstream);
  z_stream_unref(tmpstream);
  res = z_policy_call(self->handler, "stackProxy", z_policy_var_build("(OOO)", client_stream, server_stream, proxy_class), NULL, self->session_id);
  
  z_policy_var_unref(client_stream);
  z_policy_var_unref(server_stream);
  
  if (!res)
    {
      close(downpair[0]);
      close(downpair[1]);
      close(uppair[0]);
      close(uppair[1]);
      z_policy_var_unref(res);
      z_proxy_leave(self);
      return NULL;
    }
  z_policy_var_unref(res);
  rc = z_stacked_proxy_new(downpair[0], uppair[0], self);
  
  z_proxy_leave(self);
  return rc;
}

gboolean
z_proxy_get_addresses(ZProxy *self, 
                      guint *protocol,
                      ZSockAddr **client_address, ZSockAddr **client_local,
                      ZSockAddr **server_address, ZSockAddr **server_local,
                      ZSockAddr **client_listen)
{
  ZorpSockAddr *sockaddr;
  
  z_policy_thread_acquire(self->thread);

  if (protocol)
    {
      ZPolicyObj *pyproto;
      
      pyproto = z_session_getattr(self->handler, "protocol");
      if (PyInt_Check(pyproto))
        *protocol = PyInt_AsLong(pyproto);
      else
        *protocol = ZD_PROTO_TCP;
      z_policy_var_unref(pyproto);
    }

  if (client_address)
    {
      sockaddr = (ZorpSockAddr *) z_session_getattr(self->handler, "client_address");
      if (z_py_zorp_sockaddr_check(sockaddr))
        *client_address = z_sockaddr_ref(sockaddr->sa);
      else 
        *client_address = NULL;
      z_policy_var_unref(sockaddr);
    }

  if (client_local)
    {
      sockaddr = (ZorpSockAddr *) z_session_getattr(self->handler, "client_local");
      if (z_py_zorp_sockaddr_check(sockaddr))
        *client_local = z_sockaddr_ref(sockaddr->sa);
      else 
        *client_local = NULL;
      z_policy_var_unref(sockaddr);
    }

  if (client_listen)
    {
      sockaddr = (ZorpSockAddr *) z_session_getattr(self->handler, "client_listen");
      if (z_py_zorp_sockaddr_check(sockaddr))
        *client_listen = z_sockaddr_ref(sockaddr->sa);
      else 
        *client_listen = NULL;
      z_policy_var_unref(sockaddr);
    }

  if (server_address)
    {
      sockaddr = (ZorpSockAddr *) z_session_getattr(self->handler, "server_address");
      if (z_py_zorp_sockaddr_check(sockaddr))
        *server_address = z_sockaddr_ref(sockaddr->sa);
      else
        *server_address = NULL;
      z_policy_var_unref(sockaddr);
    }

  if (server_local)
    {
      sockaddr = (ZorpSockAddr *) z_session_getattr(self->handler, "server_local");
      if (z_py_zorp_sockaddr_check(sockaddr))
        *server_local = z_sockaddr_ref(sockaddr->sa);
      else
        *server_local = NULL;
      z_policy_var_unref(sockaddr);
    }

  z_policy_thread_release(self->thread);
  return TRUE;
}

/* stacked proxy */

ZStackedProxy *
z_stacked_proxy_new(gint client_fd, gint server_fd, ZProxy *proxy)
{
  ZStackedProxy *self = g_new0(ZStackedProxy, 1);
  gchar buf[MAX_SESSION_ID];
  
  z_proxy_enter(proxy);
  z_fd_set_nonblock(client_fd, TRUE);
  z_fd_set_nonblock(server_fd, TRUE);

  g_snprintf(buf, sizeof(buf), "%s/client_downstream", proxy->session_id);
  self->downstreams[EP_CLIENT] = z_stream_new(client_fd, buf);
  
  g_snprintf(buf, sizeof(buf), "%s/server_downstream", proxy->session_id);
  self->downstreams[EP_SERVER] = z_stream_new(server_fd, buf);

  z_proxy_leave(proxy);
  return self;
}

void
z_stacked_proxy_destroy(ZStackedProxy *self)
{
  int i;

  z_enter();
  for (i = 0; i < EP_MAX; i++)
    {
      if (self->downstreams[i])
        {
          z_stream_shutdown(self->downstreams[i], SHUT_RDWR, NULL);
          z_stream_close(self->downstreams[i], NULL);
          z_stream_unref(self->downstreams[i]);
        }
    }
  g_free(self);
  z_leave();
}

