/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: pydispatch.c,v 1.6.2.13 2004/01/29 13:04:02 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor : 
 * Last audited version: 
 * Notes:
 *
 ***************************************************************************/

#include <zorp/pydispatch.h>
#include <zorp/dispatch.h>
#include <zorp/policy.h>
#include <zorp/log.h>
#include <zorp/freeq.h>
#include <zorp/pysockaddr.h>
#include <zorp/pystream.h>

struct _ZorpDispatch
{
  PyObject_HEAD
  ZPolicy *policy;
  ZPolicyThread *policy_thread;
  ZDispatchEntry *dispatch;
  gboolean threaded;
  PyObject *handler;
};

static PyObject *z_py_zorp_dispatch_new_instance(PyObject *o, PyObject *args);
static void z_py_zorp_dispatch_free(ZorpDispatch *self);

static PyObject *z_py_zorp_dispatch_getattr(PyObject *o, char *name);
static PyObject *z_py_zorp_dispatch_destroy_method(ZorpDispatch *self, PyObject *args);

PyMethodDef z_py_zorp_dispatch_funcs[] =
{
  { "Dispatch", (PyCFunction) z_py_zorp_dispatch_new_instance, METH_VARARGS, NULL },
  { NULL,      NULL, 0, NULL }   /* sentinel*/
};

PyTypeObject z_py_zorp_dispatch_type = 
{
  PyObject_HEAD_INIT(&z_py_zorp_dispatch_type)
  0,
  "ZorpDispatch",
  sizeof(ZorpDispatch),
  0,
  (destructor) z_py_zorp_dispatch_free,
  0,
  (getattrfunc) z_py_zorp_dispatch_getattr,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  "ZorpDispatch class for Zorp",
  0, 0, 0, 0
};

static PyMethodDef py_zorp_dispatch_methods[] =
{
  { "destroy",     (PyCFunction) z_py_zorp_dispatch_destroy_method, 0, NULL },
  { NULL,          NULL, 0, NULL }   /* sentinel*/
};

/*
 * Called by the main thread, so it locks using the global python state
 */
static gboolean
z_py_zorp_dispatch_accept(ZConnection *conn,
                          gpointer user_data)
{
  ZorpDispatch *self = (ZorpDispatch *) user_data;
  PyObject *res, *addr, *local, *pystream, *bound;

  z_enter();
  
  z_policy_thread_acquire(self->policy_thread);

  if (conn)
    {
      ZSockAddr *tmpsa;
     
      /* NOTE: we cloning sockaddrs here as ref/unref on sockaddrs is not
       * reentrant, thus it is wise to use separate copies in each thread */
       
      tmpsa = z_sockaddr_clone(conn->dest, FALSE);
      local = z_py_zorp_sockaddr_new(tmpsa);
      z_sockaddr_unref(tmpsa);
      
      tmpsa = z_sockaddr_clone(conn->remote, FALSE);
      addr = z_py_zorp_sockaddr_new(tmpsa);
      z_sockaddr_unref(tmpsa);
      
      tmpsa = z_sockaddr_clone(conn->bound, FALSE);
      bound = z_py_zorp_sockaddr_new(tmpsa);
      z_sockaddr_unref(tmpsa);
   
      pystream = z_py_stream_new(conn->stream);
    }
  else
    {
      local = Py_None;
      addr = Py_None;
      bound = Py_None;
      pystream = Py_None;
      
      Py_XINCREF(local);
      Py_XINCREF(addr);
      Py_XINCREF(bound);
      Py_XINCREF(pystream);
      
    }
  res = PyEval_CallFunction(self->handler, "(OOOO)",
			    pystream, addr, local, bound);
  Py_XDECREF(bound);
  Py_XDECREF(addr);
  Py_XDECREF(local);
  Py_XDECREF(pystream);
  
  /* once python was called we assume that it takes care about the fd
   * we just passed. As an exception if an exception occurs we close it ourselves
   */
  if (!res)
    {
      PyErr_Print();
      if (conn)
        {
          z_stream_close(conn->stream, NULL);
        }
    }
  else if (res == Py_None)
    {
      gchar buf[256];
      /*LOG
	This message indicates that the decision layer denied the
	given connection.
      */
      z_log(NULL, CORE_POLICY, 1, "Connection denied by policy; %s", z_connection_format(conn, buf, sizeof(buf)));
      /* close(fd); */
    }
  Py_XDECREF(res);

  z_policy_thread_release(self->policy_thread);
  if (conn)
    z_connection_destroy(conn, FALSE);
  z_leave();
  return TRUE;
}

static PyObject *
z_py_zorp_dispatch_new_instance(PyObject *o G_GNUC_UNUSED, PyObject *args)
{
  ZorpDispatch *self;
  ZorpSockAddr *addr;
  PyObject *handler, *keywords, *fake_args;
  gint protocol, prio;
  gchar buf[MAX_SOCKADDR_STRING], *session_id;
  ZDispatchParams params;
  gchar *tcp_keywords[] = { "accept_one", "backlog", "threaded", NULL };
  gchar *udp_keywords[] = { "tracker", "session_limit", "threaded", NULL };

  /* called by python, so interpreter is locked */
  
  if (current_policy == NULL)
    {
      PyErr_SetString(PyExc_RuntimeError, "Parsing phase has not completed yet, Listener & Receiver must be defined in the instance init() function.");
      return NULL;
    }
  
  /* res is a borrowed reference, no need to unref it */
  if (!PyArg_ParseTuple(args, "siOiOO", &session_id, &protocol, &addr, &prio, &handler, &keywords))
    return NULL;
  
  fake_args = PyTuple_New(0);
  switch (protocol)
    {
    case ZD_PROTO_TCP:
      params.tcp.common.threaded = FALSE;
      params.tcp.accept_one = FALSE;
      params.tcp.backlog = 255;
      if (!PyArg_ParseTupleAndKeywords(fake_args, keywords, "|iii", tcp_keywords, 
                                       &params.tcp.accept_one, 
                                       &params.tcp.backlog, 
                                       &params.tcp.common.threaded))
        {
          Py_XDECREF(fake_args);
          return NULL;
        }
      break;
    case ZD_PROTO_UDP:

      /* NOTE: params.udp.tracker is a (gchar *) valid only as long as the
       * z_dispatch_register calls returns, it is discarded by Python
       * afterwards.  This is not a problem as this name is used in
       * z_conntrack_new, and never referenced again */
       
      params.udp.common.threaded = FALSE;
      params.udp.tracker = NULL;
      params.udp.session_limit = 10;
      if (!PyArg_ParseTupleAndKeywords(fake_args, keywords, "|sii", udp_keywords, 
                                       &params.udp.tracker, 
                                       &params.udp.session_limit,
                                       &params.udp.common.threaded))
        {
          Py_XDECREF(fake_args);
          return NULL;
        }
      break;
      
    }
  Py_XDECREF(fake_args);
  
  if (!z_py_zorp_sockaddr_check(addr))
    {
      PyErr_SetString(PyExc_TypeError, "addr parameter must be a ZorpSockAddr");
      return NULL;
    }
  if (!PyCallable_Check(handler))
    {
      PyErr_SetString(PyExc_TypeError, "Handler parameter must be callable");
      return NULL;
    }

  /*LOG
    This message indicates that a Listener on the given local address is
    started.
   */
  z_log(session_id, CORE_DEBUG, 7, "Dispatcher on address; proto='%d', local='%s', prio='%d'", 
        protocol, z_sockaddr_format(addr->sa, buf, sizeof(buf)), prio);

  self = PyObject_NEW(ZorpDispatch, &z_py_zorp_dispatch_type);
  if (!self)
    return NULL;
    
  Py_XINCREF(self);
  self->handler = handler;
  Py_XINCREF(handler);
  self->policy = NULL;
  self->policy = z_policy_ref(current_policy);
  self->threaded = ((ZDispatchCommonParams *) &params)->threaded;

  self->policy_thread = z_policy_thread_new(self->policy);
  z_policy_thread_ready(self->policy_thread);
  
  self->dispatch = z_dispatch_register(session_id, protocol, addr->sa, NULL, prio, &params, z_py_zorp_dispatch_accept, self, NULL);
  if (!self->dispatch)
    {
      Py_XDECREF(self);
      Py_XDECREF(self);
      PyErr_SetString(PyExc_IOError, "Error binding to interface");
      return NULL;
    }
  return (PyObject *) self;
}

static void
z_py_zorp_dispatch_free(ZorpDispatch *self)
{
  if (self->handler)
    {
      Py_XDECREF(self->handler);
      self->handler = NULL;
    }
  if (self->dispatch)
    {
      z_dispatch_unregister(self->dispatch);
      self->dispatch = NULL;
    }
  if (self->policy_thread)
    {
      Py_BEGIN_ALLOW_THREADS;
      /* python must be unlocked */
      z_policy_thread_destroy(self->policy_thread);
      Py_END_ALLOW_THREADS;
      self->policy_thread = NULL;
    }
  PyMem_DEL(self);
}

static PyObject *
z_py_zorp_dispatch_getattr(PyObject *o, char *name)
{
  return Py_FindMethod(py_zorp_dispatch_methods, o, name);
}


static void
z_py_zorp_dispatch_freer(gpointer p)
{
  ZorpDispatch *self = (ZorpDispatch *) p;
  ZPolicy *policy;
  
  policy = z_policy_ref(self->policy);
  
  z_policy_acquire_main(policy);
  Py_XDECREF(self);
  z_policy_release_main(policy);
}

static PyObject *
z_py_zorp_dispatch_destroy_method(ZorpDispatch *self, PyObject *args G_GNUC_UNUSED)
{
  if (self->dispatch)
    {
      z_dispatch_unregister(self->dispatch);
      self->dispatch = NULL;
    }
  Py_XDECREF(self->handler);
  self->handler = NULL;
  z_free_queue_add(self, z_py_zorp_dispatch_freer);
  Py_XINCREF(Py_None);
  return Py_None;
}

void
z_py_zorp_dispatch_init(void)
{
  Py_InitModule("Zorp.Zorp", z_py_zorp_dispatch_funcs);
}
