/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: pyfastpath.c,v 1.3.2.5 2003/09/25 15:06:44 bazsi Exp $
 *
 * Author  : yeti, bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/zpython.h>
#include <zorp/pyproxy.h>
#include <zorp/proxy.h>
#include <zorp/conntrack.h>
#include <zorp/stream.h>
#include <zorp/log.h>
#include <zorp/pysockaddr.h>
#include <zorp/sockaddr.h>
#include <zorp/attach.h>


typedef struct _ZFPDirectedRouter
{
  gboolean forge_addr;
  ZSockAddr *addr;
} ZFPDirectedRouter;

/**
 * z_fp_directed_router_route:
 * @session: connection to be routed
 * @data: router instance
 *
 * Do DirectedRouter's job on the fast path - that is, set the
 * destination address to a fixed value.
 */
 
static void 
z_fp_directed_router_route(ZProxyFastpathSession *session, gpointer data)
{
  ZFPDirectedRouter *self = (ZFPDirectedRouter *) data;

  z_sockaddr_unref(session->server_address);
  session->server_address = z_sockaddr_clone(self->addr, FALSE);
  if (self->forge_addr)
    {
      z_sockaddr_unref(session->server_local);
      session->server_local = z_sockaddr_clone(session->client_address, TRUE);
    }
}

/**
 * z_fp_directed_router_free:
 * @data: router instance
 *
 * free the router instance.
 */

static void 
z_fp_directed_router_free(gpointer data)
{
  ZFPDirectedRouter *self = (ZFPDirectedRouter *) data;

  z_sockaddr_unref(self->addr);
  g_free(self);
}

/**
 * z_py_router_directed_setup:
 * @s: unused
 * @args: Python argument list
 *
 * Create a fast path DirectedRouter instance and bind it to the proxy
 * specified in the argument list. Should be called from python with
 * three args: the Proxy object, a SockAddr of the destination and an
 * integer that specifies if address forging should be done (done if
 * arg is nonzero)
 *
 * Returns: Py_None, the Python None object.
 */
 
PyObject *
z_fp_directed_router_pysetup(PyObject *s G_GNUC_UNUSED, PyObject *args)
{
  ZFPDirectedRouter *self;
  ZorpProxy *pyproxy;
  ZorpSockAddr *addr;
  ZProxy *proxy;
  guint forge_addr;
  
  z_enter();

  if (!PyArg_ParseTuple(args, "OOi", &pyproxy, &addr, &forge_addr))
    {
      z_leave();
      return NULL;
    }
#if 0
  if (!z_py_zorp_proxy_check((PyObject *) pyproxy))
    {
      PyErr_SetString(PyExc_TypeError, "Proxy class required");
      return NULL;
    }
#endif
  if (!z_py_zorp_sockaddr_check(addr))
    {
      PyErr_SetString(PyExc_TypeError, "SockAddr type required");
      return NULL;
    }

  self = g_new0(ZFPDirectedRouter, 1);
  
  self->addr = z_sockaddr_ref(addr->sa);
  self->forge_addr = !!forge_addr;
  
  proxy = pyproxy->proxy;
  proxy->fastpath.router = z_fp_directed_router_route;
  proxy->fastpath.router_free = z_fp_directed_router_free;
  proxy->fastpath.router_data = self;

  z_leave();
  return PyInt_FromLong(0);
}

typedef struct _ZFPTransparentRouter
{
  gboolean forge_addr;
  guint forced_port;
} ZFPTransparentRouter;

/**
 * z_fp_transparent_router_route:
 * @session: connection to be routed
 * @data: router instance
 *
 * Do DirectedRouter's job on the fast path - that is, set the
 * destination address to a fixed value.
 */
 
static void 
z_fp_transparent_router_route(ZProxyFastpathSession *session, gpointer data)
{
  ZFPTransparentRouter *self = (ZFPTransparentRouter *) data;

  z_sockaddr_unref(session->server_address);
  session->server_address = z_sockaddr_clone(session->client_local, FALSE);
  if (self->forge_addr)
    {
      z_sockaddr_unref(session->server_local);
      session->server_local = z_sockaddr_clone(session->client_address, TRUE);
    }
  if (self->forced_port)
    {
      if (z_sockaddr_inet_check(session->server_address))
        {
          ((struct sockaddr_in *) &session->server_address->sa)->sin_port = htons(self->forced_port);
        }
      else
        {
          z_log(NULL, CORE_POLICY, 1, "Non-IPv4 addresses are not supported on fastpath;");
        }
    }
}

/**
 * z_fp_transparent_router_pysetup:
 * @s: unused
 * @args: Python argument list
 *
 * Returns: Py_None, the Python None object.
 */
 
PyObject *
z_fp_transparent_router_pysetup(PyObject *s G_GNUC_UNUSED, PyObject *args)
{
  ZFPTransparentRouter *self;
  ZorpProxy *pyproxy;
  ZProxy *proxy;
  guint forge_addr, forced_port;
  
  z_enter();

  if (!PyArg_ParseTuple(args, "Oii", &pyproxy, &forced_port, &forge_addr))
    {
      z_leave();
      return NULL;
    }
  if (z_py_zorp_proxy_check((PyObject *) pyproxy))
    {
      PyErr_SetString(PyExc_TypeError, "Proxy class required");
      return NULL;
    }

  self = g_new0(ZFPTransparentRouter, 1);
  
  self->forced_port = forced_port;
  self->forge_addr = !!forge_addr;
  
  proxy = pyproxy->proxy;
  proxy->fastpath.router = z_fp_transparent_router_route;
  proxy->fastpath.router_free = g_free;
  proxy->fastpath.router_data = self;

  z_leave();
  return PyInt_FromLong(0);
}

typedef struct _ZFPConnectChainer
{
  gchar session_id[MAX_SESSION_ID];
#if ENABLE_CONNTRACK
  gchar tracker_name[MAX_TRACKER_NAME];
#endif
  guint protocol;
} ZFPConnectChainer;

/**
 * z_fastpath_chainer_send_func:
 * @session: connection to be chained
 * @data: the chainer instance
 *
 * SendChainer implementation for the fast path - create a #ZCTsocket
 * handled by the CT specified upon chainer creation.
 */
 
ZStream * 
z_fp_connect_chainer_chain(ZProxyFastpathSession *session, gpointer data)
{
  ZFPConnectChainer *self = (ZFPConnectChainer *) data;
  ZAttach *attach;
  ZAttachParams params;
  ZStream *stream = NULL;
  ZConnection *new_conn;
  
  switch (self->protocol)
    {
    case ZD_PROTO_TCP:
      params.tcp.timeout = 30;
      break;
#if ENABLE_CONNTRACK
    case ZD_PROTO_UDP:
      params.udp.tracker = self->tracker_name;
      break;
#endif
    }
  attach = z_attach_new(self->session_id, self->protocol, session->server_local, session->server_address, &params, NULL, NULL, NULL);
  if (!attach)
    return NULL;
  
  if (!z_attach_start(attach))
    new_conn = NULL;
  else
    new_conn = z_attach_block(attach);
  z_attach_unref(attach);
  
  if (new_conn)
    {
      z_stream_ref(new_conn->stream);
      stream = new_conn->stream;
      
      z_sockaddr_unref(session->server_local);
      session->server_local = z_sockaddr_ref(new_conn->bound);
      
      z_connection_destroy(new_conn, FALSE);
    }
  
  return stream;
}

/**
 * z_fp_connect_chainer_pysetup:
 * @self: unused
 * @args: Python argument list
 *
 * Create a fast path SendChainer instance and attach it to the proxy
 * specified in Python. 
 *
 * Returns: Py_None (None at Python level).
 */
 
PyObject *
z_fp_connect_chainer_pysetup(PyObject *s G_GNUC_UNUSED, PyObject *args)
{
  ZorpProxy *pyproxy;
  ZProxy *proxy;
  gchar *session_id, *tracker_name;
  guint protocol;
  ZFPConnectChainer *self;
  
  z_enter();

  if (!PyArg_ParseTuple(args, "Osis", &pyproxy, &session_id, &protocol, &tracker_name))
    {
      z_leave();
      return NULL;
    }

  if (z_py_zorp_proxy_check((PyObject *) pyproxy))  
    {
      PyErr_SetString(PyExc_TypeError, "Proxy class required");
      return NULL;
    }

  self = g_new0(ZFPConnectChainer, 1);
  
  g_strlcpy(self->session_id, session_id, sizeof(self->session_id));
#if ENABLE_CONNTRACK
  g_strlcpy(self->tracker_name, tracker_name, sizeof(self->tracker_name));
#endif
  self->protocol = protocol;

  proxy = pyproxy->proxy;
  proxy->fastpath.chainer = z_fp_connect_chainer_chain;
  proxy->fastpath.chainer_free = g_free;
  proxy->fastpath.chainer_data = self;

  z_leave();
  return PyInt_FromLong(0);
}

static PyMethodDef fastpath_funcs[] =
{
  { "setupDirectedRouter", z_fp_directed_router_pysetup, METH_VARARGS, 0 },
  { "setupTransparentRouter", z_fp_transparent_router_pysetup, METH_VARARGS, 0 },
  { "setupConnectChainer", z_fp_connect_chainer_pysetup, METH_VARARGS, 0 },
  { NULL, NULL, 0, 0 }
};

/**:
 * z_py_zorp_fastpath_init:
 *
 * Initialize Python level fast path functions. Called from
 * z_python_init().
 */

void z_py_zorp_fastpath_init(void)
{
  Py_InitModule("Zorp.Zorp", fastpath_funcs);
}
