/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: pypolicy.c,v 1.36.2.15 2004/02/03 09:16:28 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor : kisza
 * Last audited version: 1.20
 * Notes:
 *   - comments
 *
 ***************************************************************************/

/* 
 * This is an implementation of the policy interface with the python
 * interpreter 
 */

#include <zorp/zpython.h>
#include <zorp/policy.h>
#include <zorp/log.h>

/* for python initialization functions */
#include <zorp/pycore.h>
#include <zorp/pyszig.h>
#include <zorp/pydispatch.h>
#include <zorp/pyattach.h>
#include <zorp/pystream.h>
#include <zorp/pyproxy.h>
#include <zorp/pysockaddr.h>
#include <zorp/pyfastpath.h>

ZPolicy *current_policy;

/* helper functions used by proxies */

gint 
z_policy_setattr(PyObject *handler, char *name, PyObject *value)
{
  gint res;
  
  res = PyObject_SetAttrString(handler, name, value);
  Py_XDECREF(value);
  return res;
}

PyObject *
z_policy_getattr(PyObject *handler, char *name)
{
  PyObject *res;
  
  res = PyObject_GetAttrString(handler, name);
  return res;
}

PyObject *
z_session_getattr(PyObject *handler, char *name)
{
  PyObject *res, *res2;
  
  res = PyObject_GetAttrString(handler, "session");
  if (!res)
    return NULL;
  res2 = PyObject_GetAttrString(res, name);
  Py_XDECREF(res);
  return res2;
}

PyObject *
z_policy_call_object(PyObject *func, PyObject *args, gchar *session_id)
{
  PyObject *res;
  
  PyErr_Clear();
  res = PyObject_CallObject(func, args);
  
  Py_XDECREF(args);
  
  if (!res)
    {
      PyObject *m = PyImport_AddModule("sys");
      
      if (!PyString_Check(PyErr_Occurred()))
        {
          PyErr_Print();
        }
      else
        {
          PyObject *exc, *value, *tb, *str;
          
          PyErr_Fetch(&exc, &value, &tb);
          PyErr_NormalizeException(&exc, &value, &tb);
          
          str = PyObject_Str(value);
          if (!str)
            {
              /*NOLOG*/
              z_log(session_id, CORE_ERROR, 3, "%s;", PyString_AsString(exc));
            }
          else
            {
              /*NOLOG*/
              z_log(session_id, CORE_ERROR, 3, "%s; reason='%s'", PyString_AsString(exc), PyString_AsString(str));
            }
          
          Py_XDECREF(exc);
          Py_XDECREF(value);
          Py_XDECREF(tb);
          Py_XDECREF(str);
        }
      PyObject_SetAttrString(m, "last_traceback", Py_None); 
    }

  return res;
}

PyObject *
z_policy_call(PyObject *handler, char *name, PyObject *args, gboolean *called, gchar *session_id)
{
  PyObject *attr;
  PyObject *res;

  z_enter();
  g_assert(PyThreadState_GET());
  attr = PyObject_GetAttrString(handler, name);
  if (!attr || !PyCallable_Check(attr))
    {
      if (attr)
        {
          Py_XDECREF(attr);
          PyErr_Format(PyExc_TypeError, "Event must be callable: %s", name);
          PyErr_Print(); /* produce a backtrace, and handle it immediately */
        }
      PyErr_Clear();
      Py_XDECREF(args);
      res = NULL;
      if (called)
        *called = FALSE;
    }
  else
    {
      if (called)
        *called = TRUE;
      res = z_policy_call_object(attr, args, session_id);
      Py_XDECREF(attr);
    }
  z_leave();
  return res;
}

gint 
z_policy_event(PyObject *handler, char *name, PyObject *args, gchar *session_id)
{
  PyObject *res;
  unsigned long c_res;
  gboolean called;

  Py_XINCREF(args);
  res = z_policy_call(handler, "preProcessEvent", args, &called, session_id);
  if (res)
    {
      if (PyInt_Check(res))
	{
	  c_res = PyInt_AsLong(res);
	  Py_XDECREF(res);
	  if (c_res != Z_UNSPEC)
	    {
	      Py_XDECREF(args);
	      return c_res;
	    }
	}
      else
	{
	  PyErr_Format(PyExc_TypeError, "preProcessEvent() handlers should return an int.");
	  PyErr_Print(); /* produce a backtrace, and handle it immediately */
	  Py_XDECREF(res);
	}
    }
  else
    if (called)
      return Z_ABORT;
  Py_XINCREF(args);
  res = z_policy_call(handler, name, args, &called, session_id);
  if (res)
    {
      if (PyInt_Check(res))
	{
	  c_res = PyInt_AsLong(res);
	  Py_XDECREF(res);
	  if (c_res != Z_UNSPEC)
	    {
	      Py_XDECREF(args);
	      return c_res;
	    }
	}
      else
	{
	  PyErr_Format(PyExc_TypeError, "Event handlers should return an int: %s", name);
	  PyErr_Print(); /* produce a backtrace, and handle it immediately */
	  Py_XDECREF(res);
	}
    }
  else
    if (called)
      return Z_ABORT;
  res = z_policy_call(handler, "postProcessEvent", args, &called, session_id);
  if (res)
    {
      if (PyInt_Check(res))
	{
	  c_res = PyInt_AsLong(res);
	  Py_XDECREF(res);
	  return c_res;
	}
      else
	{
	  PyErr_Format(PyExc_TypeError, "postProcessEvent() handlers should return an int.");
	  PyErr_Print(); /* produce a backtrace, and handle it immediately */
	  Py_XDECREF(res);
	}
    }
  else
    if (called)
      return Z_ABORT;
  return Z_UNSPEC;
}

/* loadable policies */

GStaticMutex policy_ref_lock = G_STATIC_MUTEX_INIT;

struct _ZPolicy
{
  gint ref_cnt;
  gchar *policy_filename;
  ZPolicyThread *main_thread;
};

struct _ZPolicyThread
{
  ZPolicy *policy;
  PyThreadState *thread;
  /* thread startup synchronization */
  gboolean startable;
  GMutex   *startable_lock;
  GCond    *startable_signal;
};

GStaticPrivate policy_thread = G_STATIC_PRIVATE_INIT;

void
z_policy_thread_ready(ZPolicyThread *self)
{
  g_mutex_lock(self->startable_lock);
  self->startable = TRUE;
  g_cond_signal(self->startable_signal);
  g_mutex_unlock(self->startable_lock);
}

static void
z_policy_thread_wait(ZPolicyThread *self)
{
  g_mutex_lock(self->startable_lock);
  while (!self->startable)
    {
      g_cond_wait(self->startable_signal, self->startable_lock);
    }
  g_mutex_unlock(self->startable_lock);
}

void
z_policy_thread_acquire(ZPolicyThread *self)
{
  z_policy_thread_wait(self);
  
  g_static_private_set(&policy_thread, self, NULL);
  PyEval_AcquireThread(self->thread);
}

void
z_policy_thread_release(ZPolicyThread *self)
{
  PyEval_ReleaseThread(self->thread);
  g_static_private_set(&policy_thread, NULL, NULL);
}

ZPolicyThread *
z_policy_thread_self(void)
{
  return g_static_private_get(&policy_thread);
}

ZPolicy *
z_policy_thread_get_policy(ZPolicyThread *self)
{
  return self->policy;
}

ZPolicyThread *
z_policy_thread_new(ZPolicy *policy)
{
  ZPolicyThread *self = g_new0(ZPolicyThread, 1);
  
  /* NOTE: requires the interpreter lock to be held */
  self->startable = FALSE;
  self->startable_lock = g_mutex_new();
  self->startable_signal = g_cond_new();
  
  self->policy = z_policy_ref(policy);
  if (policy->main_thread)
    {
      self->thread = PyThreadState_New(self->policy->main_thread->thread->interp);
    }
  else
    {
      /* initialize a new interpreter instance */
      self->thread = Py_NewInterpreter();
      PyThreadState_Swap(NULL);
    }
  return self;
}

void
z_policy_thread_destroy(ZPolicyThread *self)
{
  /* acquires the interpreter lock */

  if (self->policy->main_thread != self)
    {
      /* we are one of the secondary threads */
      z_python_lock();
      PyThreadState_Clear(self->thread);
      PyThreadState_Delete(self->thread);
      z_python_unlock();
    }
  g_mutex_free(self->startable_lock);
  g_cond_free(self->startable_signal);
  g_free(self);
}

ZPolicy * 
z_policy_ref(ZPolicy *self)
{
  return self;
}

gboolean
z_policy_boot(ZPolicy *self)
{
  FILE *bootstrap;
  
  if ((bootstrap = fopen(ZORP_POLICY_BOOT_FILE, "r")) == NULL)
    {
      /*LOG
        This message indicates that the policy.boot file couldn't be opened for
        reading.
       */
      z_log(NULL, CORE_ERROR, 0, "Error opening policy.boot file; file='%s'", ZORP_POLICY_BOOT_FILE);
      return FALSE;
    }

  z_policy_thread_acquire(self->main_thread);

  PyRun_SimpleFile(bootstrap, ZORP_POLICY_BOOT_FILE);
  fclose(bootstrap);

  /* PyImport_AddModule("Zorp.Zorp"); */
  
  z_py_zorp_core_init();
  z_py_zorp_szig_init();
  z_py_zorp_dispatch_init();
  z_py_zorp_attach_init();
  z_py_zorp_stream_init();
  z_py_zorp_proxy_init();
  z_py_zorp_sockaddr_init();

  z_py_zorp_fastpath_init();
  z_policy_thread_release(self->main_thread);
  
  return TRUE;
}

gboolean
z_policy_load(ZPolicy *self)
{
  FILE *script;
  int res = -1;
  
  script = fopen(self->policy_filename, "r");
  if (script)
    {
      z_policy_thread_acquire(self->main_thread);
      res = PyRun_SimpleFile(script, self->policy_filename);
      fclose(script);
      z_policy_thread_release(self->main_thread);
    }
  else
    {
      z_log(NULL, CORE_ERROR, 0, "Error opening policy file; filename='%s'", self->policy_filename);
    }
  
  if (res == -1)
    {
      z_log(NULL, CORE_ERROR, 0, "Error parsing policy file; filename='%s'", self->policy_filename);
      /* let the error message out */
    }

  return res != -1;
}


/* the function interface (not the implementation) here should be
   independent of python */

gboolean
z_policy_init(ZPolicy *self, const gchar *instance_name)
{
  PyObject *main_module, *init_func, *res;
  
  z_policy_thread_acquire(self->main_thread);
  
  main_module = PyImport_AddModule("__main__");
  init_func = PyObject_GetAttrString(main_module, "init");

  res = PyObject_CallFunction(init_func, "(s)", instance_name);
  Py_XDECREF(init_func);
  if (!res)
    {
      PyErr_Print();
    }
  Py_XDECREF(res);
  z_policy_thread_release(self->main_thread);

  return res != NULL;
}

void
z_policy_acquire_main(ZPolicy *self)
{
  z_policy_thread_acquire(self->main_thread);
}

void
z_policy_release_main(ZPolicy *self)
{
  z_policy_thread_release(self->main_thread);
}

ZPolicy *
z_policy_new(const gchar *filename)
{
  ZPolicy *self = g_new0(ZPolicy, 1);
  
  
  self->ref_cnt = 1;
  self->policy_filename = g_strdup(filename);
  z_python_lock();
  self->main_thread = z_policy_thread_new(self);
  z_python_unlock();
  z_policy_thread_ready(self->main_thread);
  
  return self;
}

void 
z_policy_raise_exception(gchar *exception_name, gchar *desc)
{
  PyObject *main_module, *license_exc;
  
  main_module = PyImport_AddModule("__main__");
  license_exc = PyObject_GetAttrString(main_module, exception_name);
  PyErr_SetString(license_exc, desc);
  Py_XDECREF(license_exc);
}

