/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: pysockaddr.c,v 1.30.2.13 2003/10/29 09:47:23 sasa Exp $
 *
 * Author  : bazsi
 * Auditor : kisza
 * Last audited version: 1.4
 * Notes:
 *
 ***************************************************************************/

#include <zorp/pysockaddr.h>
#include <zorp/log.h>

#include <netdb.h>
#include <arpa/inet.h>

extern PyTypeObject z_py_zorp_sockaddr_type;

static PyObject *
z_py_ntohl(PyObject *self G_GNUC_UNUSED, PyObject *args)
{
  long i;
  
  if (!PyArg_ParseTuple(args, "i", &i))
    return NULL;
    
  return PyInt_FromLong(ntohl(i));
}

static PyObject *
z_py_htonl(PyObject *self G_GNUC_UNUSED, PyObject *args)
{
  int i;
  
  if (!PyArg_ParseTuple(args, "i", &i))
    return NULL;
  return PyInt_FromLong(htonl(i));
}

#if ENABLE_IPV6
static PyObject *
z_py_inet_ntop(PyObject *self, PyObject *args)
{
  long i;
  PyObject *x, *res = NULL;
  char buf[128];
  
  z_enter();
  
  if (!PyArg_ParseTuple(args, "iO", &i, &x))
    {
      z_leave();
      return NULL;
    }
    
  switch (i)
    {
    case AF_INET:
      break;
    case AF_INET6:
      {
        struct in6_addr in6;
        
        if (!PyArg_Parse(x, "(iiiiiiii)", 
                         &in6.s6_addr16[0],
                         &in6.s6_addr16[1],
                         &in6.s6_addr16[2],
                         &in6.s6_addr16[3],
                         &in6.s6_addr16[4],
                         &in6.s6_addr16[5],
                         &in6.s6_addr16[6],
                         &in6.s6_addr16[7]))
          {
            PyErr_SetString(PyExc_ValueError, "Bad IPv6 address network representation.");
            break;
          }
        inet_ntop(AF_INET6, &in6, buf, sizeof(buf));
        res = PyString_FromString(buf);
        break;
      }
    }
  z_leave();
  return res;
}

static PyObject *
z_py_inet_pton(PyObject *self, PyObject *args)
{
  long i;
  char *addr;
  PyObject *res = NULL;
  
  z_enter();
  if (!PyArg_ParseTuple(args, "is", &i, &addr))
    {
      z_leave();
      return NULL;
    }
  
  switch (i)
    {
    case AF_INET:
      break;
    case AF_INET6:
      {
        struct in6_addr in6;
        
        if (!inet_pton(AF_INET6, addr, &in6))
          {
            PyErr_SetString(PyExc_ValueError, "Error parsing IPv6 address.");
            break;
          }
        res = Py_BuildValue("(iiiiiiii)", 
			    in6.s6_addr16[0],
			    in6.s6_addr16[1],
			    in6.s6_addr16[2],
			    in6.s6_addr16[3],
			    in6.s6_addr16[4],
			    in6.s6_addr16[5],
			    in6.s6_addr16[6],
			    in6.s6_addr16[7]);
        break;
      }
    }
  z_leave();
  return res;
}
#endif

#if HAVE_GETHOSTBYNAME_R && !HAVE_LEAK_IN_GETHOSTBYNAME_R

static PyObject *
z_py_gethostbyname(PyObject *self G_GNUC_UNUSED, PyObject *args)
{
  char *hostname;
  struct hostent hes, *he;
  char hostbuf[1024];
  char buf[32];
  int err = 0;
  int rc;
  
  if (!PyArg_ParseTuple(args, "s", &hostname))
    return NULL;
  
  Py_BEGIN_ALLOW_THREADS
#if HAVE_SUN_GETHOSTBYNAME_R
  he = gethostbyname_r(hostname, &hes, hostbuf, sizeof(hostbuf), &rc);
  err = rc;
#else
  rc = gethostbyname_r(hostname, &hes, hostbuf, sizeof(hostbuf), &he, &err);
#endif
  Py_END_ALLOW_THREADS
  if (rc == 0 && he)
    {
      z_inet_ntoa(buf, sizeof(buf), *((struct in_addr *) he->h_addr));
      return PyString_FromString(buf);
    }
  else
    {
      PyErr_SetString(PyExc_IOError, "Error resolving hostname.");
      return NULL;
    }
}

#else

static PyObject *
z_py_gethostbyname(PyObject *self G_GNUC_UNUSED, PyObject *args)
{
  char *hostname;
  struct hostent *he;
  char buf[32];
  static GStaticMutex lock = G_STATIC_MUTEX_INIT;
  PyObject *res;
  
  if (!PyArg_ParseTuple(args, "s", &hostname))
    return NULL;
  
  Py_BEGIN_ALLOW_THREADS;
  g_static_mutex_lock(&lock);
  he = gethostbyname(hostname);
  if (he)
    {
      z_inet_ntoa(buf, sizeof(buf), *((struct in_addr *) he->h_addr));
    }
  g_static_mutex_unlock(&lock);
  Py_END_ALLOW_THREADS;
  if (he)
    {
      res = PyString_FromString(buf);
    }
  else
    {
      PyErr_SetString(PyExc_IOError, "Error resolving hostname.");
      res = NULL;
    }
    
  return res;
}
#endif

PyObject *
z_py_zorp_sockaddr_new(ZSockAddr *sa)
{
  ZorpSockAddr *self;

  if (!sa)
    {
      PyErr_SetString(PyExc_ValueError, "z_py_zorp_sockaddr_new, sa is NULL");
      return NULL;
    }
  self = PyObject_NEW(ZorpSockAddr, &z_py_zorp_sockaddr_type);
  self->sa = z_sockaddr_ref(sa);
  return (PyObject *) self;
}

static PyObject *
z_py_zorp_sockaddr_inet_new_instance(PyObject *s G_GNUC_UNUSED, PyObject *args)
{
  ZorpSockAddr *self;
  gchar *ip;
  gint port;
  
  if (!PyArg_Parse(args, "(si)", &ip, &port))
    return NULL;
  
  self = PyObject_NEW(ZorpSockAddr, &z_py_zorp_sockaddr_type);
  self->sa = z_sockaddr_inet_new(ip, port);
  if (!self->sa)
    {
      Py_XDECREF(self);
      PyErr_SetString(PyExc_ValueError, "Invalid IP address");
      return NULL;
    }
  return (PyObject *) self;
}

static PyObject *
z_py_zorp_sockaddr_inet_range_new_instance(PyObject *s G_GNUC_UNUSED, PyObject *args)
{
  ZorpSockAddr *self;
  gchar *ip;
  gint min_port, max_port;
  
  if (!PyArg_Parse(args, "(sii)", &ip, &min_port, &max_port))
    return NULL;
  
  self = PyObject_NEW(ZorpSockAddr, &z_py_zorp_sockaddr_type);
  self->sa = z_sockaddr_inet_range_new(ip, min_port, max_port);
  if (!self->sa)
    {
      Py_XDECREF(self);
      PyErr_SetString(PyExc_ValueError, "Invalid IP address");
      return NULL;
    }
  return (PyObject *) self;
}

#if ENABLE_IPV6
static PyObject *
z_py_zorp_sockaddr_inet6_new_instance(PyObject *s, PyObject *args)
{
  ZorpSockAddr *self;
  gchar *ip;
  gint port;
  
  if (!PyArg_Parse(args, "(si)", &ip, &port))
    return NULL;
  
  self = PyObject_NEW(ZorpSockAddr, &z_py_zorp_sockaddr_type);
  self->sa = z_sockaddr_inet6_new(ip, port);
  if (!self->sa)
    {
      Py_XDECREF(self);
      PyErr_SetString(PyExc_ValueError, "Invalid IPv6 address");
      return NULL;
    }
  return (PyObject *) self;
}
#endif

static PyObject *
z_py_zorp_sockaddr_unix_new_instance(PyObject *s G_GNUC_UNUSED, PyObject *args)
{
  ZorpSockAddr *self;
  gchar *path;
  
  if (!PyArg_Parse(args, "(s)", &path))
    return NULL;
  
  self = PyObject_NEW(ZorpSockAddr, &z_py_zorp_sockaddr_type);
  self->sa = z_sockaddr_unix_new(path);
  return (PyObject *) self;
}

static void
z_py_zorp_sockaddr_free(ZorpSockAddr *self)
{
  z_sockaddr_unref(self->sa);
  PyMem_DEL(self);
}

static PyObject *
z_py_zorp_sockaddr_format(ZorpSockAddr *self)
{
  char buf[MAX_SOCKADDR_STRING];
  
  return PyString_FromString(z_sockaddr_format(self->sa, buf, sizeof(buf)));
}

static PyObject *
z_py_zorp_sockaddr_clone(ZorpSockAddr *self, PyObject *args)
{
  PyObject *res;
  gint wild;
  ZSockAddr *a;
  
  if (!PyArg_Parse(args, "(i)", &wild))
    return NULL;
      
  a = z_sockaddr_clone(self->sa, wild);
  res = z_py_zorp_sockaddr_new(a);
  z_sockaddr_unref(a);
  return res;
}

static PyMethodDef py_zorp_sockaddr_methods[] =
{
  { "format",      (PyCFunction) z_py_zorp_sockaddr_format, 0, NULL },
  { "clone",	   (PyCFunction) z_py_zorp_sockaddr_clone, METH_VARARGS, NULL },
  { NULL,          NULL, 0, NULL }   /* sentinel*/
};

PyMethodDef z_py_zorp_sockaddr_funcs[] =
{
  { "SockAddrInet",  z_py_zorp_sockaddr_inet_new_instance, METH_VARARGS, NULL },
  { "SockAddrInetRange",  z_py_zorp_sockaddr_inet_range_new_instance, METH_VARARGS, NULL },
#if ENABLE_IPV6
  { "SockAddrInet6", z_py_zorp_sockaddr_inet6_new_instance, METH_VARARGS, NULL },
#endif
  { "SockAddrUnix",  z_py_zorp_sockaddr_unix_new_instance, METH_VARARGS, NULL },
  { "ntohl", z_py_ntohl, METH_VARARGS, NULL },
  { "htonl", z_py_htonl, METH_VARARGS, NULL },
#if ENABLE_IPV6
  { "inet_ntop", z_py_inet_ntop, METH_VARARGS, NULL },
  { "inet_pton", z_py_inet_pton, METH_VARARGS, NULL },
#endif
  { "getHostByName", z_py_gethostbyname, METH_VARARGS, NULL },
  { NULL,            NULL, 0, NULL }   /* sentinel*/
};

static gint
z_py_zorp_sockaddr_setattr(PyObject *o, char *name, PyObject *value)
{
  ZorpSockAddr *self = (ZorpSockAddr *) o;
  switch (self->sa->sa.sa_family)
    {
#if ENABLE_IPV6
    case AF_INET6:
      {
	struct sockaddr_in6 *sin6;
	
	sin6 = (struct sockaddr_in6 *) &self->sa->sa;
	if (strcmp(name, "ip") == 0)
	  {
	    if (!PyArg_Parse(value, "(iiiiiiii)", 
	                     &sin6->sin6_addr.s6_addr16[0],
	                     &sin6->sin6_addr.s6_addr16[1],
	                     &sin6->sin6_addr.s6_addr16[2],
	                     &sin6->sin6_addr.s6_addr16[3],
	                     &sin6->sin6_addr.s6_addr16[4],
	                     &sin6->sin6_addr.s6_addr16[5],
	                     &sin6->sin6_addr.s6_addr16[6],
	                     &sin6->sin6_addr.s6_addr16[7]))
	      return -1;
	  }
	else if (strcmp(name, "ip_s") == 0)
	  {
	    gchar *ip;
	    
	    if (!PyArg_Parse(value, "s", &ip))
	      return -1;
	    inet_pton(AF_INET6, ip, &sin6->sin6_addr);
	  }
	else if (strcmp(name, "port") == 0)
	  {
	    gint port;
	    if (!PyArg_Parse(value, "i", &port))
	      return -1;

	    sin6->sin6_port = htons(port);
	  }
#if ENABLE_IPOPTIONS
	else if (strcmp(name, "options") == 0)
	  {
	    gpointer options;
	    guint options_length;

	    /* IP options */
	    if (!PyArg_Parse(value, "s#", &options, &options_length))
	      return -1;
	    z_sockaddr_inet_set_options(self->sa, options, options_length);
	  }
#endif
      }
#endif
    case AF_INET:
      {    
	struct sockaddr_in *sin;
	
	sin = (struct sockaddr_in *) &self->sa->sa;
	if (strcmp(name, "ip") == 0)
	  {
	    if (!PyArg_Parse(value, "i", &sin->sin_addr.s_addr))
	      return -1;
	  }
	else if (strcmp(name, "ip_s") == 0)
	  {
	    gchar *ip;
	    
	    if (!PyArg_Parse(value, "s", &ip))
	      return -1;
	    z_inet_aton(ip, &sin->sin_addr);
	  }
	else if (strcmp(name, "port") == 0)
	  {
	    gint port;
	    if (!PyArg_Parse(value, "i", &port))
	      return -1;

	    sin->sin_port = htons(port);
	  }
#if ENABLE_IPOPTIONS
	else if (strcmp(name, "options") == 0)
	  {
	    gpointer options;
	    guint options_length;

	    /* IP options */
	    if (!PyArg_Parse(value, "s#", &options, &options_length))
	      return -1;
	    z_sockaddr_inet_set_options(self->sa, options, options_length);
	  }
#endif
	break;
      }
    case AF_UNIX:
      break;
    default:
      /*LOG
        This message indicates an internal error, because Zorp tried to
        use an address family it doesn't currently support.
       */
      z_log(NULL, CORE_ERROR, 3, "Internal error in z_py_zorp_sockaddr_set_addr, unknown address family;");
      break;
    }
  return 0;
}

static PyObject *
z_py_zorp_sockaddr_getattr(PyObject *o, char *name)
{
  ZorpSockAddr *self = (ZorpSockAddr *) o;
  
  if (strcmp(name, "family") == 0)
    {
      return PyInt_FromLong(self->sa->sa.sa_family);
    }
  switch (self->sa->sa.sa_family)
    {
#if ENABLE_IPV6
    case AF_INET6:
      if (strcmp(name, "type") == 0)
        {
          return PyString_FromString("inet6");
        }
      else if (strcmp(name, "ip") == 0)
        {
	  struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *) &self->sa->sa;
	  
          return Py_BuildValue("(iiiiiiii)", 
			       sin6->sin6_addr.s6_addr16[0],
			       sin6->sin6_addr.s6_addr16[1],
			       sin6->sin6_addr.s6_addr16[2],
			       sin6->sin6_addr.s6_addr16[3],
			       sin6->sin6_addr.s6_addr16[4],
			       sin6->sin6_addr.s6_addr16[5],
			       sin6->sin6_addr.s6_addr16[6],
			       sin6->sin6_addr.s6_addr16[7]);
           
        }
      else if (strcmp(name, "ip_s") == 0)
        {
          gchar buf[64];
          
          inet_ntop(AF_INET6, &((struct sockaddr_in6 *) &self->sa->sa)->sin6_addr, buf, sizeof(buf));
          return PyString_FromString(buf);
        }
      else if (strcmp(name, "port") == 0)
        {
          return PyInt_FromLong(ntohs(((struct sockaddr_in *) &self->sa->sa)->sin_port));
        }
#if ENABLE_IPOPTIONS
      else if (strcmp(name, "options") == 0)
        {
	  gpointer options;
	  guint options_length;

	  z_sockaddr_inet_get_options(self->sa, &options, &options_length);
          if (options)
            return PyString_FromStringAndSize(options, options_length);
          else
            return PyString_FromString("");
        }
#endif
      break;
#endif
    case AF_INET:
      if (strcmp(name, "type") == 0)
        {
          return PyString_FromString("inet");
        }
      else if (strcmp(name, "ip") == 0)
        {
          return PyInt_FromLong(((struct sockaddr_in *) &self->sa->sa)->sin_addr.s_addr);
        }
      else if (strcmp(name, "ip_s") == 0)
        {
          gchar buf[17];
          
          z_inet_ntoa(buf, sizeof(buf), ((struct sockaddr_in *) &self->sa->sa)->sin_addr);
          return PyString_FromString(buf);
        }
      else if (strcmp(name, "port") == 0)
        {
          return PyInt_FromLong(ntohs(((struct sockaddr_in *) &self->sa->sa)->sin_port));
        }
#if ENABLE_IPOPTIONS
      else if (strcmp(name, "options") == 0)
        {
	  gpointer options;
	  guint options_length;

	  z_sockaddr_inet_get_options(self->sa, &options, &options_length);
          if (options)
            return PyString_FromStringAndSize(options, options_length);
          else
            return PyString_FromString("");
        }
#endif
      break;
    case AF_UNIX:
      if (strcmp(name, "type") == 0)
        {
          return PyString_FromString("unix");
        }
      break;
    default:
      /*LOG
        This message indicates an internal error, because Zorp tried to
        use an address family it doesn't currently support.
       */
      z_log(NULL, CORE_ERROR, 3, "Internal error in z_py_zorp_sockaddr_get_addr, unknown address family;");
      break;
    }
  return Py_FindMethod(py_zorp_sockaddr_methods, o, name);
}

PyTypeObject z_py_zorp_sockaddr_type =
{
  PyObject_HEAD_INIT(&PyType_Type)
  0,                                        /*ob_size*/
  "ZorpSockAddr",                           /*tp_name*/
  sizeof(ZorpSockAddr),                     /*tp_basicsize*/
  0,                                        /*tp_itemsize*/
  /* methods */
  (destructor)z_py_zorp_sockaddr_free,      /*tp_dealloc*/
  (printfunc)0,                             /*tp_print*/
  (getattrfunc)z_py_zorp_sockaddr_getattr,  /*tp_getattr*/
  (setattrfunc)z_py_zorp_sockaddr_setattr,  /*tp_setattr*/
  (cmpfunc)0,                               /*tp_compare*/
  (reprfunc)z_py_zorp_sockaddr_format,      /*tp_repr*/
  0,                                        /*tp_as_number*/
  0,                   		            /*tp_as_sequence*/
  0,                   		            /*tp_as_mapping*/
  (hashfunc)0,         		            /*tp_hash*/
  (ternaryfunc)0,      		            /*tp_call*/
  (reprfunc)z_py_zorp_sockaddr_format,      /*tp_str*/

  /* Space for future expansion */
  0L,0L,0L,0L,
  "ZorpSockAddr class", /* Documentation string */
  0, 0, 0, 0
};

void
z_py_zorp_sockaddr_init(void)
{
  Py_InitModule("Zorp.SockAddr", z_py_zorp_sockaddr_funcs);
}
