/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: pystream.c,v 1.19.2.5 2003/07/29 15:46:05 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor : kisza
 * Last audited version: 1.8
 * Notes:
 *
 ***************************************************************************/

#include <zorp/pystream.h>

#include <zorp/zorp.h>
#include <zorp/log.h>
#include <zorp/stream.h>
#include <zorp/streamfd.h>
#include <zorp/policy.h>

PyObject *z_py_stream_new(ZStream *Stream);
static PyObject *z_py_zorp_stream_new(PyObject *self, PyObject *args);
static void z_py_zorp_stream_destroy(PyObject *o);
static PyObject *z_py_zorp_stream_getattr(PyObject *o, char *name);
static gint z_py_zorp_stream_setattr(PyObject *o, char *name,
				     PyObject *value);
static PyObject *z_py_zorp_stream_repr(PyObject *o);
static PyObject *z_py_zorp_stream_read(PyObject *o, PyObject *args);
static PyObject *z_py_zorp_stream_write(PyObject *o, PyObject *args);
static PyObject *z_py_zorp_stream_close(PyObject *o, PyObject *args);

PyMethodDef z_py_zorp_stream_funcs[] =
{
  { "Stream",  z_py_zorp_stream_new, METH_VARARGS, NULL },
  { NULL,          NULL, 0, NULL }   /* sentinel*/
};

static PyMethodDef py_zorp_stream_methods[] =
{
  { "read",        z_py_zorp_stream_read, METH_VARARGS, NULL },
  { "write",       z_py_zorp_stream_write, METH_VARARGS, NULL },
  { "close",       (PyCFunction) z_py_zorp_stream_close, 0, NULL },
  { NULL,          NULL, 0, NULL }   /* sentinel*/
};

PyTypeObject z_py_zorp_stream_type = 
{
  PyObject_HEAD_INIT(&z_py_zorp_stream_type)
  0,
  "ZorpStream",
  sizeof(ZorpStream),
  0,
  (destructor) z_py_zorp_stream_destroy,
  0,
  (getattrfunc) z_py_zorp_stream_getattr,
  (setattrfunc) z_py_zorp_stream_setattr,
  0,
  (reprfunc) z_py_zorp_stream_repr,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  "ZorpStream class for Zorp",
  0, 0, 0, 0
};

PyObject *
z_py_stream_new(ZStream *str)
{
  ZorpStream *self;

  if (str == NULL)
    {
      /*LOG
        This message attempt when try to create pystream without stream.
       */
      z_log(NULL, CORE_ERROR, 3, "Internal error in z_py_zorp_stream_new: input ZStream is NULL;");
      return NULL;
    }

  self = PyObject_NEW(ZorpStream, &z_py_zorp_stream_type);
  z_stream_ref(str);
  self->stream = str;
  return (PyObject *) self;
}

static PyObject *
z_py_zorp_stream_new(PyObject *o G_GNUC_UNUSED, PyObject *args)
{
  ZorpStream *self;
  char *name;
  int fd;

  if (!PyArg_ParseTuple(args, "is", &fd, &name))
    return NULL;
  self = PyObject_NEW(ZorpStream, &z_py_zorp_stream_type);
  if (!self)
    return NULL;

  self->stream = z_stream_new(fd, name);
  return (PyObject *) self;
}

static void
z_py_zorp_stream_destroy(PyObject *o)
{
  ZorpStream *self = (ZorpStream *) o;

  z_stream_unref(self->stream);
  PyMem_DEL(self);
}

static PyObject *
z_py_zorp_stream_getattr(PyObject *o, char *name)
{
  ZorpStream *self = (ZorpStream *) o;
  if (strcmp(name, "fd") == 0)
    {
      return z_policy_var_build("i", z_stream_get_fd(self->stream));
    }
  else if (strcmp(name, "name") == 0)
    {
      return PyString_FromString(self->stream->name);
    }
  else if (strcmp(name, "bytes_recvd") == 0)
    {
      return PyLong_FromUnsignedLong(self->stream->bytes_recvd);
    }
  else if (strcmp(name, "bytes_sent") == 0)
    {
      return PyLong_FromUnsignedLong(self->stream->bytes_sent);
    }

  return Py_FindMethod(py_zorp_stream_methods, o, name);
}

static gint 
z_py_zorp_stream_setattr(PyObject *o, char *name,
			 PyObject *value)
{
  ZorpStream *self = (ZorpStream *) o;
  gchar *str;
  if (strcmp(name, "name") == 0)
    {
      if (!PyArg_Parse(value, "s", &str))
	{
	  PyErr_SetString(PyExc_TypeError, "Stream name is not a string");
	  return 1;
	}
      else
	{
	  g_strlcpy(self->stream->name, str, Z_STREAM_MAX_NAME);
	  return 0;
	}
    }

  PyErr_SetString(PyExc_AttributeError, "No such attribute");
  return 1;
}

static PyObject *
z_py_zorp_stream_repr(PyObject *o)
{
  ZorpStream *self = (ZorpStream *) o;

  return PyString_FromString(self->stream->name);
}

static PyObject *
z_py_zorp_stream_read(PyObject *o, PyObject *args)
{
  ZorpStream *self = (ZorpStream *) o;
  gchar *buf;
  guint length, bytes_read;
  gint res;
  
  if (!PyArg_ParseTuple(args, "i", &length))
    return NULL;

  buf = alloca(length);
  Py_BEGIN_ALLOW_THREADS
  res = z_stream_read(self->stream, buf, length, &bytes_read, NULL);
  Py_END_ALLOW_THREADS
  
  if (res != G_IO_STATUS_NORMAL)
    {
      PyErr_SetString(PyExc_IOError, "I/O error reading stream.");
      return NULL;      
    }
  return Py_BuildValue("s#", buf, bytes_read);
}

static PyObject *
z_py_zorp_stream_close(PyObject *o, PyObject *args G_GNUC_UNUSED)
{
  ZorpStream *self = (ZorpStream *) o;

  z_stream_close(self->stream, NULL);
  Py_XINCREF(Py_None);
  return Py_None;
}

static PyObject *
z_py_zorp_stream_write(PyObject *o, PyObject *args)
{
  ZorpStream *self = (ZorpStream *) o;
  gchar *buf; 
  guint length, bytes_written;
  gint res;
  
  if (!PyArg_ParseTuple(args, "s#", &buf, &length))
    return NULL;
    
  Py_BEGIN_ALLOW_THREADS
  res = z_stream_write(self->stream, buf, length, &bytes_written, NULL);
  Py_END_ALLOW_THREADS
  
  if (res != G_IO_STATUS_NORMAL)
    {
      PyErr_SetString(PyExc_IOError, "I/O error writing stream.");
      return NULL;
    }
  
  Py_XINCREF(Py_None);
  return Py_None;
}

void
z_py_zorp_stream_init(void)
{
  PyImport_AddModule("Zorp.Stream");
  Py_InitModule("Zorp.Stream", z_py_zorp_stream_funcs);
}
