/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: ftpcmd.c,v 1.64.2.13 2003/10/28 16:59:49 sasa Exp $
 *
 * Author:  Andras Kis-Szabo <kisza@sch.bme.hu>
 * Author:  Attila SZALAY <sasa@balabit.hu>
 * Auditor:
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/zorp.h>
#include <zorp/stream.h>
#include <zorp/proxy.h>
#include <zorp/policy.h>
#include <zorp/thread.h>
#include <zorp/zpython.h>
#include <zorp/log.h>
#include <zorp/pysockaddr.h>

#include "ftp.h"
#include "ftphash.h"
#include "ftpcmd.h"

#include <assert.h>
#include <arpa/inet.h>

gboolean
ftp_parse_nums(gchar *src, gint length, unsigned char *nums)
{
  int i = 0;
  gchar *newsrc;
  
  z_enter();
  while (length > 0 && i < 6)
    {
      unsigned int tmp;
      
      errno = 0;
      tmp = strtoul(src, &newsrc, 10);
      if (tmp > 255 || errno == ERANGE)
        {
          z_leave();
          return FALSE;
        }
      nums[i] = tmp;
      if (i < 5 && *newsrc != ',')
        {
          z_leave();
          return FALSE;
        }
      length -= (newsrc - src + 1);
      src = newsrc + 1;
      i++;
    }
  if (length <= 0)
    {
      z_leave();
      return TRUE;
    }
  else
    {
      z_leave();
      return FALSE;
    }
}

gboolean
ftp_parse_search_nums(gchar *src, gint length, unsigned char *nums)
{
  gchar *left, *right;

  z_enter();
  left = strchr(src,'(');
  if (left)
    {
      right = strrchr(src,')');
      if (right)
        {
          left++;
          length = right - left;
          if (length > 0)
            {
              z_leave();
              return (ftp_parse_nums(left, length, nums));
            }
        }
    }
  z_leave();
  return FALSE;
}

guint
ftp_command_parse_USER(FtpProxy *self)
{
  gchar *atsign;
  gchar *ddot;
    
  z_proxy_enter(self);

  switch (self->ftp_state)
    {
    case FTP_STATE_PRECONNECT:
    case FTP_STATE_PRECONNECT_LOGIN:
    case FTP_STATE_PRECONNECT_LOGIN_U:
    case FTP_STATE_PRECONNECT_LOGIN_P:
      /* USER in non transparent mode, we'll need to find out destination hostname */
      
      /* 
       * FIXME: there should be other ways of telling the proxy where to
       * connect, like SITE HOST <hostname> 
       */ 

      atsign = strrchr(self->request_param->str, '@');
      if (!atsign)
        {
          SET_ANSWER(MSG_USERNAME_FORMAT_INVALID);
          /*LOG
            This debug message attemp, when Client doesn't use the
            non-transparent ftp protocoll correctly.
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "USER command processing error; reason='cannot found @ sign in non-transparent mode'");
          
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      
      ddot = strrchr(self->request_param->str, ':');
      if (ddot && ddot > atsign)
        {
          *ddot = 0;
        }
      
      if (strlen(atsign + 1) > self->max_hostname_length)
        {
          SET_ANSWER(MSG_HOSTNAME_TOO_LONG);
          /*LOG
            This debug message attemp, when Client violate the
            non-transparent ftp protocoll.
           */
          z_proxy_log(self, FTP_POLICY, 3, "USER command processing error;  reason='hostname too long'");
          
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }

      *atsign = 0;
      if (strlen(self->request_param->str) > self->max_username_length)
        {
          SET_ANSWER(MSG_USERNAME_TOO_LONG);
          /*LOG
            This message attemp, when Client violate the
            non-transparent ftp protocoll.
           */
          z_proxy_log(self, FTP_POLICY, 3, "USER command processing error; reason='username too long'");
          
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
        
      /* 
       * NOTE: the length of hostname and username is checked above, this is
       * not Buffer OverFlow
       */
      strcpy(self->hostname, atsign + 1);
      
      self->username = g_string_assign(self->username, self->request_param->str);
      if (ddot)
        {
          self->hostport = atoi(ddot + 1);
      
          if (!z_port_enabled(self->target_port_range->str, self->hostport))
            {
              SET_ANSWER(MSG_USERNAME_FORMAT_INVALID);
              /*LOG
                This debug message attemp, when Client doesn't use the
                non-transparent ftp protocoll correctly.
               */
              z_proxy_log(self, FTP_POLICY, 3, "USER command processing error; reason='bad ftp port'");
          
              z_proxy_leave(self);
              return FTP_REQ_REJECT;
            }
        }
      else
        self->hostport = 21;
      
      self->ftp_state = FTP_STATE_PRECONNECT_LOGIN_U;

      SET_ANSWER(MSG_USER_OKAY);
      z_proxy_trace(self, "USER Command ok;");
      
      z_proxy_leave(self);
      return FTP_PROXY_ANS;
    case FTP_STATE_LOGIN:
    case FTP_STATE_LOGIN_U:
    case FTP_STATE_LOGIN_P:
    case FTP_STATE_LOGIN_A:
    case FTP_STATE_LOGINAUTH:
      if (self->request_param->len > self->max_username_length)
        {
          SET_ANSWER(MSG_USERNAME_TOO_LONG);

          /*LOG
            This message attemp, when Client violate the
            non-transparent ftp protocoll.
           */
          z_proxy_log(self, FTP_POLICY, 3, "USER command processing error; reason='username too long'");
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      self->username = g_string_assign(self->username, self->request_param->str);
      self->ftp_state = FTP_STATE_LOGIN_U;
      break;
    case FTP_STATE_CONVERSATION:
    case FTP_STATE_DATA:
      SET_ANSWER(MSG_USER_ALREADY_LOGGED_IN);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    default:
      z_proxy_leave(self);
      return FTP_REQ_ABORT;
    }
    
  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_command_answer_USER(FtpProxy *self)
{
  z_proxy_enter(self);
  
  switch (self->ftp_state)
    {
    case FTP_STATE_LOGIN_U:
      switch(self->answer_cmd->str[0])
        {
        case '2':
          self->ftp_state = FTP_STATE_CONVERSATION;
          break;
        case '3':
          switch(self->answer_code)
            {
            case 331:
              break;
            case 332:
              self->ftp_state = FTP_STATE_LOGIN_A;
            }
          break;
        }
      break;
    }
  z_proxy_leave(self);
  return FTP_RSP_ACCEPT;
}

guint
ftp_command_parse_PASS(FtpProxy *self)
{
  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_PRECONNECT:
    case FTP_STATE_PRECONNECT_LOGIN:
    case FTP_STATE_PRECONNECT_LOGIN_P:

      /* PASS in non-transparent startup */
      SET_ANSWER(MSG_USER_FIRST);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    case FTP_STATE_PRECONNECT_LOGIN_U:
      if (self->request_param->len > self->max_password_length)
        {
          SET_ANSWER(MSG_PASSWORD_TOO_LONG);
          /*LOG
            Placeholder ftpcmd.c.1
           */
          z_proxy_log(self, FTP_POLICY, 3, "PASS command processing error; reason='password too long'");
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      self->password = g_string_assign(self->password, self->request_param->str);
      self->ftp_state = FTP_STATE_PRECONNECT_LOGIN_P;
      z_proxy_leave(self);
      return FTP_NOOP;

    case FTP_STATE_LOGIN:
    case FTP_STATE_LOGIN_P:
    case FTP_STATE_LOGIN_A:
    case FTP_STATE_LOGINAUTH:
      SET_ANSWER(MSG_USER_FIRST);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    case FTP_STATE_LOGIN_U:
      if (strlen(self->request_param->str) > self->max_password_length)
        {
          SET_ANSWER(MSG_PASSWORD_TOO_LONG);
          /*LOG
            Placeholder ftpcmd.c.2
           */
          z_proxy_log(self, FTP_POLICY, 3, "PASS command processing error; reason='password too long'");
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      self->password = g_string_assign(self->password, self->request_param->str);
      self->ftp_state = FTP_STATE_LOGIN_P;
      break;
    case FTP_STATE_CONVERSATION:
      z_proxy_leave(self);
      return FTP_REQ_ACCEPT;
    case FTP_STATE_DATA:
      SET_ANSWER(MSG_USER_FIRST);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    default:
      /*LOG
        Placeholder ftpcmd.c.3
       */
      z_proxy_log(self, FTP_ERROR, 1, "Internal error; error='Unknown state', cmd='PASS'");
      z_proxy_leave(self);
      return FTP_REQ_ABORT;
    }
  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_command_answer_PASS(FtpProxy *self)
{
  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_LOGIN_P:
      switch (self->answer_cmd->str[0])
        {
        case '2':
          self->ftp_state = FTP_STATE_CONVERSATION;
          break;
        case '3':
          switch(self->answer_code)
            {
            case 332:
              self->ftp_state = FTP_STATE_LOGIN_A;
              break;
            }
          break;
        }
      break;
    }
  z_proxy_leave(self);
  return FTP_RSP_ACCEPT;
}

guint
ftp_command_parse_ACCT(FtpProxy *self)
{
  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_LOGIN:
    case FTP_STATE_LOGIN_U:
    case FTP_STATE_LOGIN_P:
    case FTP_STATE_LOGINAUTH:
      SET_ANSWER(MSG_USER_FIRST);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    case FTP_STATE_LOGIN_A:
    case FTP_STATE_CONVERSATION:
    case FTP_STATE_DATA:
      break;
    default:
      /*LOG
        Placeholder ftpcmd.c.4
       */
      z_proxy_log(self, FTP_ERROR, 1, "Internal error; error='Unknown state', cmd='ACCT'");
      z_proxy_leave(self);
      return FTP_REQ_ABORT;
    }
  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_command_answer_ACCT(FtpProxy *self)
{
  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_LOGIN_A:
      switch (self->answer_cmd->str[0])
        {
        case '2':
          self->ftp_state = FTP_STATE_CONVERSATION;
          break;
        }
      break;
    }
  z_proxy_leave(self);
  return FTP_RSP_ACCEPT;
}


guint
ftp_command_parse_path(FtpProxy *self)
{
  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
      break;
    case FTP_STATE_DATA:
      if (self->command_desc->need_data)
        {
          ftp_state_both(self);
          self->state = FTP_BOTH_SIDE;
        }
      break;
    default:
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }
  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_command_answer_path(FtpProxy *self)
{
  z_proxy_enter(self);
  if (!self->command_desc->need_data)
    {
      z_proxy_leave(self);
      return FTP_RSP_ACCEPT;
    }
  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
    case FTP_STATE_DATA:
      switch (self->answer_cmd->str[0])
        {
        case '1':
          self->oldstate = FTP_SERVER_TO_CLIENT;
          self->data_state |= FTP_DATA_SERVER_SAID;
          break;
        case '2':
          self->oldstate = FTP_CLIENT_TO_SERVER;
          self->ftp_state = FTP_STATE_CONVERSATION;
          break;
        case '4':
        case '5':
          self->oldstate = FTP_CLIENT_TO_SERVER;
          self->ftp_state = FTP_STATE_CONVERSATION;
          ftp_data_reset(self);
          break;
        default:
          /*LOG
            This message attempt when server say wrong answer
           */
          z_proxy_log(self, FTP_VIOLATION, 1, "Unknown answer, Abort connection; answer='%d'",self->answer_code);
          self->data_state = 0;
          self->oldstate = FTP_CLIENT_TO_SERVER;
          break;
        }
      break;
    default:
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      z_proxy_leave(self);
      return FTP_RSP_REJECT;
    }
  z_proxy_leave(self);
  return FTP_RSP_ACCEPT;
}

guint
ftp_command_parse_QUIT(FtpProxy *self)
{
  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_PRECONNECT:
    case FTP_STATE_PRECONNECT_LOGIN_U:
    case FTP_STATE_PRECONNECT_LOGIN_P:
      if (self->request_param->len > 0)
        {
          SET_ANSWER(MSG_INVALID_PARAMETER);
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      SET_ANSWER(MSG_GOODBYE);
      self->ftp_state = FTP_STATE_PRECONNECT_QUIT;
      z_proxy_leave(self);
      return FTP_REQ_ABORT;
    case FTP_STATE_LOGIN:
    case FTP_STATE_LOGIN_U:
    case FTP_STATE_LOGIN_P:
    case FTP_STATE_LOGIN_A:
    case FTP_STATE_LOGINAUTH:
    case FTP_STATE_CONVERSATION:
    case FTP_STATE_DATA:
      if (self->request_param->len > 0)
        {
          SET_ANSWER(MSG_INVALID_PARAMETER);
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      self->ftp_state = FTP_STATE_QUIT;
      break;
    default:
      /*LOG
        Placeholder ftpcmd.c.5
       */
      z_proxy_log(self, FTP_ERROR, 1, "Internal error; error='Unknown state', cmd='QUIT'");
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }
  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_command_answer_QUIT(FtpProxy *self)
{
  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_QUIT:
      self->state = FTP_QUIT;
      break;
    }
  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_command_parse_TYPE(FtpProxy *self)
{
  gchar mytype;
  z_proxy_enter(self);

  if(self->ftp_state == FTP_STATE_CONVERSATION ||
     self->ftp_state == FTP_STATE_DATA)
    {

      if (self->request_param->len == 0)
        {
          SET_ANSWER(MSG_MISSING_PARAMETER);
          /*LOG
            Placeholder ftpcmd.c.6
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "Missing parameter; req='TYPE'");
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      
      mytype = self->request_param->str[0];
      switch(mytype)
        {
        case 'a':
        case 'A':
        case 'i':
        case 'I':
          self->request_param = g_string_assign(self->request_param, "");
          self->request_param = g_string_append_c(self->request_param, mytype);
          self->request_param = g_string_up(self->request_param);
          break;
        case 'l':
        case 'L':
        case 'e':
        case 'E':
          SET_ANSWER(MSG_COMMAND_NOT_IMPLEMENTED);
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        default:
          SET_ANSWER(MSG_COMMAND_NOT_RECOGNIZED);
          /*LOG
            Placeholder ftpcmd.c.7
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "Bad parameter; req='TYPE'");
          z_proxy_leave(self);
          return FTP_REQ_REJECT; 
        }

    }
  else
    {
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }
  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_command_parse_ABOR(FtpProxy *self)
{
  char buf[3];
  guint len;

  z_proxy_enter(self);

  buf[0]=0xff;
  buf[1]=0xf4;
  buf[2]=0xff;

  if (self->ftp_state == FTP_STATE_CONVERSATION || self->ftp_state == FTP_STATE_DATA)
    {
      z_stream_write_pri(self->super.endpoints[EP_SERVER], buf, 3, &len, NULL);
      buf[0]=0xf2;
      z_stream_write(self->super.endpoints[EP_SERVER], buf, 1, &len, NULL);
//      ftp_state_set(EP_SERVER);
      self->state = FTP_SERVER_TO_CLIENT;
      self->ftp_state = FTP_STATE_CONVERSATION;
      z_proxy_leave(self);
      return FTP_REQ_ACCEPT;
    }
  else if (self->ftp_state == FTP_STATE_RENAME)
    {
      self->ftp_state = FTP_STATE_CONVERSATION;
    }

  SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
  z_proxy_leave(self);
  return FTP_REQ_REJECT;
}

guint
ftp_command_answer_ABOR(FtpProxy *self)
{
  z_proxy_enter(self);
  if(self->ftp_state == FTP_STATE_CONVERSATION ||
     self->ftp_state == FTP_STATE_DATA)
    {

      if(self->answer_cmd->str[0] == '2')
        {
          self->ftp_state = FTP_STATE_CONVERSATION;
          self->oldstate = FTP_CLIENT_TO_SERVER;
        }

      else if(self->answer_cmd->str[0] == '4')
        {
          self->oldstate = FTP_SERVER_TO_CLIENT;
          self->data_state = 0;
        }

    }
  z_proxy_leave(self);
  return FTP_RSP_ACCEPT;
}

guint
ftp_command_parse_noarg(FtpProxy *self)
{
  z_proxy_enter(self);
  switch (self->ftp_state)
  {
  case FTP_STATE_CONVERSATION:
  case FTP_STATE_DATA:
    self->request_param = g_string_assign(self->request_param, "");
    break;
  default:
    SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
    z_proxy_leave(self);
    return FTP_RSP_REJECT;
  }
  z_proxy_leave(self);
  return FTP_RSP_ACCEPT;
}

guint
ftp_command_parse_HELP(FtpProxy *self G_GNUC_UNUSED)
{
  return FTP_RSP_ACCEPT;
}

guint
ftp_command_parse_MODE(FtpProxy *self)
{
  char mymode;

  z_proxy_enter(self);
  if (self->ftp_state == FTP_STATE_CONVERSATION ||
      self->ftp_state == FTP_STATE_DATA)
    {

      if (self->request_param->len == 0)
        {
          SET_ANSWER(MSG_MISSING_PARAMETER);
          /*LOG
            Placeholder ftpcmd.c.8
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "Missing parameter; req='MODE'");
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }

      mymode = self->request_param->str[0];
      switch(mymode)
        {
        case 's':
        case 'S':
        case 'b':
        case 'B':
        case 'c':
        case 'C':
          self->request_param = g_string_assign(self->request_param, "");
          self->request_param = g_string_append_c(self->request_param, mymode);
          self->request_param = g_string_up(self->request_param);
          break;
        default:
          /*LOG
            Placeholder ftpcmd.c.9
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "Bad parameter; req='MODE'");
          SET_ANSWER(MSG_COMMAND_NOT_RECOGNIZED);
          z_proxy_leave(self);
          return FTP_REQ_REJECT; 
        }
    }
  else
    {
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }
  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_command_parse_string(FtpProxy *self)
{
  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
    case FTP_STATE_DATA:
      break;
    default:
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }
  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_command_parse_STRU(FtpProxy *self)
{
  char mystru;
  
  z_proxy_enter(self);

  if (self->ftp_state == FTP_STATE_CONVERSATION ||
      self->ftp_state == FTP_STATE_DATA)
    {
      if (self->request_param->len == 0)
        {
          SET_ANSWER(MSG_MISSING_PARAMETER);
          /*LOG
            Placeholder ftpcmd.c.10
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "Missing parameter; req='STRU'");
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
        
      mystru = self->request_param->str[0];
      switch(mystru)
        {
        case 'f':
        case 'F':
          self->request_param = g_string_assign(self->request_param, "");
          self->request_param = g_string_append_c(self->request_param, mystru);
          self->request_param = g_string_up(self->request_param);
          break;
        default:
          SET_ANSWER(MSG_COMMAND_NOT_RECOGNIZED);
          /*LOG
            Placeholder ftpcmd.c.11
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "BAD parameter; req='STRU'");
          z_proxy_leave(self);
          return FTP_REQ_REJECT; 
        }
    }
  else
    {
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }
  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_data_server_start_PORT(FtpProxy *self)
{
  guint port;
  gchar tmpaddr[16];
  gchar tmpport[10];

  z_proxy_enter(self);

  if (!ftp_data_prepare(self, EP_SERVER, 'L'))
    {
      SET_ANSWER(MSG_ERROR_PARSING_PORT);
      self->data_state = 0;
      /*LOG
        Placeholder ftpcmd.c.12
       */
      z_proxy_log(self, FTP_ERROR, 2, "Error preparing server listen;");
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }

  if (self->masq_address[EP_SERVER]->len)
    {
      g_strlcpy(tmpaddr, self->masq_address[EP_SERVER]->str, sizeof(tmpaddr));
    }
  else
    {
      z_inet_ntoa(tmpaddr, sizeof(tmpaddr), ((struct sockaddr_in *) &self->data_local[EP_SERVER]->sa)->sin_addr);
    }

  g_strdelimit(tmpaddr, ".", ',');

  port = ntohs(((struct sockaddr_in *) &self->data_local[EP_SERVER]->sa)->sin_port);
  if(port == 0)
    {
      SET_ANSWER(MSG_ERROR_PARSING_PORT);

      /*LOG
        Placeholder ftpcmd.c.13
       */
      z_proxy_log(self, FTP_ERROR, 2, "Cannot bind to given address;");
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }

  g_snprintf(tmpport, sizeof(tmpport), ",%d,%d", (port & 0xff00) >> 8, port & 0x00ff);

  self->request_param = g_string_assign(self->request_param, "");
  self->request_param = g_string_append(self->request_param, tmpaddr);
  self->request_param = g_string_append(self->request_param, tmpport);

  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_data_server_start_PASV(FtpProxy *self)
{
  guchar nums[6], ip[17];
  guint16 port;

  z_proxy_enter(self);
  if (!ftp_parse_search_nums(self->answer_param->str, self->answer_param->len, nums))
    {
      SET_ANSWER(MSG_ERROR_PARAMETER_PASV);
      /*LOG
        Placeholder ftpcmd.c.14
       */
      z_proxy_log(self, FTP_VIOLATION, 2, "Bad PASV params;");
      z_proxy_leave(self);
      return FTP_RSP_REJECT;
    }

  g_snprintf(ip, sizeof(ip), "%d.%d.%d.%d", nums[0], nums[1], nums[2], nums[3]);

  port = nums[4] * 256 + nums[5];

  self->data_remote[EP_SERVER] = z_sockaddr_inet_new(ip, port);

  if (!ftp_data_prepare(self, EP_SERVER, 'C'))
    {
      SET_ANSWER(MSG_ERROR_PARSING_PASV);
      self->data_state = 0;
      /*LOG
        Placeholder ftpcmd.c.15
       */
      z_proxy_log(self, FTP_ERROR, 2, "Cannot prepare server connect;");
      z_proxy_leave(self);
      return FTP_RSP_REJECT;
    }
  z_proxy_leave(self);
  return FTP_RSP_ACCEPT;
}

guint
ftp_data_server_start_EPRT(FtpProxy *self)
{
  guint port;
  gchar tmpaddr[16];
  gchar tmpport[10];

  z_proxy_enter(self);

  if (!ftp_data_prepare(self, EP_SERVER, 'L'))
    {
      SET_ANSWER(MSG_ERROR_PARSING_PORT);
      self->data_state = 0;
      /*LOG
        Placeholder ftpcmd.c.16
       */
      z_proxy_log(self, FTP_ERROR, 2, "Cannot prepare server listen (EPRT);");
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }

  if (self->masq_address[EP_SERVER]->len)
    g_strlcpy(tmpaddr, self->masq_address[EP_SERVER]->str, sizeof(tmpaddr));
  else
    z_inet_ntoa(tmpaddr, sizeof(tmpaddr), ((struct sockaddr_in *) &self->data_local[EP_SERVER]->sa)->sin_addr);

  port = ntohs(((struct sockaddr_in *) &self->data_local[EP_SERVER]->sa)->sin_port);
  if(port == 0)
    {
      SET_ANSWER(MSG_ERROR_PARSING_PORT);

      /*LOG
        Placeholder ftpcmd.c.17
       */
      z_proxy_log(self, FTP_ERROR, 2, "Cannot bind to given address (EPRT);");
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }

  g_snprintf(tmpport, sizeof(tmpport), "|%d|", port);

  self->request_param = g_string_assign(self->request_param, "|1|");
  self->request_param = g_string_append(self->request_param, tmpaddr);
  self->request_param = g_string_append(self->request_param, tmpport);

  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_data_server_start_EPSV(FtpProxy *self)
{
  gchar **split;
  guint port;
  gchar *err;
  gchar tmpline[FTP_LINE_MAX_LEN];
  gchar *start, *end;
  gchar delim[2];
  ZorpSockAddr *sockaddr;
  ZSockAddr *tmpaddr;
  gchar tmpip[16];

  z_proxy_enter(self);
  
  if (self->answer_param->len <= 0)
    {
      /*LOG
        Placeholder ftpcmd.c.18
       */
      z_proxy_log(self, FTP_VIOLATION, 2, "Missing param (EPSV);");
      z_proxy_leave(self);
      return FTP_RSP_REJECT;
    }

  g_strlcpy(tmpline, self->answer_param->str, sizeof(tmpline));

  start = strchr(tmpline, '(');
  if (!start)
    {
      /*LOG
        Placeholder ftpcmd.c.19
       */
      z_proxy_log(self, FTP_VIOLATION, 2, "Bad param (EPSV);");
      z_proxy_leave(self);
      return FTP_RSP_REJECT;
    }
  *start = 0;
  end = strchr(start + 1, ')');
  *end = 0;

  delim[0] = start[1];
  delim[1] = 0;
  split = g_strsplit(start + 1, delim, 6);
  
  if (split[0] == NULL ||
      split[1] == NULL ||
      split[2] == NULL ||
      split[3] == NULL ||
      split[4] == NULL ||
      split[5] != NULL)
    {
      SET_ANSWER(MSG_ERROR_PARAMETER_EPSV);
      g_strfreev(split);
      /*LOG
        Placeholder ftpcmd.c.20
       */
      z_proxy_log(self, FTP_VIOLATION, 2, "Bad param (EPSV);");
      z_proxy_leave(self);
      return FTP_RSP_REJECT;
    }
  
  if (strlen(split[1]) == 0 || strcmp(split[1],"1") == 0)
    {
      port = strtol(split[3], &err, 10);
  
      if (port == 0 || *err != 0)
        {
          SET_ANSWER(MSG_ERROR_PARAMETER_EPSV);
          g_strfreev(split);
          /*LOG
            Placeholder ftpcmd.c.21
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "Bad param (EPSV);");
          z_proxy_leave(self);
          return FTP_RSP_REJECT;
        }

      if (strlen(split[2]) > 0)
        self->data_remote[EP_SERVER] = z_sockaddr_inet_new(split[2], port);
      else
        {
          z_policy_lock(self->super.thread);
          sockaddr = (ZorpSockAddr *)z_session_getattr(self->super.handler, "server_address");
          tmpaddr = sockaddr->sa;
          z_policy_unlock(self->super.thread);
          z_inet_ntoa(tmpip, sizeof(tmpip), ((struct sockaddr_in *) &tmpaddr->sa)->sin_addr);
          self->data_remote[EP_SERVER] = z_sockaddr_inet_new(tmpip, port);
        }
    }
  else
    {
      SET_ANSWER(MSG_ERROR_PARAMETER_EPSV);
      g_strfreev(split);
      /*LOG
        Placeholder ftpcmd.c.22
       */
      z_proxy_log(self, FTP_VIOLATION, 1, "Unknown protocoll type (EPSV);");
      z_proxy_leave(self);
      return FTP_RSP_REJECT;
    }

  g_strfreev(split);

  if (!ftp_data_prepare(self, EP_SERVER, 'C'))
    {
      SET_ANSWER(MSG_ERROR_PARSING_PASV);
      self->data_state = 0;
      /*LOG
        Placeholder ftpcmd.c.23
       */
      z_proxy_log(self, FTP_ERROR, 2, "Error preparing client connect (EPSV);");
      z_proxy_leave(self);
      return FTP_RSP_REJECT;
    }
  z_proxy_leave(self);
  return FTP_RSP_ACCEPT;
}

guint
ftp_command_parse_PORT(FtpProxy *self)
{
  guchar nums[6], ip[17];
  guint16 port;
  guint res = FTP_REQ_ACCEPT;

  if (self->ftp_state == FTP_STATE_DATA)
    {
      self->ftp_state = FTP_STATE_CONVERSATION;
      ftp_data_reset(self);
    }

  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
      if (!ftp_parse_nums(self->request_param->str,
                          self->request_param->len,
                          nums))
        {
          SET_ANSWER(MSG_ERROR_PARAMETER_PORT);
          /*LOG
            Placeholder ftpcmd.c.24
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "Bad parameter (PORT);");
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      g_snprintf(ip, sizeof(ip), "%d.%d.%d.%d", nums[0], nums[1], nums[2], nums[3]);

      port = nums[4] * 256 + nums[5];
      
      self->data_remote[EP_CLIENT] = z_sockaddr_inet_new(ip, port);
      
      switch (self->data_mode)
        {
        case FTP_DATA_PASSIVE:
           self->request_cmd = g_string_assign(self->request_cmd, "PASV");
           self->request_param = g_string_assign(self->request_param, "");
           break;
        case FTP_DATA_ACTIVE:
        case FTP_DATA_KEEP:
          res = ftp_data_server_start_PORT(self);
          break;
        default:
          /*LOG
            This message attempt when data_mode set to wrong value.
           */
          z_proxy_log(self, FTP_POLICY, 1, "Connection mode not supported; data_mode='%d'", self->data_mode);
          SET_ANSWER(MSG_ERROR_PARSING_PORT);
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      break;
    default:
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }
  z_proxy_leave(self);
  return res;
}

guint ftp_command_answer_PORT(FtpProxy *self)
{
  guint res = FTP_RSP_ACCEPT;

  z_proxy_enter(self);
  
  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
      switch (self->data_mode)
        {
        case FTP_DATA_PASSIVE:
          switch (self->answer_cmd->str[0])
            {
            case '2':
              res = ftp_data_server_start_PASV(self);
              if (res == FTP_RSP_ACCEPT)
                {
                  if (!ftp_data_prepare(self, EP_CLIENT, 'C'))
                    {
                      self->data_state = 0;
                      SET_ANSWER(MSG_ERROR_PARSING_PORT);
                      /*LOG
                        Placeholder ftpcmd.c.25
                       */
                      z_proxy_log(self, FTP_ERROR, 2, "Error preparing Client connect (PORT);");
                      z_proxy_leave(self);
                      return FTP_RSP_REJECT;
                    }

                  SET_ANSWER(MSG_PORT_SUCCESFULL);
                  res = FTP_RSP_ACCEPT;
                }
              self->ftp_state = FTP_STATE_DATA;
              break;
            case '4':
            case '5':
                ftp_data_reset(self);
                z_proxy_leave(self);
                return FTP_RSP_ACCEPT;
            default:
                {
                  SET_ANSWER(MSG_ERROR_PARSING_PORT);
                  /*LOG
                    Placeholder ftpcmd.c.26
                   */
                  z_proxy_log(self, FTP_VIOLATION, 2, "Error parsing server answer (PORT);");
                  ftp_data_reset(self);
                  z_proxy_leave(self);
                  return FTP_RSP_REJECT;
                }
            }
          break;
        case FTP_DATA_ACTIVE:
        case FTP_DATA_KEEP:
          switch (self->answer_cmd->str[0])
            {
            case '2':
              if (!ftp_data_prepare(self, EP_CLIENT, 'C'))
                {
                  self->data_state = 0;
                  SET_ANSWER(MSG_ERROR_PARSING_PORT);
                  /*LOG
                    Placeholder ftpcmd.c.27
                   */
                  z_proxy_log(self, FTP_ERROR, 2, "Cannot prepare client connect (PORT);");
                  z_proxy_leave(self);
                  return FTP_RSP_REJECT;
                }
              self->ftp_state = FTP_STATE_DATA;
              break;
            case '4':
            case '5':
                ftp_data_reset(self);
                z_proxy_leave(self);
                return FTP_RSP_ACCEPT;
            default:
              SET_ANSWER(MSG_ERROR_PARSING_PORT);
              /*LOG
                Placeholder ftpcmd.c.28
               */
              z_proxy_log(self, FTP_VIOLATION, 2, "Bad server answer (PORT);");
              ftp_data_reset(self);
              z_proxy_leave(self);
              return FTP_RSP_ACCEPT;
            }
          break;
        default:
          break;
        }
    }
  z_proxy_leave(self);
  return res;
}

guint
ftp_command_parse_PASV(FtpProxy *self)
{
  guint res = FTP_REQ_ACCEPT;

  z_proxy_enter(self);

  if (self->ftp_state == FTP_STATE_DATA)
    {
      self->ftp_state = FTP_STATE_CONVERSATION;
      ftp_data_reset(self);
    }

  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
      self->request_param = g_string_assign(self->request_param, "");
      self->data_state = 0;

      switch (self->data_mode)
        {
        case FTP_DATA_KEEP:
        case FTP_DATA_PASSIVE:
          break;
        case FTP_DATA_ACTIVE:
          self->request_cmd = g_string_assign(self->request_cmd, "PORT");
          self->request_param = g_string_assign(self->request_param, "");
          res = ftp_data_server_start_PORT(self);
          break;
        default:
          /*LOG
            This message attempt when data_mode set to wrong value.
           */
          z_proxy_log(self, FTP_POLICY, 1, "connection mode not supported; data_mode='%d'", self->data_mode);
          SET_ANSWER(MSG_ERROR_PARSING_PORT);
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      break;
    default:
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }

  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_command_answer_PASV(FtpProxy *self)
{
  gchar *start, *end;
  guint port;
  gchar tmpline[FTP_LINE_MAX_LEN];
  gchar tmpaddr[16];
  gchar tmpport[10];
  guint ret = FTP_RSP_ACCEPT;

  z_proxy_enter(self);

  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
      switch (self->data_mode)
        {
        case FTP_DATA_PASSIVE:
        case FTP_DATA_KEEP:
          switch (self->answer_cmd->str[0])
            {
            case '2':
              ret = ftp_data_server_start_PASV(self);
              if (ret == FTP_RSP_ACCEPT)
                {

                  if (!ftp_data_prepare(self, EP_CLIENT, 'L'))
                    {
                      self->data_state = 0;
                      SET_ANSWER(MSG_ERROR_PARSING_PASV);
                      /*LOG
                        Placeholder ftpcmd.c.29
                       */
                      z_proxy_log(self, FTP_ERROR, 2, "Error preparing client listen (PASV);");
                      z_proxy_leave(self);
                      return FTP_RSP_REJECT;
                    }
                  
                  if (self->masq_address[EP_CLIENT]->len)
                    g_strlcpy(tmpaddr, self->masq_address[EP_CLIENT]->str, sizeof(tmpaddr));
                  else
                    z_inet_ntoa(tmpaddr, sizeof(tmpaddr), ((struct sockaddr_in *) &self->data_local[EP_CLIENT]->sa)->sin_addr);
                  g_strdelimit(tmpaddr, ".", ',');

                  port = ntohs(((struct sockaddr_in *) &self->data_local[EP_CLIENT]->sa)->sin_port);
                  if (port == 0)
                    {
                      SET_ANSWER(MSG_ERROR_PARSING_PASV);
                      self->data_state = 0;
                      /*LOG
                        Placeholder ftpcmd.c.30
                       */
                      z_proxy_log(self, FTP_ERROR, 2, "Error preparing client listen (PASV);");
                      z_proxy_leave(self);
                      return FTP_RSP_REJECT;
                    }

                  g_snprintf(tmpport, sizeof(tmpport), ",%d,%d", (port & 0xff00) >> 8, port & 0x00ff);

                  g_strlcpy(tmpline, self->answer_param->str, sizeof(tmpline));

                  start = strchr(tmpline, '(');
                  end = NULL;
                  if (start)
                    {
                      *start = 0;
                      end = strchr(start, ')');
                      self->answer_param = g_string_assign(self->answer_param, tmpline);
                    }

                  self->answer_param = g_string_append_c(self->answer_param, '(');
                  self->answer_param = g_string_append(self->answer_param, tmpaddr);
                  self->answer_param = g_string_append(self->answer_param, tmpport);
                  self->answer_param = g_string_append_c(self->answer_param, ')');

                  if (end)
                    self->answer_param = g_string_append(self->answer_param, end + 1);
                }
              self->ftp_state = FTP_STATE_DATA;
              break;
            default:
              self->data_state = 0;
              z_proxy_leave(self);
              return FTP_RSP_ACCEPT;
            }
          break;
        case FTP_DATA_ACTIVE:
          switch (self->answer_cmd->str[0])
            {
            case '2':
              if (!ftp_data_prepare(self, EP_CLIENT, 'L'))
                {
                  self->data_state = 0;
                  SET_ANSWER(MSG_ERROR_PARSING_PASV);
                  /*LOG
                    Placeholder ftpcmd.c.31
                   */
                  z_proxy_log(self, FTP_ERROR, 2, "Error preparing client listen (PASV);");
                  z_proxy_leave(self);
                  return FTP_RSP_REJECT;
                }

              self->answer_cmd = g_string_assign(self->answer_cmd, "227");
              self->answer_param = g_string_assign(self->answer_param, "Entering Passive mode (");
              if (self->masq_address[EP_CLIENT]->len)
                g_strlcpy(tmpaddr, self->masq_address[EP_CLIENT]->str, sizeof(tmpaddr));
              else
                z_inet_ntoa(tmpaddr, sizeof(tmpaddr), ((struct sockaddr_in *) &self->data_local[EP_CLIENT]->sa)->sin_addr);
              
              g_strdelimit(tmpaddr, ".", ',');

              port = ntohs(((struct sockaddr_in *) &self->data_local[EP_CLIENT]->sa)->sin_port);
              if(port==0)
                {
                  SET_ANSWER(MSG_ERROR_PARSING_PASV);
                  self->data_state = 0;
                  /*LOG
                    Placeholder ftpcmd.c.32
                   */
                  z_proxy_log(self, FTP_ERROR, 2, "Error preparing client listen (PASV);");
                  z_proxy_leave(self);
                  return FTP_RSP_REJECT;
                }

              g_snprintf(tmpport, sizeof(tmpport), ",%d,%d", (port & 0xff00) >> 8, port & 0x00ff);
              self->answer_param = g_string_append(self->answer_param, tmpaddr);
              self->answer_param = g_string_append(self->answer_param, tmpport);
              self->answer_param = g_string_append_c(self->answer_param, ')');
              self->answer_param = g_string_append_c(self->answer_param, '.');

              self->ftp_state = FTP_STATE_DATA;
              break;
            default:
              SET_ANSWER(MSG_ERROR_PARSING_PASV);
              self->data_state = 0;
              /*LOG
                Placeholder ftpcmd.c.33
               */
              z_proxy_log(self, FTP_VIOLATION, 2, "Bad parameter (PASV);");
              z_proxy_leave(self);
              return FTP_RSP_REJECT;
            }
          break;
        default:
          break;
        }
    }
  z_proxy_leave(self);
  return ret;
}

guint
ftp_command_parse_EPRT(FtpProxy *self)
{
  guint16 port;
  guint res = FTP_REQ_ACCEPT;
  gchar **split;
  gchar delim[2];
  gchar *err;

  if (self->ftp_state == FTP_STATE_DATA)
    {
      self->ftp_state = FTP_STATE_CONVERSATION;
      ftp_data_reset(self);
    }

  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
  
      if (self->request_param->len <= 0)
        {
          /*LOG
            Placeholder ftpcmd.c.34
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "Missing parameter (PASV);");
          z_proxy_leave(self);
          return FTP_RSP_REJECT;
        }

      delim[0] = self->request_param->str[0];
      delim[1] = 0;
      split = g_strsplit(self->request_param->str, delim, 6);
  
      if (split[0] == NULL ||
          split[1] == NULL ||
          split[2] == NULL ||
          split[3] == NULL ||
          split[4] == NULL ||
          split[5] != NULL)
        {
          SET_ANSWER(MSG_ERROR_PARAMETER_EPRT);
          g_strfreev(split);
          /*LOG
            Placeholder ftpcmd.c.35
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "Bad parameter (EPRT);");
          z_proxy_leave(self);
          return FTP_RSP_REJECT;
        }
  
      if (strcmp(split[1],"1") == 0)
        {
          port = strtol(split[3], &err, 10);
  
          if (port == 0 || *err != 0)
            {
              SET_ANSWER(MSG_ERROR_PARAMETER_EPSV);
              g_strfreev(split);
              /*LOG
                Placeholder ftpcmd.c.36
               */
              z_proxy_log(self, FTP_VIOLATION, 2, "Bad port parameter (EPRT);");
              z_proxy_leave(self);
              return FTP_RSP_REJECT;
            }
        }
      else
        {
          SET_ANSWER(MSG_ERROR_PARAMETER_EPRT);
          g_strfreev(split);
          /*LOG
            Placeholder ftpcmd.c.37
           */
          z_proxy_log(self, FTP_VIOLATION, 2, "Unknown protocol method (EPRT);");
          z_proxy_leave(self);
          return FTP_RSP_REJECT;
        }
      
      self->data_remote[EP_CLIENT] = z_sockaddr_inet_new(split[2], port);
      
      g_strfreev(split);

      switch (self->data_mode)
        {
        case FTP_DATA_PASSIVE:
           self->request_cmd = g_string_assign(self->request_cmd, "EPSV");
           self->request_param = g_string_assign(self->request_param, "");
           break;
        case FTP_DATA_ACTIVE:
        case FTP_DATA_KEEP:
          res = ftp_data_server_start_EPRT(self);
          break;
        default:
          /*LOG
            This message attempt when data_mode set to wrong value.
           */
          z_proxy_log(self, FTP_POLICY, 1, "Connection mode not supported; data_mode='%d'", self->data_mode);
          SET_ANSWER(MSG_ERROR_PARSING_PORT);
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      break;
    default:
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }
  z_proxy_leave(self);
  return res;
}

guint ftp_command_answer_EPRT(FtpProxy *self)
{
  guint res = FTP_RSP_ACCEPT;

  z_proxy_enter(self);

  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
      switch (self->data_mode)
        {
        case FTP_DATA_PASSIVE:
          switch (self->answer_cmd->str[0])
            {
            case '2':
              res = ftp_data_server_start_EPSV(self);
              if (res == FTP_RSP_ACCEPT)
                {
                  if (!ftp_data_prepare(self, EP_CLIENT, 'C'))
                    {
                      self->data_state = 0;
                      SET_ANSWER(MSG_ERROR_PARSING_PORT);
                      /*LOG
                        Placeholder ftpcmd.c.38
                       */
                      z_proxy_log(self, FTP_ERROR, 2, "Error preparing client connect (EPRT);");
                      z_proxy_leave(self);
                      return FTP_RSP_REJECT;
                    }

                  SET_ANSWER(MSG_PORT_SUCCESFULL);
                  res = FTP_RSP_ACCEPT;
                }
              self->ftp_state = FTP_STATE_DATA;
              break;
            default:
                {
                  SET_ANSWER(MSG_ERROR_PARSING_PORT);
                  self->data_state = 0;
                  /*LOG
                    Placeholder ftpcmd.c.39
                   */
                  z_proxy_log(self, FTP_VIOLATION, 2, "Bad server answer (EPRT);");
                  z_proxy_leave(self);
                  return FTP_RSP_REJECT;
                }
            }
          break;
        case FTP_DATA_ACTIVE:
        case FTP_DATA_KEEP:
          switch (self->answer_cmd->str[0])
            {
            case '2':

              if (!ftp_data_prepare(self, EP_CLIENT, 'C'))
                {
                  self->data_state = 0;
                  SET_ANSWER(MSG_ERROR_PARSING_PORT);
                  /*LOG
                    Placeholder ftpcmd.c.40
                   */
                  z_proxy_log(self, FTP_ERROR, 2, "Error preparing client connect (EPRT);");
                  z_proxy_leave(self);
                  return FTP_RSP_REJECT;
                }
              self->ftp_state = FTP_STATE_DATA;
              break;
            default:
              self->data_state = 0;
              z_proxy_leave(self);
              return FTP_RSP_ACCEPT;
            }
          break;
        default:
          break;
        }
    }
  z_proxy_leave(self);
  return res;
}

guint
ftp_command_parse_EPSV(FtpProxy *self)
{
  guint res = FTP_REQ_ACCEPT;

  z_proxy_enter(self);

  if (self->ftp_state == FTP_STATE_DATA)
    {
      self->ftp_state = FTP_STATE_CONVERSATION;
      ftp_data_reset(self);
    }

  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
      self->request_param = g_string_assign(self->request_param, "");
      self->data_state = 0;

      switch (self->data_mode)
        {
        case FTP_DATA_KEEP:
        case FTP_DATA_PASSIVE:
          break;
        case FTP_DATA_ACTIVE:
          self->request_cmd = g_string_assign(self->request_cmd, "EPRT");
          self->request_param = g_string_assign(self->request_param, "");
          res = ftp_data_server_start_EPRT(self);
          break;
        default:
          /*LOG
            This message attempt when data_mode set to wrong value.
           */
          z_proxy_log(self, FTP_POLICY, 1, "Connection mode not supported; data_mode='%d'", self->data_mode);
          SET_ANSWER(MSG_ERROR_PARSING_PORT);
          z_proxy_leave(self);
          return FTP_REQ_REJECT;
        }
      break;
    default:
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }

  z_proxy_leave(self);
  return FTP_REQ_ACCEPT;
}

guint
ftp_command_answer_EPSV(FtpProxy *self)
{
  gchar *start, *end;
  guint port;
  gchar tmpline[FTP_LINE_MAX_LEN];
  gchar tmpport[10];
  guint ret = FTP_RSP_ACCEPT;

  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
      switch (self->data_mode)
        {
        case FTP_DATA_PASSIVE:
        case FTP_DATA_KEEP:
          switch (self->answer_cmd->str[0])
            {
            case '2':
              ret = ftp_data_server_start_EPSV(self);
              if (ret == FTP_RSP_ACCEPT)
                {
                  if (!ftp_data_prepare(self, EP_CLIENT, 'L'))
                    {
                      self->data_state = 0;
                      SET_ANSWER(MSG_ERROR_PARSING_PASV);
                      /*LOG
                        Placeholder ftpcmd.c.41
                       */
                      z_proxy_log(self, FTP_ERROR, 2, "Error preparing client listen (EPSV);");
                      z_proxy_leave(self);
                      return FTP_RSP_REJECT;
                    }

//                  if (self->masq_address[EP_CLIENT]->len)
//                    g_strlcpy(tmpaddr, self->masq_address[EP_CLIENT]->str, sizeof(tmpaddr));
//                  else
//                    z_inet_ntoa(tmpaddr, sizeof(tmpaddr), ((struct sockaddr_in *) &self->data_local[EP_CLIENT]->sa)->sin_addr);

                  port = ntohs(((struct sockaddr_in *) &self->data_local[EP_CLIENT]->sa)->sin_port);
                  if (port == 0)
                    {
                      SET_ANSWER(MSG_ERROR_PARSING_PASV);
                      self->data_state = 0;
                      /*LOG
                        Placeholder ftpcmd.c.42
                       */
                      z_proxy_log(self, FTP_ERROR, 2, "Error preparing client listen (EPSV);");
                      z_proxy_leave(self);
                      return FTP_RSP_REJECT;
                    }

                  g_snprintf(tmpport, sizeof(tmpport), "%d", port );

                  g_strlcpy(tmpline, self->answer_param->str, sizeof(tmpline));

                  start = strchr(tmpline, '(');
                  end = NULL;
                  if (start)
                    {
                      *start = 0;
                      end = strchr(start + 1, ')');
                      self->answer_param = g_string_assign(self->answer_param, tmpline);
                    }

                  self->answer_param = g_string_append(self->answer_param, "(|||");
//                  self->answer_param = g_string_append(self->answer_param, tmpaddr);
                  self->answer_param = g_string_append(self->answer_param, tmpport);
                  self->answer_param = g_string_append(self->answer_param, "|)");

                  if (end)
                    self->answer_param = g_string_append(self->answer_param, end + 1);
                }
              self->ftp_state = FTP_STATE_DATA;
              break;
            default:
              self->data_state = 0;
              z_proxy_leave(self);
              return FTP_RSP_ACCEPT;
            }
          break;
        case FTP_DATA_ACTIVE:
          switch (self->answer_cmd->str[0])
            {
            case '2':

              if (!ftp_data_prepare(self, EP_CLIENT, 'L'))
                {
                  self->data_state = 0;
                  SET_ANSWER(MSG_ERROR_PARSING_PASV);
                  /*LOG
                    Placeholder ftpcmd.c.43
                   */
                  z_proxy_log(self, FTP_ERROR, 2, "Error preparing client listen (EPSV);");
                  z_proxy_leave(self);
                  return FTP_RSP_REJECT;
                }

              self->answer_cmd = g_string_assign(self->answer_cmd, "229");
              self->answer_param = g_string_assign(self->answer_param, "Entering Extended Passive Mode (|||");

//              if (self->masq_address[EP_CLIENT]->len)
//                g_strlcpy(tmpaddr, self->masq_address[EP_CLIENT]->str, sizeof(tmpaddr));
//              else
//                z_inet_ntoa(tmpaddr, sizeof(tmpaddr), ((struct sockaddr_in *) &self->fw_client_data->sa)->sin_addr);

              port = ntohs(((struct sockaddr_in *) &self->data_local[EP_CLIENT]->sa)->sin_port);
              if(port==0)
                {
                  SET_ANSWER(MSG_ERROR_PARSING_PASV);
                  self->data_state = 0;
                  /*LOG
                    Placeholder ftpcmd.c.44
                   */
                  z_proxy_log(self, FTP_ERROR, 2, "Error preparing client listen (EPSV);");
                  z_proxy_leave(self);
                  return FTP_RSP_REJECT;
                }

              g_snprintf(tmpport, sizeof(tmpport), "%d", port);
//              self->answer_param = g_string_append(self->answer_param, tmpaddr);
              self->answer_param = g_string_append(self->answer_param, tmpport);
              self->answer_param = g_string_append(self->answer_param, "|)");

              self->ftp_state = FTP_STATE_DATA;
              break;
            default:
              SET_ANSWER(MSG_ERROR_PARSING_PASV);
              self->data_state = 0;
              /*LOG
                Placeholder ftpcmd.c.45
               */
              z_proxy_log(self, FTP_VIOLATION, 2, "Bad server answer (EPSV);");
              z_proxy_leave(self);
              return FTP_RSP_REJECT;
            }
          break;
        default:
          break;
        }
    }
  z_proxy_leave(self);
  return ret;
}

guint
ftp_command_answer_RNFR(FtpProxy *self)
{
  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
      switch(self->answer_code)
        {
          case 350:
            self->ftp_state = FTP_STATE_RENAME;
          break;
        }
      break;
    default:
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      return FTP_RSP_REJECT;
    }
  z_proxy_leave(self);
  return FTP_RSP_ACCEPT;
}

guint
ftp_command_parse_RNTO(FtpProxy *self)
{
  guint res = FTP_REQ_ACCEPT;
  
  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_RENAME:
      self->ftp_state = FTP_STATE_CONVERSATION;
      res = ftp_command_parse_path(self);
      break;
    default:
      SET_ANSWER(MSG_RNFR_RNTO);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }
  z_proxy_leave(self);
  return res;
}

guint
ftp_command_parse_ALLO(FtpProxy *self)
{
  long int num1;
  long int num2;
  char *str;
  char *endptr;
  
  z_proxy_enter(self);
  switch (self->ftp_state)
    {
    case FTP_STATE_CONVERSATION:
    case FTP_STATE_DATA:
      str = self->request_param->str;
      num1 = strtol(str, &endptr, 10);
      if (*endptr == 0)
        {
          z_proxy_leave(self);
          return FTP_REQ_ACCEPT;
        }
      
      if (endptr[0] == ' ' && endptr[1] == 'R' && endptr[2] == ' ')
        {
          str = endptr + 2;
          num2 = strtol(str, &endptr, 10);
          if (*endptr == 0)
            {
              z_proxy_leave(self);
              return FTP_REQ_ACCEPT;
            }
        }
      break;
    default:
      SET_ANSWER(MSG_COMMAND_NOT_ALLOWED_HERE);
      z_proxy_leave(self);
      return FTP_REQ_REJECT;
    }

  /*LOG
    Placeholder ftpcmd.c.46
   */
  z_proxy_log(self, FTP_VIOLATION, 2, "Error parsing command (ALLO);");
  z_proxy_leave(self);
  return FTP_REQ_REJECT;
}

guint
ftp_command_parse_REIN(FtpProxy *self G_GNUC_UNUSED)
{
  return FTP_RSP_ACCEPT;
}

guint
ftp_command_answer_REIN(FtpProxy *self)
{
  switch (self->answer_cmd->str[0])
    {
    case '1':
      return FTP_NOOP;
      break;
    case '2':
      self->ftp_state = FTP_STATE_LOGIN;
      self->username = g_string_assign(self->username, "");
      self->password = g_string_assign(self->password, "");
    }
  return FTP_RSP_ACCEPT;
}
