/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: httpfltr.c,v 1.41 2002/10/25 15:44:20 bazsi Exp $
 *
 * Author: Balazs Scheidler <bazsi@balabit.hu>
 * Auditor: 
 * Last audited version: 
 * Notes:
 *   Based on the code by: Viktor Peter Kovacs <vps__@freemail.hu>
 *
 ***************************************************************************/
 
#include "http.h"

#include <zorp/log.h>
#include <zorp/io.h>
 
#include <assert.h>

#define HTTP_FLTR_BLOCKING 1

#if HTTP_FLTR_BLOCKING

gboolean
http_transfer_filter_copy(HttpProxy *self, int from, int to, gint content_length)
{
  gchar buf[HTTP_BLOCKSIZE];
  guint bytes, bytes_copied = 0;
  GIOStatus rc;

  z_proxy_enter(self);  
  if ((from ^ to) != 1)
    {
      z_proxy_leave(self);
      return FALSE;
    }
  if (self->max_body_length && ((guint) content_length > self->max_body_length))
    {
      z_proxy_log(self, HTTP_POLICY, 2, "Body too long;");
      z_proxy_leave(self);
      return FALSE;
    }
  while (content_length == -1 || content_length > 0)
    {
      bytes = MIN(sizeof(buf), (guint) content_length);

      rc = http_read(self, from, buf, bytes, &bytes);

      if (rc == G_IO_STATUS_EOF)
        {
          bytes = 0;
        }
      else if (rc != G_IO_STATUS_NORMAL)
        {
          z_proxy_leave(self);
          return FALSE;
        }

      if (self->max_body_length && (bytes_copied + bytes > self->max_body_length))
        {
          bytes = self->max_body_length - bytes_copied;
        }
        
      if (bytes == 0)
        break;
      
      if (http_write(self, to, buf, bytes) != G_IO_STATUS_NORMAL)
        {
          z_proxy_leave(self);
          return FALSE;
        }
      bytes_copied += bytes;
      if (self->max_body_length && bytes_copied >= self->max_body_length)
        {
          /*LOG
            Placeholder fltr.1
           */
          z_proxy_log(self, HTTP_POLICY, 2, "Body too long;");
          z_proxy_leave(self);
          return FALSE;
        }
      if (content_length != -1)
        content_length -= bytes;
    }
  z_proxy_leave(self);
  return TRUE;
}

gboolean
http_transfer_filter_chunked(HttpProxy *self, int from, int to, gint content_length G_GNUC_UNUSED)
{
  gchar buf[32];
  guint bytes_copied = 0;
  
  /* NOTE: we strip off extra data from the end of the block size, ie:
     HTTP permits using the following format:
     
       0020; eof
     
     where eof is the extradata, we strip this off, since we have no way
     verifying it. */
  z_proxy_enter(self);

  if ((from ^ to) != 1)
    {
      z_proxy_leave(self);
      return FALSE;
    }

  while (1)
    {
      gchar *line;
      guint line_length, res, content_length;
      
      res = z_stream_line_get(self->super.endpoints[from], &line, &line_length, NULL);
      if (res != G_IO_STATUS_NORMAL)
        {
          z_proxy_leave(self);
          return FALSE;
        }
        
      {
        gchar *end;
              
        if (line_length > sizeof(buf) - 3)
          {
            z_proxy_leave(self);
            return FALSE;
          }
        strncpy(buf, line, line_length);
        buf[line_length] = 0;
        content_length = strtoul(buf, &end, 16);
        if (end == buf)
          {
            /* no characters processed */
            z_proxy_leave(self);
            return FALSE;
          }
        else
          line_length = end - buf;
        buf[line_length] = '\r';
        buf[line_length + 1] = '\n';
        buf[line_length + 2] = 0;
      }
      
      if (self->max_chunk_length && content_length > self->max_chunk_length)
        {
          /*LOG
            Placeholder fltr.2
           */
          z_proxy_log(self, HTTP_POLICY, 2, "Chunk too large; length='%d', max_chunk_length='%d'", content_length, self->max_chunk_length);
          z_proxy_leave(self);
          return FALSE;
        }
      if (self->max_body_length && bytes_copied + content_length > self->max_body_length)
        {
          content_length = self->max_body_length - bytes_copied;
          line_length = g_snprintf(buf, sizeof(buf), "%d\r\n", content_length) - 2;
        }
      /* send block size */
      if (http_write(self, to, buf, line_length + 2) != G_IO_STATUS_NORMAL)
        {
          z_proxy_leave(self);
          return FALSE;
        }
      if (content_length != 0)
        {
          /* copy block */
          if (!http_transfer_filter_copy(self, from, to, content_length))
            {
              z_proxy_leave(self);
              return FALSE;
            }
        }

      bytes_copied += content_length;
      if (self->max_body_length && bytes_copied >= self->max_body_length)
        {
          strcpy(buf, "\r\n0\r\n");
          if (http_write(self, to, buf, 5) != G_IO_STATUS_NORMAL)
            {
              z_proxy_leave(self);
              return FALSE;
            }
          /*LOG
            Placeholder fltr.3
           */
          z_proxy_log(self, HTTP_POLICY, 2, "Body too long;");
          z_proxy_leave(self);
          return FALSE;
        }

      res = z_stream_line_get(self->super.endpoints[from], &line, &line_length, NULL);
      if (res != G_IO_STATUS_NORMAL || line_length != 0) //FIXME This is good?
        {
          z_proxy_leave(self);
          return FALSE;
        }

      strcpy(buf, "\r\n");
      if (http_write(self, to, buf, 2) != G_IO_STATUS_NORMAL)
        {
          z_proxy_leave(self);
          return FALSE;
        }
      if (content_length == 0)
        break;
    }
  z_proxy_leave(self);
  return TRUE;
}

gboolean
http_data_transfer_setup(HttpProxy *self, guint from, guint to, gboolean expect_data)
{
  HttpHeaders *headers = &self->headers[from];
  HttpHeader *h;

  z_proxy_enter(self);

  self->transfer_from = from;
  self->transfer_to = to;

  self->content_length = -2;  // -1 unknown, -2 no data
  self->transfer_filter = http_transfer_filter_copy;

  if (http_lookup_header(headers, "Transfer-Encoding", &h) &&
      strcasecmp(h->value->str, "chunked") == 0)
    {
      self->transfer_filter = http_transfer_filter_chunked;
      self->content_length = -1;
    }
  else if (http_lookup_header(headers, "Content-Length", &h))
    {
      char *end;

      self->content_length = strtoul(h->value->str, &end, 10);
      if ((guint) (end - h->value->str) != h->value->len)
        {
          /* content-length not a number */
          /*LOG
            Placeholder fltr.4
           */
	  z_proxy_log(self, HTTP_VIOLATION, 1, "The header 'Content-Length' was present, but is not a number;");
          z_proxy_leave(self);
          return FALSE;
        }
      if (self->content_length < 0)
        {
          /*LOG
            Placeholder fltr.5
           */
          z_proxy_log(self, HTTP_VIOLATION, 1, "The header 'Content-Length' contained a negative number; value='%d'", self->content_length);
          z_proxy_leave(self);
          return FALSE;
        }
      if (self->max_body_length && self->max_body_length < (guint) self->content_length)
        {
          g_string_sprintf(h->value, "%d", self->max_body_length);
          self->content_length = self->max_body_length;
        }
      if (self->content_length == 0)
	self->content_length = -2;

      self->transfer_filter = http_transfer_filter_copy;
    }
  else if (expect_data)
    {
      self->content_length = -1;
      self->connection_mode = HTTP_CONNECTION_CLOSE;
    }
  else
    { 
      self->content_length = -2;
    }

  z_proxy_leave(self);
  return TRUE;
}

gboolean
http_data_transfer(HttpProxy *self)
{
  gint from = self->transfer_from;
  gint to = self->transfer_to;

  z_proxy_enter(self);

  if (self->content_length != -2)
    {
      if (!self->transfer_filter(self, from, to, self->content_length))
        {
          /* FIXME: transfer unsuccessful */
          /*LOG
            Placeholder fltr.6
           */
          z_proxy_log(self, HTTP_VIOLATION, 1, "Data transfer failed;");
          self->error_code = HTTP_MSG_OK; /* error message may corrupt already sent data */
          z_proxy_leave(self);
          return FALSE;
        }
    }
  z_proxy_leave(self);
  return TRUE;
}

#else

static inline void
http_entity_append(HttpProxy *self, char *buf, gint buflen)
{
  self->entity_buffer = g_realloc(self->entity_buffer, self->entity_len + buflen);
  memcpy(&self->entity_buffer[self->entity_len], buf, buflen);
  self->entity_len += buflen;
}

static inline ZStream *
http_lookup_stream(HttpProxy *self, gint ndx)
{
  if (ndx < EP_MAX)
    {
      return self->super.endpoints[ndx];
    }
  else
    {
      assert(self->stacked);
      return self->stacked->downstreams[ndx & 3];
    }
}

static inline GIOStatus
http_transfer_read(HttpProxy *self, gint side, gchar *buf, gint buflen, gint *bytes_read)
{
  ZStream *input = http_lookup_stream(self, side);
  guint rc;

  if (side & HTTP_EP_DOWN)
    rc = z_stream_read(input, buf, buflen, bytes_read);
  else
    rc = http_read(self, side, buf, buflen, bytes_read);

  return rc;
}

static inline GIOStatus
http_transfer_write(HttpProxy *self, gint side, gchar *buf, gint buflen, gint *bytes_written)
{
  ZStream *output = http_lookup_stream(self, side);
  guint rc;

  /* buffer not empty */
  if (side & HTTP_EP_DOWN)
    rc = z_stream_write(output, buf, buflen, bytes_written);
  else
    {
      rc = http_write(self, side, buf, buflen);
      *bytes_written = buflen;
    }
  
  return rc;
}

static guint
http_transfer_read_buf(HttpProxy *self, gint side, HttpIOBuffer *buf, gint max_len)
{
  gint rc;
  gint br;

  z_proxy_enter(self);

  if (max_len == 0)
    {
      z_proxy_leave(self);
      return ST_CLOSE | ST_EOF;
    }
  rc = http_transfer_read(self, side, &buf->buf[buf->end], max_len == -1 ? sizeof(buf->buf) - buf->end : MIN(max_len, sizeof(buf->buf) - buf->end), &br);
  buf->end += br;
  z_proxy_leave(self);
  return rc;
}

static guint
http_transfer_write_buf(HttpProxy *self, gint side, HttpIOBuffer *buf)
{
  int rc;
  guint bw;
  
  z_proxy_enter(self);
  if (buf->ofs != buf->end)
    {
      rc = http_transfer_write(self, side, &buf->buf[buf->ofs], buf->end - buf->ofs, &bw);
      if (ST_ERROR(rc) == ST_OK || ST_ERROR(rc) == ST_AGAIN)
        {
          buf->ofs += bw;
        }
      else
        {
          return rc;
        }
      if (buf->ofs != buf->end)
        {
          z_proxy_leave(self);
          return ST_AGAIN | ST_GOON;
        }
    }
  z_proxy_leave(self);
  return ST_OK | ST_GOON;
}

/*
 * copies the given amount of data from from_ndx to to_ndx using buf
 * as buffer. It returns the number of bytes copied in the bytes_copied 
 * argument.
 *
 * It returns ST_AGAIN if the function is to be called again, and in this
 * case it sets up want_read & want_write properties of the streams.  
 * 
 * It returns ST_OK if bytes up to max_len were successfully copied.
 * want_read & want_write are both set to FALSE in this case.
 */
static guint 
http_transfer_copy_data(HttpProxy *self, gint from_ndx, gint to_ndx, HttpIOBuffer *buf, gint max_len, gint *bytes_copied)
{
  ZStream *from = http_lookup_stream(self, from_ndx);
  ZStream *to = http_lookup_stream(self, to_ndx);
  guint rc;
  
  from->want_read = FALSE;
  to->want_write = FALSE;
  *bytes_copied = 0;
  
  /* flush already buffered data if present */
  rc = http_transfer_write_buf(self, to_ndx, buf);

  if (ST_ERROR(rc) != ST_OK)
    {
      z_proxy_leave(self);
      return rc;
    }

  while (1)
    {
      buf->ofs = buf->end = 0;
      
      if (max_len == 0)
        {
          z_proxy_leave(self);
          return ST_OK | ST_GOON;
        }
      
      rc = http_transfer_read_buf(self, from_ndx, buf, max_len);
      if (ST_ERROR(rc) == ST_OK)
        {
          max_len -= buf->end;
          *bytes_copied += buf->end;
          rc = http_transfer_write_buf(self, to_ndx, buf);
          
          if (ST_ERROR(rc) == ST_AGAIN)
            {
              to->want_write = TRUE;
              break;
            }
          else if (ST_FAILED(rc))
            {
              /*LOG
                Placeholder fltr.7
                This is'n a policy comformant log message
                FIXMEEE
               */
              z_proxy_log(self, HTTP_ERROR, 1, "Error writing %s (%m)", to->name);
              z_proxy_leave(self);
              return ST_QUIT | ST_OK;
            }
        }
      else if (ST_ERROR(rc) == ST_AGAIN)
        {
          break;
        }
      else if (ST_ERROR(rc) == ST_EOF)
        {
          z_proxy_leave(self);
          return ST_EOF | ST_GOON;
        }
      else if (ST_FAILED(rc))
        {
          /*LOG
            Placeholder fltr.8
            This is'n a policy comformant log message
            FIXMEEE
           */
          z_proxy_log(self, HTTP_ERROR, 3, "Error reading %s (%m)", from->name);
          return ST_QUIT | ST_OK;
        }
    }

  /* we have ST_AGAIN, either we need to read on, or write the already buffered data */
  
  if (!to->want_write && buf->ofs == buf->end)
    {
      from->want_read = TRUE;
    }
    
  z_proxy_leave(self);
  return rc;
}

static guint
http_transfer_dechunk_client_to_down(ZStream *stream, GIOCondition cond)
{
  HttpProxy *self = (HttpProxy *) stream->user_data;
  ZStream *from = self->super.endpoints[self->transfer_from];
  ZStream *to = self->stacked->downstreams[EP_CLIENT];
  HttpIOBuffer *buf = self->transfer_buffer[0];
  guint res;
  gchar *line, line_buf[32];
  guint line_length, chunk_length;
  gint bytes_copied;

  z_proxy_enter(self);

  while (1)
    {
      from->want_read = FALSE;
      to->want_write = FALSE;
      
      switch (self->transfer_state[0])
        {
        case 0:
          {
            res = z_read_line_get(self->readline[self->transfer_from], &line, &line_length);
            if (ST_ERROR(res) == ST_AGAIN)
              {
                from->want_read = TRUE;
                z_proxy_leave(self);
                return ST_AGAIN | ST_GOON;
              }
            else if (ST_ERROR(res) != ST_OK)
              {
                z_proxy_leave(self);
                return ST_ERR | ST_QUIT;
              }
              
            {
              gchar *end;
                    
              if (line_length > sizeof(line_buf) - 3)
                {
                  z_proxy_leave(self);
                  return ST_ERR | ST_QUIT;
                }
              strncpy(line_buf, line, line_length);
              line_buf[line_length] = 0;
              chunk_length = strtoul(line_buf, &end, 16);
              if (end == line_buf)
                {
                  /* no characters processed */
                  z_proxy_leave(self);
                  return ST_ERR | ST_QUIT;
                }
            }
            
            self->content_length = self->transfer_length[0] = chunk_length;
            self->transfer_state[0] = 1;
          }
        case 1:
          {
            res = http_transfer_copy_data(self, self->transfer_from, EP_CLIENT | HTTP_EP_DOWN, 
                                          buf, self->transfer_length[0], 
                                          &bytes_copied);
            if (ST_ERROR(res) == ST_AGAIN)
              {
                self->transfer_length[0] -= bytes_copied;
                return ST_AGAIN | ST_GOON;
              }
            else if (ST_ERROR(res) == ST_OK)
              {
                self->transfer_length[0] -= bytes_copied;
                assert(!self->transfer_length[0]);
                self->transfer_state[0] = 2; /* go on reading the footer */
              }
            else if (self->transfer_length[0] != -1 && ST_ERROR(res) == ST_EOF)
              {
                /* unexpected EOF */
                return ST_QUIT | ST_ERR;
              }
            else
              {
                return ST_QUIT | ST_ERR;
              }
            /* fallthrough */
          }
        case 2:
          {
            /* fetch footer */
            
            res = z_read_line_get(self->readline[self->transfer_from], &line, &line_length);
            if (ST_ERROR(res) == ST_AGAIN)
              {
                from->want_read = TRUE;
                z_proxy_leave(self);
                return ST_AGAIN | ST_GOON;
              }
            else if (ST_ERROR(res) != ST_OK || line_length != 0)
              {
                z_proxy_leave(self);
                return ST_ERR | ST_QUIT;
              }

            if (self->content_length == 0)
              {
                /* the previous was the last block */
                self->transfer_state[0] = 3;
              }
            else
              {
                /* fetch next block */
                self->transfer_state[0] = 0;
              }
            
            /* fall through */
          }
        case 3:
          return ST_QUIT | ST_OK;

        }
    }
  z_proxy_leave(self);
  return ST_OK | ST_GOON;
}

static guint
http_transfer_enchunk_down_to_server(ZStream *stream, GIOCondition cond)
{
  HttpProxy *self = (HttpProxy *) stream->user_data;
  ZStream *from = self->stacked->downstreams[EP_SERVER];
  ZStream *to = self->super.endpoints[self->transfer_to];
  HttpIOBuffer *buf = self->transfer_buffer[1];
  guint res;
  gchar line_buf[32];
  guint line_length;

  z_proxy_enter(self);
  
  while (1)
    {
      from->want_read = FALSE;
      to->want_write = FALSE;

      switch (self->transfer_state[1])
        {
        case 0:
          {
            /* some ugly trickery to get the block prefix right before 
             * to the datablock in our output buffer */
            buf->ofs = buf->end = 6;
            res = http_transfer_read_buf(self, EP_SERVER | HTTP_EP_DOWN, buf, -1);
            
            self->transfer_state[1] = 1;
            if (ST_ERROR(res) == ST_EOF)
              {
                memcpy(buf->buf, "0\r\n\r\n", 5);
                self->transfer_state_next[1] = 2;
              }
            else if (ST_ERROR(res) == ST_AGAIN)
              {
                from->want_read = TRUE;
                return ST_AGAIN | ST_GOON;
              }
            else
              {
                line_length = snprintf(line_buf, sizeof(line_buf), "%x\r\n", buf->end - 6);
                memcpy(&buf->buf[6 - line_length], line_buf, line_length);
                buf->ofs = 6 - line_length;
                
                self->transfer_state_next[1] = 0;
              }
            /* fallthrough */
          }
        case 1:
          {
            res = http_transfer_write_buf(self, self->transfer_to, buf);
            if (ST_ERROR(res) == ST_AGAIN)
              {
                to->want_write = TRUE;
                return ST_AGAIN | ST_GOON;
              }
            else if (ST_ERROR(res) == ST_OK)
              {
                self->transfer_state[1] = self->transfer_state_next[1];
              }
            else
              {
                return ST_ERR | ST_QUIT;
              }
          }
        case 2:
          {
            return ST_QUIT | ST_OK;
          }
        }
    }
  
  z_proxy_leave(self);
  return ST_OK | ST_GOON;
}

static guint
http_transfer_copy_any_to_any(HttpProxy *self, gint side, gint from, gint to)
{
  gint bytes_copied;
  guint rc;

  if ((from & HTTP_EP_DOWN) == 0 && self->transfer_state[side] == 0)
    {
      self->transfer_length[side] = self->content_length;
      self->transfer_state[side]++;
    }
  else
    {
      self->transfer_length[side] = -1;
    }
  rc = http_transfer_copy_data(self, from, to, self->transfer_buffer[side], self->transfer_length[side], &bytes_copied);
  if (self->transfer_length[side] != -1 && ST_ERROR(rc) == ST_EOF)
    {
      /* unexpected EOF */
      return ST_QUIT | ST_OK;
    }
  else if (ST_ERROR(rc) == ST_AGAIN)
    {
      if (self->transfer_length[side] != -1)
        self->transfer_length[side] -= bytes_copied;
      return ST_OK | ST_GOON;
    }
  else 
    {
      return ST_QUIT | ST_OK;
    }
}

static guint 
http_transfer_copy_client_to_down(ZStream *stream, GIOCondition cond)
{
  HttpProxy *self = (HttpProxy *) stream->user_data;
  
  return http_transfer_copy_any_to_any(self, 0, self->transfer_from, EP_CLIENT | HTTP_EP_DOWN);
}

static guint 
http_transfer_copy_down_to_server(ZStream *stream, GIOCondition cond)
{
  HttpProxy *self = (HttpProxy *) stream->user_data;
  
  return http_transfer_copy_any_to_any(self, 1, EP_SERVER | HTTP_EP_DOWN, self->transfer_to);
}

static guint 
http_transfer_buffer_down_to_server(ZStream *stream, GIOCondition cond)
{
  HttpProxy *self = (HttpProxy *) stream->user_data;
  ZStream *from = self->stacked->downstreams[EP_SERVER];
  ZStream *to = self->super.endpoints[self->transfer_to];
  HttpIOBuffer *buf = self->transfer_buffer[1];
  guint rc;

  while (1)
    {
      from->want_read = FALSE;
      to->want_write = FALSE;
      switch (self->transfer_state[1])
        {
        case 0:
          {
            buf->ofs = buf->end = 0;
            rc = http_transfer_read_buf(self, EP_SERVER | HTTP_EP_DOWN, buf, -1);
            if (ST_ERROR(rc) == ST_AGAIN)
              {
                from->want_read = TRUE;
                return ST_AGAIN | ST_GOON;
              }
            else if (ST_ERROR(rc) == ST_OK)
              {
                http_entity_append(self, buf->buf, buf->end);
              }
            else if (ST_ERROR(rc) == ST_EOF)
              {
                self->transfer_state[1] = 1;
                buf->ofs = 0;
                buf->end = g_snprintf(buf->buf, sizeof(buf->buf), "Content-Length: %d\r\n\r\n", self->entity_len);
              }
            else
              return ST_ERR | ST_QUIT;
            break;
          }
        case 1:
          {
            rc = http_transfer_write_buf(self, self->transfer_to, buf);
            if (ST_ERROR(rc) == ST_AGAIN)
              {
                to->want_write = TRUE;
                return ST_AGAIN | ST_GOON;
              }
            else if (ST_ERROR(rc) == ST_OK)
              {
                self->transfer_state[1] = 2;
                self->transfer_length[1] = 0;
              }
            else
              {
                return ST_ERR | ST_QUIT;
              }
            break;
          }
        case 2:
          {
            gint bw;
            
            rc = http_transfer_write(self, self->transfer_to, &self->entity_buffer[self->transfer_length[1]], self->transfer_length[1], &bw);
            if (ST_ERROR(rc) == ST_AGAIN || ST_ERROR(rc) == ST_OK)
              {
                self->transfer_length[1] += bw;
              }
            else
              {
                return ST_ERR | ST_QUIT;
              }
            if (self->transfer_length[1] == self->entity_len)
              {
                return ST_OK | ST_QUIT;
              }
            else
              {
                to->want_write = TRUE;
                return ST_AGAIN | ST_GOON;
              }
            break;
          }
        }
    }
  return ST_OK | ST_GOON;
}

static guint 
http_transfer_copy_client_to_server(ZStream *stream, GIOCondition cond)
{
  HttpProxy *self = (HttpProxy *) stream->user_data;
  return http_transfer_copy_any_to_any(self, 0, self->transfer_from, self->transfer_to);
}

static guint 
http_transfer_copy_chunk_client_to_server(ZStream *stream, GIOCondition cond)
{
  HttpProxy *self = (HttpProxy *) stream->user_data;
  ZStream *from = self->super.endpoints[self->transfer_from];
  ZStream *to = self->super.endpoints[self->transfer_to];
  HttpIOBuffer *buf = self->transfer_buffer[0];
  guint res;
  gchar *line, line_buf[32];
  guint line_length, chunk_length;
  gint bytes_copied;

  z_proxy_enter(self);

  while (1)
    {
      from->want_read = FALSE;
      to->want_write = FALSE;
      
      switch (self->transfer_state[0])
        {
        case 0:
          {
            res = z_read_line_get(self->readline[self->transfer_from], &line, &line_length);
            if (ST_ERROR(res) == ST_AGAIN)
              {
                from->want_read = TRUE;
                z_proxy_leave(self);
                return ST_AGAIN | ST_GOON;
              }
            else if (ST_ERROR(res) != ST_OK)
              {
                z_proxy_leave(self);
                return ST_ERR | ST_QUIT;
              }
              
            {
              gchar *end;
                    
              if (line_length > sizeof(line_buf) - 3)
                {
                  z_proxy_leave(self);
                  return ST_ERR | ST_QUIT;
                }
              strncpy(line_buf, line, line_length);
              line_buf[line_length] = 0;
              chunk_length = strtoul(line_buf, &end, 16);
              if (end == line_buf)
                {
                  /* no characters processed */
                  z_proxy_leave(self);
                  return ST_ERR | ST_QUIT;
                }
              else
                line_length = end - line_buf;
              line_buf[line_length] = '\r';
              line_buf[line_length + 1] = '\n';
              line_buf[line_length + 2] = 0;
            }
            
            memcpy(buf->buf, line_buf, line_length + 2);
            buf->end = line_length + 2;
            buf->ofs = 0;
            
            self->content_length = self->transfer_length[0] = chunk_length;
            self->transfer_state[0] = 3;
            self->transfer_state_next[0] = 1;
            break;
          }
        case 1:
          {
            res = http_transfer_copy_data(self, self->transfer_from, self->transfer_to, 
                                          self->transfer_buffer[0], self->transfer_length[0], 
                                          &bytes_copied);
            if (ST_ERROR(res) == ST_AGAIN)
              {
                self->transfer_length[0] -= bytes_copied;
                return ST_AGAIN | ST_GOON;
              }
            else if (ST_ERROR(res) == ST_OK)
              {
                self->transfer_length[0] -= bytes_copied;
                assert(!self->transfer_length[0]);
                self->transfer_state[0] = 2; /* go on reading the footer */
                from->want_read = TRUE;
              }
            else if (self->transfer_length[0] != -1 && ST_ERROR(res) == ST_EOF)
              {
                /* unexpected EOF */
                z_proxy_log(self, HTTP_ERROR, 3, "Unexpected EOF while dechunking;");
                return ST_QUIT | ST_ERR;
              }
            else
              {
                return ST_QUIT | ST_ERR;
              }
            break;
          }
        case 2:
          {
            /* fetch footer */
            
            res = z_read_line_get(self->readline[self->transfer_from], &line, &line_length);
            if (ST_ERROR(res) == ST_AGAIN)
              {
                from->want_read = TRUE;
                z_proxy_leave(self);
                return ST_AGAIN | ST_GOON;
              }
            else if (ST_ERROR(res) != ST_OK || line_length != 0)
              {
                z_proxy_leave(self);
                /*LOG
                  Placeholder fltr.9
                 */
                z_proxy_log(self, HTTP_VIOLATION, 1, "Non-empty line at the end of chunk;");
                return ST_ERR | ST_QUIT;
              }

            memcpy(buf->buf, "\r\n", 2);
            buf->end = 2;
            buf->ofs = 0;
            
            self->transfer_state[0] = 3;
            
            if (self->content_length == 0)
              {
                /* the previous was the last block */
                self->transfer_state_next[0] = 4;
              }
            else
              {
                /* fetch next block */
                self->transfer_state_next[0] = 0;
              }
            
            break;
          }

        case 3:
          {
            /* flush output buffer and return to self->transfer_state_next[0] */

            res = http_transfer_write_buf(self, self->transfer_to, buf);
            if (ST_ERROR(res) == ST_AGAIN)
              {
                to->want_write = TRUE;
                return ST_AGAIN | ST_GOON;
              }
            else if (ST_ERROR(res) == ST_OK)
              {
                self->transfer_state[0] = self->transfer_state_next[0];
                from->want_read = TRUE;
              }
            else
              {
                z_proxy_leave(self);
                return ST_ERR | ST_QUIT;
              }
            break;
          }
        case 4:
          return ST_QUIT | ST_OK;

        }
    }
  z_proxy_leave(self);
  return ST_OK | ST_GOON;
}

gboolean
http_data_transfer_setup(HttpProxy *self, guint from, guint to, gboolean expect_data)
{
  GString *headers = self->headers[from];
  ZPolicyObj *proxy;
  gboolean called;
  gboolean chunked; /* original entity is chunked */
  gboolean append_crlf = TRUE;
  gint i;
  
  z_proxy_enter(self);

  z_policy_lock(self->super.thread);
  /* FIXME: parameters
   *   request or response
   *   content-type? 
   *   ??
   */
  proxy = z_policy_call(self->super.handler, "requestDataProxy", NULL, &called, self->super.session_id);
  if (called && !proxy)
    {
      z_policy_unlock(self->super.thread);
      z_proxy_leave(self);
      return FALSE;
    }
  if (proxy)
    {
      self->stacked = z_proxy_stack_proxy(&self->super, proxy);
      z_policy_var_unref(proxy);
    }
  z_policy_unlock(self->super.thread);
  
  chunked = strcasecmp(self->transfer_encoding->str, "chunked") == 0;

  if (self->stacked)
    {
      if (chunked)
        {
          /* Original entity is chunked, we send it on, Transfer-Encoding is added */
          self->super.endpoints[from]->read = http_transfer_dechunk_client_to_down;
          self->stacked->downstreams[EP_CLIENT]->write = http_transfer_dechunk_client_to_down;
          self->stacked->downstreams[EP_SERVER]->read = http_transfer_enchunk_down_to_server;      
          self->super.endpoints[to]->write = http_transfer_enchunk_down_to_server;

          g_string_append(headers, "Transfer-Encoding: chunked\r\n");
         }
      else if (to == EP_CLIENT && expect_data)
        {
          if (self->proto_version[to] > 0x0100)
            {
              /* The original entity is not chunked, we enchunk it to avoid bufferring and 
               * keeping the connection open.
               * Transfer-Encoding is added, Content-Length is removed.
               */
              self->super.endpoints[from]->read = http_transfer_copy_client_to_down;
              self->stacked->downstreams[EP_CLIENT]->write = http_transfer_copy_client_to_down;
              self->stacked->downstreams[EP_SERVER]->read = http_transfer_enchunk_down_to_server;
              self->super.endpoints[to]->write = http_transfer_enchunk_down_to_server;

              g_string_append(headers, "Transfer-Encoding: chunked\r\n");
            }
          else
            {
              /* chunking is not supported, the entity's end is indicated by an EOF
               * neither Content-Length nor Transfer-Encoding is added
               */
              self->connection_mode = HTTP_CONNECTION_CLOSE;
              self->super.endpoints[from]->read = http_transfer_copy_client_to_down;
              self->stacked->downstreams[EP_CLIENT]->write = http_transfer_copy_client_to_down;
              self->stacked->downstreams[EP_SERVER]->read = http_transfer_copy_down_to_server;
              self->super.endpoints[to]->write = http_transfer_copy_down_to_server;
            }
        }
      else
        {
          /* we transfer to the server, which doesn't support chunking, thus 
           * we have to add a Content-Length header and remove Transfer-Encoding.
           * As we need the size of the transferred entity, we have to read the whole
           * thing into memory and send it afterwards.
           */
          
          if (self->content_length >= 0 || (expect_data && self->content_length == -1))
            {
              self->super.endpoints[from]->read = http_transfer_copy_client_to_down;
              self->stacked->downstreams[EP_CLIENT]->write = http_transfer_copy_client_to_down;
              self->stacked->downstreams[EP_SERVER]->read = http_transfer_buffer_down_to_server;
              self->super.endpoints[from]->write = http_transfer_buffer_down_to_server;
            }
          else
            {
              self->content_length = -2;
            }
          append_crlf = FALSE;
     
        }
      self->super.endpoints[from]->want_read = TRUE;
      self->super.endpoints[from]->want_write = FALSE;
      self->stacked->downstreams[EP_CLIENT]->want_read = FALSE;
      self->stacked->downstreams[EP_CLIENT]->want_write = FALSE;
      self->stacked->downstreams[EP_SERVER]->want_read = TRUE;
      self->stacked->downstreams[EP_SERVER]->want_write = FALSE;
      self->super.endpoints[to]->want_read = FALSE;
      self->super.endpoints[to]->want_write = FALSE;
      
      for (i = EP_CLIENT; i < EP_MAX; i++)
        {
          if (!self->transfer_buffer[i])
            {
              self->transfer_buffer[i] = g_new(HttpIOBuffer, 1);
              self->transfer_buffer[EP_CLIENT]->ofs = self->transfer_buffer[EP_CLIENT]->end = 0;
            }
        }
    }
  else
    {
      if (chunked)
        {
          self->super.endpoints[from]->read = http_transfer_copy_chunk_client_to_server;
          self->super.endpoints[to]->write = http_transfer_copy_chunk_client_to_server;
          g_string_append(headers, "Transfer-Encoding: chunked\r\n");
        }
      else if (self->content_length >= 0)
        {
          self->super.endpoints[from]->read = http_transfer_copy_client_to_server;
          self->super.endpoints[to]->write = http_transfer_copy_client_to_server;
          
          g_string_sprintfa(headers, "Content-Length: %d\r\n", self->content_length);
        }
      else if (expect_data && self->content_length == -1)
        {
          self->super.endpoints[from]->read = http_transfer_copy_client_to_server;
          self->super.endpoints[to]->write = http_transfer_copy_client_to_server;          
        }
      else 
        {
          self->content_length = -2;
        }
      
      self->super.endpoints[from]->want_read = TRUE;
      self->super.endpoints[from]->want_write = FALSE;
      self->super.endpoints[to]->want_read = FALSE;    
      self->super.endpoints[to]->want_write = FALSE;    
      if (!self->transfer_buffer[EP_CLIENT])
        {
          self->transfer_buffer[EP_CLIENT] = g_new(HttpIOBuffer, 1);
          self->transfer_buffer[EP_CLIENT]->ofs = self->transfer_buffer[EP_CLIENT]->end = 0;
        }
    }

  if (append_crlf)
    g_string_append_len(headers, "\r\n", 2);

  self->transfer_state[0] = 0;
  self->transfer_from = from;
  self->transfer_to = to;
  
#if 0
  self->filter_transfer = http_transfer_filter_copy;

  content_length = -2;  // -1 unknown, -2 no data

  if (strcasecmp(self->transfer_encoding->str, "chunked") == 0)
    {
      self->filter_transfer = http_transfer_filter_chunked;
      content_length = -1;
      g_string_append(headers, "Transfer-Encoding: chunked\r\n");
    }
  else if (self->content_length != -1)
    {
      self->filter_transfer = http_transfer_filter_copy;
    }
  else if (expect_data)
    {
      content_length = -1;
      self->connection_mode = HTTP_CONNECTION_CLOSE; 
    }
  else 
    {
      content_length = -2;
    }

  if (content_length >= 0)
    g_string_sprintfa(headers, "Content-Length: %d\r\n", content_length);
  
  self->content_length = content_length;
#endif

  z_proxy_leave(self);
  return TRUE;
}

gboolean 
http_data_transfer(HttpProxy *self)
{
  z_proxy_enter(self);
  
  if (self->content_length != -2)
    {
      int i;
      
      for (i = 0; i < EP_MAX; i++)
        {
          z_fd_set_nonblock(self->super.endpoints[i]->fd, 1);
          z_poll_add_stream(self->poll, self->super.endpoints[i]);
          self->super.endpoints[i]->user_data = self;
          if (self->stacked)
            {
              z_fd_set_nonblock(self->stacked->downstreams[i]->fd, 1);
              z_poll_add_stream(self->poll, self->stacked->downstreams[i]);
              self->stacked->downstreams[i]->user_data = self;
            }
        }
      
      while (z_poll_iter_timeout(self->poll, self->timeout))
        {
          ;
        }

      for (i = 0; i < EP_MAX; i++)
        {
          z_fd_set_nonblock(self->super.endpoints[i]->fd, 0);
          z_poll_remove_stream(self->poll, self->super.endpoints[i]);
          if (self->stacked)
            {
              z_fd_set_nonblock(self->stacked->downstreams[i]->fd, 0);
              z_poll_remove_stream(self->poll, self->stacked->downstreams[i]);
            }

        }
      if (self->entity_buffer)
        {
          g_free(self->entity_buffer);
          self->entity_len = 0;
        }
      if (self->stacked)
        {
          z_stacked_proxy_destroy(self->stacked);
          self->stacked = NULL;
        }

#if 0
      if (!self->filter_transfer(self, from, to, self->content_length))
	{
	  /* FIXME: transfer unsuccessful */
	  z_proxy_log(self, HTTP_ERROR, 2, "Data transfer failed;");
	  self->error_code = HTTP_MSG_OK; /* error message may corrupt already buffered data, close connection silently */
	  z_proxy_leave(self);
	  return FALSE;
	}
#endif
    }
  z_proxy_leave(self);
  return TRUE;
}

#endif
