/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: httpmisc.c,v 1.16 2002/10/25 15:44:20 bazsi Exp $
 *
 * Author: Balazs Scheidler <bazsi@balabit.hu>
 * Auditor: 
 * Last audited version: 
 * Notes:
 *   Based on the code by: Viktor Peter Kovacs <vps__@freemail.hu>
 *
 ***************************************************************************/

#include "http.h"

#include <zorp/log.h>

#include <ctype.h>
#include <zorp/log.h>

gboolean
http_split_url(gchar *url, 
	       gchar **proto, guint *proto_len,
	       gchar **user, guint *user_len,
	       gchar **passwd, guint *passwd_len,
	       gchar **host, guint *host_len,
	       guint *port,
	       gchar **file, guint *file_len)
{
  gchar *p, *end, *part[4], *sep[4];
  int i;

  z_enter();
  
  *proto = url;
  *user = NULL;
  *passwd = NULL;
  *host = NULL;
  *file = NULL;
  *file_len = 0;
  *port = 0;

  p = url;
  while (*p && *p != ':')
    p++;
    
  if (!*p)
    {
      z_leave();
      return FALSE;
    }

  if (*(p + 1) != '/' || *(p + 2) != '/')
    {
      /* protocol not terminated by '//' */
      z_leave();
      return FALSE;
    }
  p += 3;
  *proto_len = p - url;

  for (i = 0; i < 4; i++)
    {
      part[i] = p;
      while (*p && *p != ':' && *p != '/' && *p != '@')
        p++;
      sep[i] = p;
      if (!*p || *p == '/')
        break;
      p++;
    }
  switch (i)
    {
    case 0:
      /* hostname only */
      *host = part[0];
      *host_len = sep[0] - part[0];
      break;
    case 1:
      /* username && host || hostname && port number */
      if (*sep[0] == ':')
        {
          *host = part[0];
          *host_len = sep[0] - part[0];
          /* port number */
          *port = strtoul(part[1], &end, 10);
          if (end != sep[1])
            {
              z_leave();
              return FALSE;
            }
        }
      else if (*sep[0] == '@')
        {
          /* username */
          *user = part[0];
          *user_len = sep[0] - part[0];
          *host = part[1];
          *host_len = sep[1] - part[1];
        }
      else
        {
          z_leave();
          return FALSE;
        }
      break;
    case 2:
      /* username && host && port || username && password && host */
      if (*sep[0] == '@' && *sep[1] == ':')
        {
          /* username, host, port */
          *user = part[0];
          *user_len = sep[0] - part[0];
          *host = part[1];
          *host_len = sep[1] - part[1];
          *port = strtoul(part[2], &end, 10);
          if (end != sep[2])
            {
              z_leave();
              return FALSE;
            }
        }
      else if (*sep[0] == ':' && *sep[1] == '@')
        {
          /* username, password, host */
          *user = part[0];
          *user_len = sep[0] - part[0];
          *passwd = part[1];
          *passwd_len = sep[1] - part[1];
          *host = part[2];
          *host_len = sep[2] - part[2];
        }
      else
        {
          z_leave();
          return FALSE;
        }
      break;
    case 3:
      /* username && password && hostname && port */
      if (*sep[0] == ':' && *sep[1] == '@' && *sep[2] == ':')
        {
          *user = part[0];
          *user_len = sep[0] - part[0];
          *passwd = part[1];
          *passwd_len = sep[1] - part[1];
          *host = part[2];
          *host_len = sep[2] - part[2];
          *port = strtoul(part[3], &end, 10);
          if (end != sep[3])
            {
              z_leave();
              return FALSE;
            }
        }
      else
        {
          z_leave();
          return FALSE;
        }
      break;
    default:
      z_leave();
      return FALSE;
    }

  *file = p;
  *file_len = strlen(*file);

  z_leave();
  return TRUE;
}

gint 
xdigit(char c)
{
  c = tolower(c);
  if (c >= '0' && c <= '9')
    return c - '0';
  else if (c >= 'a' && c <= 'f')
    return c - 'a';
  return -1;
}

gboolean
http_check_url(gchar *url, gboolean permit_unicode)
{
  guint length = strlen(url);
  guint t = 0;
  
  gchar *proto, *user, *passwd, *host, *file;
  guint proto_len, user_len, passwd_len, host_len, file_len, port;

  if (!http_split_url(url, 
                      &proto, &proto_len, 
                      &user, &user_len, 
                      &passwd, &passwd_len, 
                      &host, &host_len, 
                      &port, 
                      &file, &file_len))
    {
      return FALSE;
    }


  while (t < length)
    {
      switch (url[t])
        {
        case '%':
          if ((t + 1) >= length)
            return FALSE;
          
          if (url[t + 1] == '%')
            {
              t++;
              break;
            }
          else if (url[t + 1] == 'u' && permit_unicode)
            {
              if ((t + 5) >= length)
                return FALSE;
              if (!isxdigit(url[t + 2]) || !isxdigit(url[t + 3]) || !isxdigit(url[t + 4]) || !isxdigit(url[t + 5]))
                return FALSE;
              t += 5;
            }
          else
            {
              if ((t + 2) >= length)
                return FALSE;
          
              if (!isxdigit(url[t + 1]) || !isxdigit(url[t + 2]))
                return FALSE;
              t += 2;
            }
          break;
        default:
          break;
        }
      t++;
    }
  return TRUE;
}

#if 0
gboolean
http_check_url(GString *url)
{
  guint length = url->len;
  guchar buf[url->len + 1];
  gint src, dst;

  src = 0;
  dst = 0;
  while (src < length)
    {
      switch (url->str[src])
        {
        case '%':
          if ((src + 2) >= length)
            return FALSE;
            
          if (!isxdigit(url->str[src + 1]) || !isxdigit(url->str[src + 2]))
            return FALSE;
          buf[dst] = xdigit(url->str[src + 1]) << 4 | xdigit(url->str[src + 2]);
          dst++;
          src += 2;
          break;
        case '+':
          buf[dst] = ' ';
          dst++;
        default:
          buf[dst++] = url->str[src];
          break;
        }
      src++;
    }
    
  length = dst;
  /* we have a converted URL in buf now, reencode it */
  src = 0;
  g_string_truncate(url, 0);
  while (src < length)
    {
      if (buf[src] == ' ')
        g_string_append_c(url, '+');
      else if (buf[src] < 32 || buf[src] >= 128)
        g_string_sprintfa(url, "%02x", buf[src]);
      else
        g_string_append_c(url, buf[src]);
      src++;
    }
  
  return TRUE;
}
#endif

#define SKIP_SPACES \
  do \
    { \
      while (left > 0 && *src == ' ') \
        { \
          src++; \
          left--; \
        } \
    } \
  while (0)
  
#define COPY_SPACE \
  do \
    { \
      while (left > 0 && avail > 0 && *src != ' ' && *src) \
        { \
          *dst++ = *src++; \
          left--; \
          avail--; \
        } \
      *dst = 0; \
    } \
  while (0)
	

gboolean
http_split_request(HttpProxy *self, gchar *line, guint length)
{
  gchar *src, *dst;
  gint left, avail;

  z_proxy_enter(self);
  self->request_method[0] = 0;
  self->request_flags = -1;
  self->request_version[0] = 0;

  g_string_truncate(self->request_url, 0);

  src = line;
  left = length;
  dst = self->request_method;
  avail = sizeof(self->request_method) - 1;

  COPY_SPACE;

  if (!self->request_method[0] || (*src != ' ' && avail == 0))
    {
      z_proxy_log(self, HTTP_VIOLATION, 1, "Request method empty, or too long; line='%.*s'", length, src);
      z_proxy_leave(self);
      /* request method empty, or request buffer overflow */
      return FALSE;
    }

  SKIP_SPACES;

  avail = self->max_url_length;
  g_string_truncate(self->request_url, 0);

  while (left > 0 && avail > 0 && *src != ' ' && *src) 
    { 
      g_string_append_c(self->request_url, *src++);
      left--; 
      avail--; 
    } 

  if (!self->request_url->str[0] || (*src != ' ' && avail == 0))
    {
      /* url missing, or too long */
      z_proxy_log(self, HTTP_VIOLATION, 1, "URL missing, or too long; line='%.*s'", length, src);
      z_proxy_leave(self);
      return FALSE;
    }

  SKIP_SPACES;

  dst = self->request_version;
  avail = sizeof(self->request_version) - 1;
  
  COPY_SPACE;

  if (*src != ' ' && avail == 0)
    {
      /* protocol version too long */
      z_proxy_log(self, HTTP_VIOLATION, 1, "Protocol version missing, or too long; line='%.*s'", length, src);
      z_proxy_leave(self);
      return FALSE;
    }

  z_proxy_log(self, HTTP_REQUEST, 6, "Request details; command='%s', url='%s', version='%s'", self->request_method, self->request_url->str, self->request_version);
  z_proxy_leave(self);
  return TRUE;
}

gboolean
http_split_response(HttpProxy *self, gchar *line, guint line_length, gboolean *reply09)
{
  gchar *src, *dst;
  gint left, avail;

  z_proxy_enter(self);
  self->response_version[0] = 0;
  self->response[0] = 0;
  self->response_msg[0] = 0;
  *reply09 = FALSE;
  
  src = line;
  left = line_length;
  
  dst = self->response_version;
  avail = sizeof(self->response_version) - 1;

  COPY_SPACE;
  
  if (memcmp(self->response_version, "HTTP", 4) != 0)
    {
      /* no status line */
      *reply09 = TRUE;
      z_proxy_log(self, HTTP_RESPONSE, 6, "No HTTP status line. Treating as version 0.9 response; line='%s'", dst);
      z_proxy_leave(self);
      return FALSE;
    }

  if (!self->response_version[0] || (*src != ' ' && avail == 0))
    {
      /* response version empty or too long */
      z_proxy_log(self, HTTP_VIOLATION, 1, "Response version empty or too long; line='%.*s'", line_length, line);
      z_proxy_leave(self);
      return FALSE;
    }
  
  SKIP_SPACES;
  
  dst = self->response;
  avail = sizeof(self->response) - 1;

  COPY_SPACE;
  
  if (!self->response[0] || (*src != ' ' && left && avail == 0))
    {
      /* response code empty or too long */
      z_proxy_log(self, HTTP_VIOLATION, 1, "Response code empty or too long; line='%.*s'", line_length, line);
      z_proxy_leave(self);
      return FALSE;
    }
  
  self->response_code = atoi(self->response);
    
  SKIP_SPACES;

  dst = self->response_msg;
  avail = sizeof(self->response_msg) - 1;

  while (left > 0 && avail > 0) 
    { 
      *dst++ = *src++; 
      left--; 
      avail--; 
    } 
  *dst = 0; 
  
  z_proxy_log(self, HTTP_RESPONSE, 7, "Response details; version='%s', response_code='%s', response_msg='%s'", self->response_version, self->response, self->response_msg);
  z_proxy_leave(self);
  return TRUE;
}

#undef SKIP_SPACES
#undef COPY_SPACE

gboolean
http_parse_version(HttpProxy *self, gint side, gchar *version_str)
{
  z_proxy_enter(self);
  if (strcasecmp(version_str, "HTTP/1.1") == 0)
    {
      self->proto_version[side] = 0x0101;
    }
  else if (strcasecmp(version_str, "HTTP/1.0") == 0)
    {
      self->proto_version[side] = 0x0100;
    }
  else if (version_str[0] == 0)
    {
      self->proto_version[side] = 0x0009;
    }
  else
    {
      /* unknown protocol version */
      if (side == EP_CLIENT)
        z_proxy_log(self, HTTP_REQUEST, 3, "Unknown protocol version; version='%s'", version_str);
      else
        z_proxy_log(self, HTTP_RESPONSE, 3, "Unknown protocol version; version='%s'", version_str);
      self->proto_version[side] = 0x0100;
      z_proxy_leave(self);
      return FALSE;
    }
  z_proxy_leave(self);
  return TRUE;
}
