############################################################################
##
## Copyright (c) 2000, 2001, 2002 BalaBit IT Ltd, Budapest, Hungary
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##
##
## $Id: Auth.py,v 1.34.2.5 2003/04/26 09:55:55 sasa Exp $
##
## Author  : Bazsi
## Auditor : kisza
## Last audited version: 1.3
## Notes:
##
############################################################################

import Zorp
from Zorp import *

Z_AUTH_UNKNOWN    = 0
Z_AUTH_GETMETHODS = 1
Z_AUTH_METHODS    = 2
Z_AUTH_SETMETHOD  = 3
Z_AUTH_REQUEST    = 4
Z_AUTH_RESPONSE   = 5
Z_AUTH_ACCEPT     = 6
Z_AUTH_ABORT      = 7
Z_AUTH_REJECT     = 8

class AuthCache:
	from time import time
	from thread import allocate_lock
	
	def __init__(self, timeout=600, update_stamp=TRUE, service_equiv=FALSE):
		"""
		  timeout         -- [INTEGER]
		  update_stamp    -- [BOOLEAN]
		  service_equiv   -- [BOOLEAN]
		"""

		self.timeout = timeout
		self.update_stamp = update_stamp
		self.service_equiv = service_equiv
		self.auth_cache = {}
		self.lookups = 0
		self.lock = self.allocate_lock()
	
	def makeKey(self, session):
		if self.service_equiv:
			return session.client_address.ip_s
		else:
			return (session.client_address.ip_s, session.service.name)

	def cleanup(self):
		now = self.time()
		for x in self.auth_cache.keys():
			if now - self.auth_cache[x][0] > self.timeout:
				del self.auth_cache[x]
	
	def lookup(self, session):
		self.lock.acquire()
		
		self.lookups = self.lookups + 1
		if self.lookups % 1000:
			self.cleanup()
		key = self.makeKey(session)
		if self.auth_cache.has_key(key):
			entry = self.auth_cache[key]
			if self.time() - entry[0] > self.timeout:
				del self.auth_cache[key]
			else:
				if self.update_stamp:
					entry[0] = self.time()
				self.lock.release()
				return entry[1]
		self.lock.release()
		return None
	
	def store(self, session, entity):
		self.lock.acquire()
		key = self.makeKey(session)
		self.auth_cache[key] = [self.time(), entity]
		self.lock.release()
		

class AbstractAuthenticationProvider:
	"""Class encapsulating an authentication Provider.
	
	This is an abstract class to encapsulate an authentication Provider,
	which is responsible for checking authentication credentials.
	
	You shouldn't need to instantiate this class, use a derived one
	instead.
	
	Attributes
	
	  methods  -- set of allowed methods set by the administrator
	"""
	def __init__(self, name):
		"""Constructor to initialize an 'AbstractAuthorization' instance.
		
		This constructor initializes an 'AbstractAuthorization' instance
		by storing its methods argument as an attribute.
		
		Arguments
		
		  self -- this instance
		
		  name -- [QSTRING]
		"""

		if Globals.authentication_pv.has_key(name):
			raise AuthException, "Duplicate authorization db"
		Globals.authentication_pv[name] = self
	
	def startSession(self, session_id, session):
		"""Method to be called when an authentication session starts.
		
		This method is called when an authentication session
		identified by 'session_id' starts. 'session_id' can be used
		to associate data with this session, as each subsequent
		calls to AbstractAuthorization methods will get this value.
		
		Arguments
		
		  self -- this instance
		  
		  session_id -- session identifier represented as a string
		"""
		pass
	
	def stopSession(self, session_id):
		"""Method to be called when an authentication session ends.
		
		This method is called when an authentication session is ended.
		It's a placeholder for freeing up any resources associated to
		a given session.
		"""
		pass
	
	def getMethods(self, session_id, entity):
		"""Function returning the allowed set of methods.
		
		This function calculates and returns the set of allowed methods
		a user is allowed to authenticate with. We return an empty
		set here, overridden methods should return something more
		interesting.
		
		Arguments
		
		  self -- this instance
		  
		  session_id -- authentication session id
		  
		  entity -- username
		  
		Returns
		
		  return a tuple. First value is Z_AUTH_*, the second is a
		  array of applicable methods. (if any)
		"""
		return ()
	
	def setMethod(self, session_id, method):
		"""Function to set the authentication Method.
		
		This function should return a challenge for a given entity
		using the given method, or None if challenge is not
		applicable for the given method.
		
		Arguments
		
		  self -- this instance
		  
		  session_id -- authentication session id
		  
		  entity -- username
		  
		  method -- authentication method
		  
		Returns
		
		  return a tuple. First value is one of Z_AUTH*, second
		  value is a string containing the challenge, or None if not applicable
		"""
		raise NotImplementedError
	
	def converse(self, session_id, credentials):
		"""Function checking the presented credentials of an entity.
		
		This function is called to check the credentials presented
		by the client for validity. It should return either TRUE, if
		the credentials for the given challenge & method & username
		are valid.
		
		Arguments
		
		  self -- this instance
		  
		  session_id -- authentication session identifier
		  
		  entity -- username
		  
		  challenge -- a previously issued challenge (might be None or an empty string)
		  
		  credentials -- response for the given challenge

		Returns

		  return a tuple. First value is one of Z_AUTH_*, second is
		  depending on the first.
		"""
		raise NotImplementedError

class AbstractAuthentication:
	"""Abstract class encapsulating interface different authentication means.
	
	This class encapsulates interfaces for inband and outband
	authentication procedures. You should not use this class directly,
	use 'InbandAuthentication' or 'SatyrAuthentication'.
	
	Attributes
	
	  authentication_provider -- authentication provider to authenticate against
	"""
	def __init__(self, authentication_provider, auth_cache = None):
		"""Constructor to initialize an AbstractAuthentication instance.
		
		This constructor initializes an 'AbstractAuthentication' instance
		by storing arguments as attributes.
		
		Arguments
		
		  self -- this instance
		  
		  authentication_provider -- [CLASS_authprov] The authentication
		                             provider class

		  auth_cache              -- [INST_authcache] The
					     authentication Cache Object
		"""
		try:
			self.authentication_provider = Globals.authentication_pv[authentication_provider]
			self.cache = auth_cache
		except KeyError:
			raise AuthException, 'Authorization db not found.'

	def performAuth(self, session):
		raise AuthException, 'Authentication not implemented'
		
		
	def performOutbandAuth(self, session):
		"""Function called to perform outband authentication before the session is started.
		
		This function is called to perform outband authentication
		before a session is started. It should raise AuthException
		if the authentication was not successful.
		
		Arguments
		
		  self -- this instance
		
		  session -- the session object which is to be started
		"""
		raise AuthException, 'Outband authentication not implemented'
		
	def performInbandAuth(self, proxy):
		"""Function called to enable inband authentication in a session.
		
		This function is called when a proxy was started, and it should
		enable inband authentication in the proxy, given it has support
		to perform it.
		
		Arguments
		
		  self -- this instance
		  
		  proxy -- proxy instance to enable authentication in
		"""
		raise AuthException, 'Inband authentication not implemented'


class InbandAuthentication(AbstractAuthentication):
	"""Class encapsulating inband authentication.
	
	This class encapsulates inband authentication. Instances
	of this class can be used as the 'auth' parameter for 
	Service definitions.
	
	Example
	
	  r = TISAuthorization(SockAddrInet('192.168.1.1', 7777))
	  Service("intra_http", HttpProxy,
	          auth=InbandAuthentication(r))
	
	"""

	def performAuth(self, proxy):
		return self.performInbandAuth(proxy)

	def performInbandAuth(self, proxy):
		if hasattr(proxy, 'auth_inband_supported') and proxy.auth_inband_supported:
			proxy.auth = self.authentication_provider
			self.authentication_provider.startSession(proxy.session.session_id, proxy.session)
		else:
			raise AuthException, 'Inband authentication not supported by the underlying proxy'

