############################################################################
##
## Copyright (c) 2000, 2001, 2002 BalaBit IT Ltd, Budapest, Hungary
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##
##
## $Id: Proxy.py,v 1.65.2.10 2004/03/11 16:31:28 bazsi Exp $
##
## Author  : Bazsi
## Auditor : kisza
## Last audited version: 1.13
## Notes:
##
############################################################################

"""Module defining classes encapsulating native proxies.

This module defines the class 'Proxy', encapsulating the ZorpProxy component
implemented by the Zorp core. Each protocol specific proxy is derived from
this class.
"""

from Zorp import *
from Stream import Stream
from SockAddr import SockAddrInet, getHostByName
from Session import StackedSession

from traceback import print_exc
import string

def proxyLog(self, type, level, msg):
	"""Function to log a proxy message.

	This function logs a message with the 'session_id' prepended.

	Arguments

	  self  -- should be a Proxy derivative

	  type  -- log class

	  level -- message verbosity level

	  msg   -- msg to log
	  
	"""
 	log(self.session.session_id, type, level, msg)

class Proxy(ZorpProxy):
	"""Base class for the proxies.

	This class is inherited from an ExtensionClass implemented by the
	Zorp core. It encapsulates a protocol specific proxy. As an instance
	is created, the required module is loaded (identified by the
	name attribute), and a new thread is started to handle the connection.
	
	Attributes
	
	  session      -- descriptor of the connection
	  
	  name         -- the native proxy module used for this session
	"""
	tracker_name = None
	name = None

	def __init__(self, session):
		"""Constructor to initialize a Proxy instance.

		This constructor initializes a Proxy instance, sets
		attributes and calls the underlying C extensionclass
		to actually create a proxy instance.
		
		Arguments
		
		  name	  -- The type of the proxy
		  
		  session -- reference to the session
		"""
		self.session = session
		self.server_fd_picked = FALSE
		self.proxy_started = FALSE
		session.setProxy(self.name)
		log(session.session_id, CORE_SESSION, 5, "Proxy starting; class='%s', module='%s'" % (self.__class__.__name__, self.name))
		ZorpProxy.__init__(self, self.name, session.session_id, session.client_stream)
		self.proxy_started = TRUE

	def __del__(self):
		"""Destructor to deinitialize a Proxy instance.
		
		This destructor is called when this object instance is
		freed. It simply sends a message about this event to the
		log.
		
		Arguments
		
		  self -- this instance
		"""

		log(self.session.session_id, CORE_SESSION, 5, "Proxy ending; class='%s', module='%s'" % (self.__class__.__name__, self.name))

	def __startup__(self):
		service = self.session.service
		try:
	                service.router.setupFastpath(self)
		        service.chainer.setupFastpath(self)
	        	if service.snat:
		                service.snat.setupFastpath(self)
	   	        if service.dnat:
		                service.dnat.setupFastpath(self)
		except:
			# setting up the fastpath was not successful
			# FIXME: remove fastpath state completely
			pass


	def __config__(self):
		"""Function called by the proxy core to perform internal proxy initialization.

		This function is similar to config() to perform initialization
		of internal proxy related data. It is not meant as a user
		interface, currently it is used to perform outband authentication.

		Arguments

		  self -- this instance
		"""
		if not self.session.auth_user and self.session.auth:
			self.session.auth.performAuth(self)

	def config(self):
		"""Function called by the proxy core to perform proxy initialization.

		This function is called when the proxy module is started and
		its configuration is to be initialized.

		Arguments

		  self -- this instance

		"""
		pass

	def __destroy__(self):
		"""Function called by the proxy core when the session is to be freed.
		
		This function is called when the proxy module is to be freed. It
		simply sends a message about this event to the log.
		
		Arguments
		
		  self -- this instance
		"""
		# NOTE: if C proxy was started but the chaining process was
		# not completed then the server side of the connection is
		# still hanging there unpicked. Close it.

		if self.proxy_started and self.session.server_stream and not self.server_fd_picked:
			self.session.server_stream.close()

		log(self.session.session_id, CORE_DEBUG, 6, "Proxy destroy; class='%s', module='%s'" % (self.__class__.__name__, self.name))

        def stackProxy(self, client_stream, server_stream, proxy_class):
                """Function to actually stack a proxy within this proxy in a subsession.

                This function is called by the underlying C proxy to
                actually do stacking, it handles over the downstream
		filedescriptors and the class to create an instance of.
		The way the underlying proxy decides which proxy_class
		to use is proxy specific.

                Arguments

                  self -- this instance

                  client_stream -- client stream

                  server_stream -- server stream
                  
                  proxy_class -- proxy class to instantiate
                """

                proxyLog(self, CORE_DEBUG, 7, "Stacking child proxy; client_fd='%d', server_fd='%d', class='%s'" % (client_stream.fd, server_stream.fd, proxy_class.__name__))
		
                subsession = StackedSession(self.session)
		session_id = string.split(self.session.session_id, '/')
		if len(session_id):
			session_id[len(session_id)-1] = proxy_class.name
			session_id = string.join(session_id, '/')
		else:
			# hmm, funny session_id ...
			session_id = self.session.session_id
                subsession.client_stream = client_stream
                subsession.client_stream.name = "%s/client_upstream" % (session_id)
                subsession.server_stream = server_stream
		subsession.server_stream.name = "%s/server_upstream" % (session_id)
		try:
	                return proxy_class(subsession)
		except:
			subsession.destroy()
	
		return None

	def setServerAddress(self, host, port):
		# resolve host, port and store it in session.server_address
		# may raise an exception
		if self.session.server_address_inband:
			try:
				ip = getHostByName(host)
				self.session.server_address = SockAddrInet(ip, port)
			except IOError:
				proxyLog(self, CORE_ERROR, 3, "Error resolving hostname; host='%s'" % host)
				return FALSE
		return TRUE

	def connectServer(self):
		"""Function called to connect to the server endpoint or to the parent proxy.

		This function is called to establish connection with our
		server endpoint, which can be either a parent proxy we are
		stacked in, or a real connection to the server.

		If the chainer is defined, it is called with the given host
		and port parameters to connect to the remote server.

		If there is no chainer defined, it tries to return the
		server-side stream from the session. If there's no
		server_stream it raises an error.

		Arguments

		  self          -- this instance
		  
		Returns
		
		  The descriptor of the server stream

		"""
		try:
			self.server_fd_picked = TRUE
			if self.session.chainer == None:
			
				# we have no chainer, the server side fd
				# should be available by now, used in stacked
				# proxies

				if self.session.server_stream == None:
					raise InternalException, "No chainer and server_stream is None"
			else:
				self.session.server_stream = None
				self.session.chainer.chainParent(self.session)
		except ZoneException, s:
 			log(self.session.session_id, CORE_POLICY, 1, "Zone not found; zone='%s'" % (s))
		except DACException:
 			log(self.session.session_id, CORE_POLICY, 1, "DAC policy violation;")
		except MACException:
 			log(self.session.session_id, CORE_POLICY, 1, "MAC policy violation;")
		except AuthException:
 			log(self.session.session_id, CORE_POLICY, 1, "Authentication failure;")
		except LimitException:
 			log(self.session.session_id, CORE_POLICY, 1, "Connection over permitted limits;")
		except LicenseException:
			log(self.session.session_id, CORE_POLICY, 1, "Attempt to use an unlicensed component, or number of licensed hosts exceeded;")
		except:
			print_exc()

		return self.session.server_stream
	
	def userAuthenticated(self, entity, auth_info=''):
		"""Function called when inband authentication succeeds.

		This function is called by the low-level proxy module to
		indicate that inband authentication was successfully
		performed. The name of the client entity can be found in
		the 'entity' parameter.

		Arguments

		  self -- this instance

		  entity -- textual representation of the entity authenticated

		"""
		self.session.auth_user = entity
		proxyLog(self, CORE_AUTH, 3, "User authentication successful; entity='%s', auth_info='%s'" % (entity, auth_info))

