/***************************************************************************
 *
 * Copyright (c) 2000, 2001 BalaBit IT Ltd, Budapest, Hungary
 * All rights reserved.
 *
 * $Id: main.c,v 1.99.2.19 2004/02/03 09:16:28 sasa Exp $
 *
 * Author  : bazsi
 * Auditor : kisza
 * Last version : 1.24
 * Notes   : 
 *
 ***************************************************************************/

#include <zorp/policy.h>
#include <zorp/zorp.h>
#include <zorp/io.h>
#include <zorp/stream.h>
#include <zorp/registry.h>
#include <zorp/thread.h>
#include <zorp/log.h>
#include <zorp/proxyvars.h>
#include <zorp/freeq.h>
#include <zorp/cap.h>
#include <zorp/ssl.h>
#include <zorp/sysdep.h>
#include <zorp/poll.h>
#include <zorp/szig.h>
#include <zorp/tpsocket.h>
#include <zorp/dispatch.h>

#include <zorp/conntrack.h>
#include <zorp/stackdump.h>

#include <pwd.h>
#include <grp.h>
#include <signal.h>

#include <sys/resource.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef USE_DMALLOC
#include <dmalloc.h>
#endif

#if HAVE_SYS_PRCTL_H
#include <sys/prctl.h>
#endif

int usr1_received = 0;
int usr2_received = 0;
int hup_received = 0;
int term_received = 0;

void
z_sigterm_handler(int signo G_GNUC_UNUSED)
{
  if (main_loop)
    term_received = 1;
  else
    exit(1);
}

void 
z_ignore_signal_handler(int signo G_GNUC_UNUSED)
{
}

void
z_fatal_signal_handler(int signo)
{
  ZSignalContext *p = z_stackdump_get_context(p);
  struct sigaction act;

  memset(&act, 0, sizeof(act));
  act.sa_handler = SIG_DFL;
  sigaction(signo, &act, NULL);

  /*LOG
    This message is logged when Zorp caught a fatal signal.
    Possible reason is bad RAM or other hardware.
   */
  z_log(NULL, CORE_ERROR, 0, "Signal (%d) received, stackdump follows;", signo);
  z_mem_trace_stats();
  z_stackdump_log(p);
  kill(getpid(), signo);
}

void
z_sigusr1_handler(int signo G_GNUC_UNUSED)
{
  usr1_received = 1;
}

void
z_sigusr2_handler(int signo G_GNUC_UNUSED)
{
  usr2_received = 1;
}

void
z_write_pid_file(const gchar *fname)
{
  FILE *fd;
  
  fd = fopen(fname, "w");
  if (fd) 
    {
      fprintf(fd, "%d\n", getpid());
      fclose(fd);
    }
  else
    {
      fprintf(stderr, "Error opening pidfile; fname='%s', error='%m'\n", fname);
    }
}

void
z_remove_pid_file(const gchar *fname)
{
  if (unlink(fname) == -1)
    z_log(NULL, CORE_ERROR, 3, "Error removing pidfile; fname='%s', error='%m'", fname);
}

void
z_setup_signals(void)
{
  struct sigaction act;
  
  memset(&act, 0, sizeof(act));
  act.sa_handler = z_ignore_signal_handler;
  sigaction(SIGPIPE, &act, NULL);
  sigaction(SIGINT, &act, NULL);
  sigaction(SIGTERM, &act, NULL); 

  memset(&act, 0, sizeof(act));
  act.sa_handler = z_sigterm_handler;
  sigaction(SIGINT, &act, NULL);
  sigaction(SIGTERM, &act, NULL); 
  
  memset(&act, 0, sizeof(act));
  act.sa_handler = z_fatal_signal_handler;
  sigaction(SIGSEGV, &act, NULL);
  sigaction(SIGABRT, &act, NULL);
  sigaction(SIGILL, &act, NULL);
  
  memset(&act, 0, sizeof(act));
  act.sa_handler = z_ignore_signal_handler;
  sigaction(SIGTRAP, &act, NULL);
  sigaction(SIGHUP, &act, NULL);
  
  memset(&act, 0, sizeof(act));
  act.sa_handler = z_sigusr1_handler;
  sigaction(SIGUSR1, &act, NULL);
  
  memset(&act, 0, sizeof(act));
  act.sa_handler = z_sigusr2_handler;
  sigaction(SIGUSR2, &act, NULL);
}

gboolean 
z_resolve_user(const gchar *user, uid_t *uid)
{
  struct passwd *pw;
  gchar *errstr;

  *uid = 0;
  pw = getpwnam(user);
  if (pw) 
    {
      *uid = pw->pw_uid;
    }
  else 
    {
      *uid = strtol(user, &errstr, 10);
      if (!errstr || *errstr)
        {
          return FALSE;
        }
    }
  return TRUE;
}

int 
z_resolve_group(const gchar *group, gid_t *gid)
{
  struct group *gr;
  gchar *errstr;

  *gid = 0;
  gr = getgrnam(group);
  if (gr) 
    {
      *gid = gr->gr_gid;
    }
  else 
    {
      *gid = strtol(group, &errstr, 10);
      if (!errstr || *errstr)
        {
          return FALSE;
        }
    }
  return TRUE;
}

gboolean
z_load_policy(const gchar *policy_file, const gchar *instance_name)
{
  ZPolicy *policy;

  policy = z_policy_new(policy_file);
  if (!z_policy_boot(policy) || !z_policy_load(policy))
    {
      z_log(NULL, CORE_ERROR, 0, "Error booting & parsing policy;");
      return FALSE;
    }
  current_policy = policy;
  if (!z_policy_init(policy, instance_name))
    {
      z_log(NULL, CORE_ERROR, 0, "Error initializing policy;");
      return FALSE;
    }
  z_log_set_fake_session_id(instance_name);
  return TRUE;
}

void
z_main_loop(const gchar *policy_file, const gchar *instance_name)
{
  if (!z_load_policy(policy_file, instance_name))
    {
      /* hack to let our messages get out */
      sleep(3);
      z_log(NULL, CORE_ERROR, 0, "Error loading initial policy, exiting;");
      return;
    }
  main_loop = g_main_new(TRUE);
  while (g_main_is_running(main_loop))
    {
      g_main_iteration(TRUE);
      if (usr1_received)
        {
          usr1_received = 0;
          verbose_level++;
          z_log_clear_hash();
          /*NOLOG*/
          z_log(NULL, CORE_DEBUG, 0, "SIGUSR1 received; verbose_level='%d'", verbose_level);
          z_mem_trace_stats();
        }
      if (usr2_received)
        {
          usr2_received = 0;
          verbose_level--;
          z_log_clear_hash();
          /*NOLOG*/
          z_log(NULL, CORE_DEBUG, 1, "SIGUSR2 received; verbose_level='%d'", verbose_level);
        }
      if (term_received)
        {
          z_main_quit(0);
          break;
        }
      z_free_queue_flush();
    }
  z_mem_trace_dump();
  g_main_destroy(main_loop);

}

void
z_usage(void)
{
  puts("zorp [options] - start a zorp instance\n\n"
       "--as <name>         -a      use <name> as the name of this instance\n"
       "--policy <file>     -p      use <file> as policy\n"
       "--verbose [num]     -v      set verbosity to num\n"
       "--no-syslog         -l      log to console\n"
       "--log-spec <spec>   -s      set log specification to spec\n"
       "--log-tags          -T      log message tags in messages\n"
       "--pidfile <file>    -P      write pid to file\n"
       "--version           -V      query zorp version\n"
       "--threads <num>     -t      specify the maximum number of simultaneous threads\n"
       "--idle-threads <num>-I      specify the maximum number of simultaneous idle threads\n"
       "--uid <uid>         -u      uid to run as\n"
       "--gid <gid>         -g      gid to run as\n"
#if ZORPLIB_ENABLE_CAPS
       "--caps <caps>       -C      capabilities to set\n"
       "--no-caps           -N      don't attempt to set capabilities\n"
#endif
#if ENABLE_NETFILTER_TPROXY
       "--autobind-ip       -B      autobind address for transparent proxying\n"
#endif
       );
}

#define ON_OFF_STR(x) (x ? "on" : "off")

void
z_version(void)
{
  printf("Zorp %s\n"
         "Compile-Date: %s %s\n"
         "Config-Date: %s\n"
         "Trace: %s\n"
         "Debug: %s\n"
         "IPOptions: %s\n"
         "IPFilter-Tproxy: %s\n"
         "Netfilter-Tproxy: %s\n"
         "Netfilter-Linux22-Fallback: %s\n"
         "Linux22-Tproxy: %s\n"
         "Conntrack: %s\n\n"
         "%s",
         VERSION, __DATE__, __TIME__,
         ZORP_CONFIG_DATE,
         ON_OFF_STR(ZORPLIB_ENABLE_TRACE),
         ON_OFF_STR(ENABLE_DEBUG),
         ON_OFF_STR(ENABLE_IPOPTIONS),
         ON_OFF_STR(ENABLE_IPFILTER_TPROXY),
         ON_OFF_STR(ENABLE_NETFILTER_TPROXY),
         ON_OFF_STR(ENABLE_NETFILTER_LINUX22_FALLBACK),
         ON_OFF_STR(ENABLE_LINUX22_TPROXY),
         ON_OFF_STR(ENABLE_CONNTRACK),
         z_zorplib_version_info());
}

#if 0
void
z_enable_core(void)
{
  struct rlimit limit;

  limit.rlim_cur = RLIM_INFINITY;
  limit.rlim_max = RLIM_INFINITY;

  chdir("/var/tmp");
  setrlimit(RLIMIT_CORE, &limit);
}
#endif

#ifdef HAVE_GETOPT_LONG
static struct option zorp_options[] = 
{
  { "as", required_argument, NULL, 'a' },
  { "policy", required_argument, NULL, 'p' },
  { "verbose", required_argument, NULL, 'v' },
  { "debug", no_argument, NULL, 'd' },
  { "pidfile", required_argument, NULL, 'P' },
  { "no-syslog", no_argument, NULL, 'l' },
  { "help", no_argument, NULL, 'h' },
  { "version", no_argument, NULL, 'V' },
  { "log-spec", required_argument, NULL, 's' },
  { "logspec", required_argument, NULL, 's' },
  { "log-tags", no_argument, NULL, 'T' },
  { "threadpools", no_argument, NULL, 'O' },
  { "threads", required_argument, NULL, 't' },
  { "idle-threads", required_argument, NULL, 'I' },
  { "uid", required_argument, NULL, 'u' },
  { "gid", required_argument, NULL, 'g' },
  { "chroot", required_argument, NULL, 'R' },
#if ENABLE_NETFILTER_TPROXY
  { "autobind-ip", required_argument, NULL, 'B' },
#endif
#if ZORPLIB_ENABLE_CAPS
  { "caps", required_argument, NULL, 'C' },
  { "no-caps", no_argument, NULL, 'N' },
#endif
  { NULL, 0, NULL, 0 }
};
#endif

#if ZORPLIB_ENABLE_CAPS
#define OPT_CAPS "C:N"
#else
#define OPT_CAPS 
#endif

#if ENABLE_NETFILTER_TPROXY
#define OPT_NF_TPROXY "B:"
#else
#define OPT_NF_TPROXY
#endif


int 
main(int argc, char *argv[])
{
  int opt;
  gchar *instance_name = "zorp";
  gchar *policy_file = ZORP_POLICY_FILE;
  gchar *log_spec = "";
  gchar *getopt_str = "a:s:p:v:dhlVu:g:TI:R:O" OPT_CAPS OPT_NF_TPROXY;
  gchar pid_file_buf[128], *pid_file = NULL;
  gchar *zorp_user = "root";
  gint zorp_uid = -1, zorp_gid = -1;
  gboolean use_syslog = 1;
  gboolean log_tags = FALSE;
  gchar *root_dir = NULL;

  z_mem_trace_init(NULL);

#if HAVE_GETOPT_LONG
  while ((opt = getopt_long(argc, argv, getopt_str, zorp_options, NULL)) != -1)
#else
  while ((opt = getopt(argc, argv, getopt_str)) != -1)
#endif
    {
      switch (opt)
	{
	case 'a':
	  instance_name = optarg;
	  break;
	case 'p':
	  policy_file = optarg;
	  break;
	case 'v':
	  if (optarg)
	    verbose_level = atoi(optarg);
	  else
	    verbose_level++;
	  break;
	case 'd':
	  printf("--debug is obsolete, use --log-spec instead\n");
	  break;
	case 's':
	  log_spec = optarg;
	  break;
	case 'P':
	  pid_file = optarg;
	  break;
	case 'h':
	  z_usage();
	  return 0;
	case 'l':
	  use_syslog = 0;
	  break;
	case 'T':
	  log_tags = 1;
	  break;
	case 'V':
	  z_version();
	  return 0;
	case 't':
	  max_threads = atoi(optarg);
	  break;
	case 'u':
	  zorp_user = optarg;
	  if (!z_resolve_user(optarg, &zorp_uid))
	    {
              fprintf(stderr, "Invalid uid: %s\n", optarg);
              return 1;
            }
	  break;
	case 'g':
	  if (!z_resolve_group(optarg, &zorp_gid))
	    {
	      fprintf(stderr, "Invalid gid: %s\n", optarg);
	      return 1;
	    }
	  break;
	case 'I':
	  idle_threads = atoi(optarg);
	  break;
	case 'R':
	  root_dir = optarg;
	  break;
	case 'O':
	  use_threadpools = TRUE;
	  break;
#if ZORPLIB_ENABLE_CAPS
	case 'C':
	  zorp_caps = optarg;
	  break;
	case 'N':
	  zorp_caps = NULL;
	  break;
#endif
#if ENABLE_NETFILTER_TPROXY
        case 'B':
          auto_bind_ip = optarg;
          break;
#endif
	default:
	  fprintf(stderr, "Invalid arguments\n");
	  return 1;
	}
    }
  
  if (optind != argc)
    {
      fprintf(stderr, "Invalid arguments\n");
      return 1;
    }
  if (pid_file == NULL)
    {
      g_snprintf(pid_file_buf, sizeof(pid_file_buf), "%s/zorp-%s.pid", ZORP_PID_FILE_DIR, instance_name);
      pid_file = pid_file_buf;
    }
  z_write_pid_file(pid_file);
  if (root_dir)
    chroot(root_dir);

#if HAVE_PRCTL && HAVE_PR_SET_KEEPCAPS
  prctl(PR_SET_KEEPCAPS, 1, 0, 0, 0);
#endif
  
  if (zorp_gid != -1)
    {
      setgid(zorp_gid);
      initgroups(zorp_user, zorp_gid);
    }
    
  if (zorp_uid != -1) 
    {
      setuid(zorp_uid);
    }
    
#if ZORPLIB_ENABLE_CAPS
  if (zorp_caps) 
    {
      cap_t cap = cap_from_text(zorp_caps);
    
      if (cap == NULL)
        {
          fprintf(stderr, "Error parsing capabilities: %s\n", zorp_caps);
        }
      else
        {
          if (cap_set_proc(cap) == -1)
            {
              fprintf(stderr, "Error setting capabilities due to %m\n");
            }
          cap_free(cap);
        }
    }
#endif

  z_thread_init();
  g_main_context_acquire(NULL);

  /*NOLOG*/
  z_log_init(log_spec, instance_name, 
             (use_syslog ? ZLF_SYSLOG : 0) |
             (log_tags ? ZLF_TAGS : 0) |
             ZLF_STDERR | ZLF_THREAD);
  z_log(NULL, CORE_DEBUG, 0, "Verbosity level: %d", verbose_level);

  z_sysdep_init();
  z_free_queue_init();
  z_ssl_init();  
  z_szig_init(instance_name);
  z_dispatch_init();
  if (!z_conntrack_init())
    {
      exit_code = 1;
      goto deinit_exit;
    }
  z_registry_init();
  
  /* only used for PORT allocation within a given range */
  srand(time(NULL) ^ getpid()); 

  if (!z_python_init())
    {
      z_llog(CORE_ERROR, 0, "Error initializing Python policy engine.");
      exit_code = 1;
      goto deinit_exit;
    }
  

  z_setup_signals();

  z_llog(CORE_INFO, 5, "%s version %s starting up", PACKAGE, VERSION);
  z_main_loop(policy_file, instance_name);

  /*LOG
    This debug message is logged when Zorp shuts down.
   */

 deinit_exit:
 
  z_llog(CORE_INFO, 5, "%s version %s going down.", PACKAGE, VERSION);
  
  z_thread_destroy();
  z_python_destroy();
  z_dispatch_destroy();
  z_conntrack_destroy();
  z_free_queue_destroy();
  z_sysdep_destroy();
  z_ssl_destroy();
  z_remove_pid_file(pid_file);
  z_log_destroy();
  z_mem_trace_dump();
#ifdef USE_DMALLOC
  dmalloc_shutdown();
  /* avoid second dump of dmalloc */
  rename("logfile", "logfile.dm");
#endif
  return exit_code;
}
